#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
This module contains quadrature rule tables copied from
    P. Solin, K Segeth, and I Dolezel. Higher–Order Finite Element Methods.
    Studies in Advanced Mathematics. Chapman and Hall/CRC, 2004.

Use the function load_table(polygon, family) to get a list of QuadRule objects.
The family can be one of {"composite_gauss", "economical_gauss"}, the polygon name one of the UFC polygon names.
"""

# Copyright (C) 2008 Martin Sandve Alnes and Simula Resarch Laboratory
#
# This file is part of SyFi.
#
# SyFi is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 2 of the License, or
# (at your option) any later version.
#
# SyFi is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with SyFi. If not, see <http://www.gnu.org/licenses/>.
#
# First added:  2008-09-01
# Last changed: 2008-09-01

from sfc.quadrature.QuadRule import QuadRule


# TODO: To add non-Gauss rules here, just write them in the same format
#       as the gauss strings, and call them myrule_polygonname_table,
#       just like economical_gauss_triangle_table etc.

from ufl.geometry import domain2dim

def load_rule(polygon, order, family):

    rules = load_table(polygon, family)

    while 1:
        # find the first rule with this order
        for r in rules:
            if r.order == order:
                return r
        # if this failed, find the first rule order+1
        order += 1
    
    raise RuntimeError("Found no rule for a %s with order %d in family %s" % (polygon, order, family))


def load_table(polygon, family):
    """Return list of quadrature rules matching polygon and family."""
    rules = parse_table_string(polygon, family)
    rules = scale_table(rules)
    return rules


def parse_table_string(polygon, family):
    # Fetch a table string with quadrature points and weights for different orders.
    name = "%s_%s_table" % (family, polygon)
    tablestring = eval(name)

    nsd = domain2dim[polygon]
    lines = tablestring.split('\n')
    rules = []
    i = 0
    while i<len(lines):
        n = int(lines[i])
        i += 1
        points  = []
        weights = []
        for j in range(n):
            values   = lines[i].split()
            points  += [ map(float, values[:nsd]) ]
            weights += [ float(values[nsd]) ]
            i += 1
        comment = lines[i].strip()
        i += 2
        order = int(comment.split('=')[-1].split()[-1])
        rules.append( QuadRule(polygon, nsd, order, points, weights, comment) )
    return rules


def scale_table(rules):
    """Scales a set of quadrature rules from [-1,+1]^nsd to [0,1]^nsd."""
    point_offset = 0.5
    point_scale  = 0.5
    for rule in rules:
        ws = point_scale**rule.nsd
        rule.weights = [ w*ws for w in rule.weights ]
        rule.points  = [ [x*point_scale+point_offset for x in p] for p in rule.points ]
    return rules


# Gauss rules
gauss_interval_table = """\
1
0.0000000000000000000000000  2.0000000000000000000000000
  Gauss  quadrature  points  and  weights  order  2n-1=1

2
-0.5773502691896257645091488  1.0000000000000000000000000
0.5773502691896257645091488  1.0000000000000000000000000
  Gauss  quadrature  points  and  weights  order  2n-1=3

3
-0.7745966692414833770358531  0.5555555555555555555555556
0.0000000000000000000000000  0.8888888888888888888888889
0.7745966692414833770358531  0.5555555555555555555555556
  Gauss  quadrature  points  and  weights  order  2n-1=5

4
-0.8611363115940525752239465  0.3478548451374538573730639
-0.3399810435848562648026658  0.6521451548625461426269361
0.3399810435848562648026658  0.6521451548625461426269361
0.8611363115940525752239465  0.3478548451374538573730639
  Gauss  quadrature  points  and  weights  order  2n-1=7

5
-0.9061798459386639927976269  0.2369268850561890875142640
-0.5384693101056830910363144  0.4786286704993664680412915
0.0000000000000000000000000  0.5688888888888888888888889
0.5384693101056830910363144  0.4786286704993664680412915
0.9061798459386639927976269  0.2369268850561890875142640
  Gauss  quadrature  points  and  weights  order  2n-1=9

6
-0.9324695142031520278123016  0.1713244923791703450402961
-0.6612093864662645136613996  0.3607615730481386075698335
-0.2386191860831969086305017  0.4679139345726910473898703
0.2386191860831969086305017  0.4679139345726910473898703
0.6612093864662645136613996  0.3607615730481386075698335
0.9324695142031520278123016  0.1713244923791703450402961
  Gauss  quadrature  points  and  weights  order  2n-1=11

7
-0.9491079123427585245261897  0.1294849661688696932706114
-0.7415311855993944398638648  0.2797053914892766679014678
-0.4058451513773971669066064  0.3818300505051189449503698
0.0000000000000000000000000  0.4179591836734693877551020
0.4058451513773971669066064  0.3818300505051189449503698
0.7415311855993944398638648  0.2797053914892766679014678
0.9491079123427585245261897  0.1294849661688696932706114
  Gauss  quadrature  points  and  weights  order  2n-1=13

8
-0.9602898564975362316835609  0.1012285362903762591525314
-0.7966664774136267395915539  0.2223810344533744705443560
-0.5255324099163289858177390  0.3137066458778872873379622
-0.1834346424956498049394761  0.3626837833783619829651504
0.1834346424956498049394761  0.3626837833783619829651504
0.5255324099163289858177390  0.3137066458778872873379622
0.7966664774136267395915539  0.2223810344533744705443560
0.9602898564975362316835609  0.1012285362903762591525314
  Gauss  quadrature  points  and  weights  order  2n-1=15

9
-0.9681602395076260898355762  0.0812743883615744119718922
-0.8360311073266357942994297  0.1806481606948574040584720
-0.6133714327005903973087020  0.2606106964029354623187429
-0.3242534234038089290385380  0.3123470770400028400686304
0.0000000000000000000000000  0.3302393550012597631645251
0.3242534234038089290385380  0.3123470770400028400686304
0.6133714327005903973087020  0.2606106964029354623187429
0.8360311073266357942994298  0.1806481606948574040584720
0.9681602395076260898355762  0.0812743883615744119718922
  Gauss  quadrature  points  and  weights  order  2n-1=17

10
-0.9739065285171717200779640   0.0666713443086881375935688
-0.8650633666889845107320967  0.1494513491505805931457763
-0.6794095682990244062343274  0.2190863625159820439955349
-0.4333953941292471907992659  0.2692667193099963550912269
-0.1488743389816312108848260  0.2955242247147528701738930
0.1488743389816312108848260  0.2955242247147528701738930
0.4333953941292471907992659  0.2692667193099963550912269
0.6794095682990244062343274  0.2190863625159820439955349
0.8650633666889845107320967  0.1494513491505805931457763
0.9739065285171717200779640  0.0666713443086881375935688
  Gauss  quadrature  points  and  weights  order  2n-1=19

11
-0.9782286581460569928039380  0.0556685671161736664827537
-0.8870625997680952990751578  0.1255803694649046246346943
-0.7301520055740493240934162  0.1862902109277342514260976
-0.5190961292068118159257257  0.2331937645919904799185237
-0.2695431559523449723315320  0.2628045445102466621806889
0.0000000000000000000000000  0.2729250867779006307144835
0.2695431559523449723315320  0.2628045445102466621806889
0.5190961292068118159257257  0.2331937645919904799185237
0.7301520055740493240934163  0.1862902109277342514260980
0.8870625997680952990751578  0.1255803694649046246346940
0.9782286581460569928039380  0.0556685671161736664827537
  Gauss  quadrature  points  and  weights  order  2n-1=21

12
-0.9815606342467192506905491  0.0471753363865118271946160
-0.9041172563704748566784659  0.1069393259953184309602547
-0.7699026741943046870368938  0.1600783285433462263346525
-0.5873179542866174472967024  0.2031674267230659217490645
-0.3678314989981801937526915  0.2334925365383548087608499
-0.1252334085114689154724414  0.2491470458134027850005624
0.1252334085114689154724414  0.2491470458134027850005624
0.3678314989981801937526915  0.2334925365383548087608499
0.5873179542866174472967024  0.2031674267230659217490645
0.7699026741943046870368938  0.1600783285433462263346525
0.9041172563704748566784659  0.1069393259953184309602547
0.9815606342467192506905491  0.0471753363865118271946160
  Gauss  quadrature  points  and  weights  order  2n-1=23

16
-0.9894009349916499325961542  0.0271524594117540948517806
-0.9445750230732325760779884  0.0622535239386478928628438
-0.8656312023878317438804679  0.0951585116824927848099251
-0.7554044083550030338951012  0.1246289712555338720524763
-0.6178762444026437484466718  0.1495959888165767320815017
-0.4580167776572273863424194  0.1691565193950025381893121
-0.2816035507792589132304605  0.1826034150449235888667637
-0.0950125098376374401853193  0.1894506104550684962853967
0.0950125098376374401853193  0.1894506104550684962853967
0.2816035507792589132304605  0.1826034150449235888667637
0.4580167776572273863424194  0.1691565193950025381893121
0.6178762444026437484466718  0.1495959888165767320815017
0.7554044083550030338951012  0.1246289712555338720524763
0.8656312023878317438804679  0.0951585116824927848099251
0.9445750230732325760779884  0.0622535239386478928628438
0.9894009349916499325961542  0.0271524594117540948517806
  Gauss  quadrature  points  and  weights  order  2n-1=31

20
-0.9931285991850949247861224  0.0176140071391521183118620
-0.9639719272779137912676661  0.0406014298003869413310400
-0.9122344282513259058677524  0.0626720483341090635695065
-0.8391169718222188233945291  0.0832767415767047487247581
-0.7463319064601507926143051  0.1019301198172404350367501
-0.6360536807265150254528367  0.1181945319615184173123774
-0.5108670019508270980043641  0.1316886384491766268984945
-0.3737060887154195606725482  0.1420961093183820513292983
-0.2277858511416450780804962  0.1491729864726037467878287
-0.0765265211334973337546404  0.1527533871307258506980843
0.0765265211334973337546404  0.1527533871307258506980843
0.2277858511416450780804962  0.1491729864726037467878287
0.3737060887154195606725482  0.1420961093183820513292983
0.5108670019508270980043641  0.1316886384491766268984945
0.6360536807265150254528367  0.1181945319615184173123774
0.7463319064601507926143051  0.1019301198172404350367501
0.8391169718222188233945291  0.0832767415767047487247581
0.9122344282513259058677524  0.0626720483341090635695065
0.9639719272779137912676661  0.0406014298003869413310400
0.9931285991850949247861224  0.0176140071391521183118620
  Gauss  quadrature  points  and  weights  order  2n-1=39

24
-0.9951872199970213601799974  0.0123412297999871995468057
-0.9747285559713094981983919  0.0285313886289336631813078
-0.9382745520027327585236490  0.0442774388174198061686027
-0.8864155270044010342131543  0.0592985849154367807463678
-0.8200019859739029219539499  0.0733464814110803057340336
-0.7401241915785543642438281  0.0861901615319532759171852
-0.6480936519369755692524958  0.0976186521041138882698807
-0.5454214713888395356583756  0.1074442701159656347825773
-0.4337935076260451384870842  0.1155056680537256013533445
-0.3150426796961633743867933  0.1216704729278033912044632
-0.1911188674736163091586398  0.1258374563468282961213754
-0.0640568928626056260850431  0.1279381953467521569740562
0.0640568928626056260850431  0.1279381953467521569740562
0.1911188674736163091586398  0.1258374563468282961213754
0.3150426796961633743867933  0.1216704729278033912044632
0.4337935076260451384870842  0.1155056680537256013533445
0.5454214713888395356583756  0.1074442701159656347825773
0.6480936519369755692524958  0.0976186521041138882698807
0.7401241915785543642438281  0.0861901615319532759171852
0.8200019859739029219539499  0.0733464814110803057340336
0.8864155270044010342131543  0.0592985849154367807463678
0.9382745520027327585236490  0.0442774388174198061686027
0.9747285559130949819839199  0.0285313886289336631813078
0.9951872199970213601799974  0.0123412297999871995468057
  Gauss  quadrature  points  and  weights  order  2n-1=47

32
-0.9972638618494815635449811  0.0070186100094700966004071
-0.9856115115452683354001750  0.0162743947309056706051706
-0.9647622555875064307738119  0.0253920653092620594557526
-0.9349060759377396891709191  0.0342738629130214331026877
-0.8963211557660521239653072  0.0428358980222266806568786
-0.8493676137325699701336930  0.0509980592623761761961632
-0.7944837959679424069630973  0.0586840934785355471452836
-0.7321821187402896803874267  0.0658222227763618468376501
-0.6630442669302152009751152  0.0723457941088485062253994
-0.5877157572407623290407455  0.0781938957870703064717409
-0.5068999089322293900237475  0.0833119242269467552221991
-0.4213512761306353453641194  0.0876520930044038111427715
-0.3318686022821276497799168  0.0911738786957638847128686
-0.2392873622521370745446032  0.0938443990808045656391802
-0.1444719615827964934851864  0.0956387200792748594190820
-0.0483076656877383162348126  0.0965400885147278005667648
0.0483076656877383162348126  0.0965400885147278005667648
0.1444719615827964934851864  0.0956387200792748594190820
0.2392873622521370745446032  0.0938443990808045656391802
0.3318686022821276497799168  0.0911738786957638847128686
0.4213512761306353453641194  0.0876520930044038111427715
0.5068999089322293900237475  0.0833119242269467552221991
0.5877157572407623290407455  0.0781938957870703064717409
0.6630442669302152009751152  0.0723457941088485062253994
0.7321821187402896803874267  0.0658222227763618468376501
0.7944837959679424069630973  0.0586840934785355471452836
0.8493676137325699701336930  0.0509980592623761761961632
0.8963211557660521239653072  0.0428358980222266806568787
0.9349060759377396891709191  0.0342738629130214331026877
0.9647622555875064307738119  0.0253920653092620594557526
0.9856115115452683354001750  0.0162743947309056706051706
0.9972638618494815635449811  0.0070186100094700966004071
  Gauss  quadrature  points  and  weights  order  2n-1=63

40
-0.9982377097105592003496227  0.0045212770985331912584717
-0.9907262386994570064530544  0.0104982845311528136147422
-0.9772599499837742626633703  0.0164210583819078887128635
-0.9579168192137916558045410  0.0222458491941669572615043
-0.9328128082786765333608522  0.0279370069800234010984892
-0.9020988069688742967282533  0.0334601952825478473926782
-0.8659595032122595038207818  0.0387821679744720176399720
-0.8246122308333116631963202  0.0438709081856732719916747
-0.7783056514265193876949715  0.0486958076350722320614342
-0.7273182551899271032809965  0.0532278469839368243549965
-0.6719566846141795483793545  0.0574397690993915513666177
-0.6125538896679802379526125  0.0613062424929289391665380
-0.5494671250951282020759313  0.0648040134566010380745545
-0.4830758016861787129085666  0.0679120458152339038256901
-0.4137792043716050015248797  0.0706116473912867796954836
-0.3419940908257584730074925  0.0728865823958040590605107
-0.2681521850072536811411843  0.0747231690579682642001893
-0.1926975807013710997155169  0.0761103619006262423715581
-0.1160840706752552084834513  0.0770398181642479655883075
-0.0387724175060508219331934 0.0775059479784248112637240
0.0387724175060508219331934  0.0775059479784248112637240
0.1160840706752552084834513  0.0770398181642479655883075
0.1926975807013710997155169  0.0761103619006262423715581
0.2681521850072536811411843  0.0747231690579682642001893
0.3419940908257584730074925  0.0728865823958040590605107
0.4137792043716050015248797  0.0706116473912867796954836
0.4830758016861787129085666  0.0679120458152339038256901
0.5494671250951282020759313  0.0648040134566010380745545
0.6125538896679802379526125  0.0613062424929289391665380
0.6719566846141795483793545  0.0574397690993915513666177
0.7273182551899271032809965  0.0532278469839368243549965
0.7783056514265193876949715  0.0486958076350722320614342
0.8246122308333116631963202  0.0438709081856732719916747
0.8659595032122595038207818  0.0387821679744720176399720
0.9020988069688742967282533  0.0334601952825478473926782
0.9328128082786765333608522  0.0279370069800234010984892
0.9579168192137916558045410  0.0222458491941669572615040
0.9772599499837742626633703  0.0164210583819078887128635
0.9907262386994570064530544  0.0104982845311528136147422
0.9982377097105592003496227  0.0045212770985331912584717
  Gauss  quadrature  points  and  weights  order  2n-1=79

48
-0.9987710072524261186005415  0.0031533460523058386326773
-0.9935301722663507575479288  0.0073275539012762621023840
-0.9841245837228268577445836  0.0114772345792345394895927
-0.9705915925462472504614120  0.0155793157229438487281770
-0.9529877031604308607229607  0.0196161604573555278144607
-0.9313866907065543331141744  0.0235707608393243791405193
-0.9058791367155696728220748  0.0274265097083569482000738
-0.8765720202742478859056936  0.0311672278327980889020658
-0.8435882616243935307110898  0.0347772225647704388925486
-0.8070662040294426270825530  0.0382413510658307063172173
-0.7671590325157403392538554  0.0415450829434647492140588
-0.7240341309238146546744822  0.0446745608566942804194486
-0.6778723796326639052118513  0.0476166584924904748259066
-0.6288673967765136239951649  0.0503590355538544749578076
-0.5772247260839727038178092  0.0528901894851936670955051
-0.5231609747222330336782259  0.0551995036999841628682035
-0.4669029047509584045449289  0.0572772921004032157051502
-0.4086864819907167299162255  0.0591148396983956357464748
-0.3487558862921607381598179  0.0607044391658938800529692
-0.2873624873554555767358865  0.0620394231598926639041978
-0.2247637903946890612248654  0.0631141922862540256571260
-0.1612223560688917180564374  0.0639242385846481866239062
-0.0970046992094626989300540  0.0644661644359500822065042
-0.0323801709628693620333222  0.0647376968126839225030249
0.0323801709628693620333222  0.0647376968126839225030249
0.0970046992094626989300540  0.0644661644359500822065042
0.1612223560688917180564374  0.0639242385846481866239062
0.2247637903946890612248654  0.0631141922862540256571260
0.2873624873554555767358865  0.0620394231598926639041978
0.3487558862921607381598179  0.0607044391658938800529692
0.4086864819907167299162255  0.0591148396983956357464748
0.4669029047509584045449289  0.0572772921004032157051502
0.5231609747222330336782259  0.0551995036999841628682035
0.5772247260839727038178092  0.0528901894851936670955051
0.6288673967765136239951649  0.0503590355538544749578076
0.6778723796326639052118513  0.0476166584924904748259066
0.7240341309238146546744822  0.0446745608566942804194486
0.7671590325157403392538554  0.0415450829434647492140588
0.8070662040294426270825530  0.0382413510658307063172173
0.8435882616243935307110898  0.0347772225647704388925486
0.8765720202742478859056936  0.0311672278327980889020658
0.9058791367155696728220748  0.0274265097083569482000738
0.9313866907065543331141744  0.0235707608393243791405193
0.9529877031604308607229607  0.0196161604573555278144607
0.9705915925462472504614120  0.0155793157229438487281770
0.9841245837228268577445836  0.0114772345792345394895927
0.9935301722663507575479288  0.0073275539012762621023840
0.9987710072524261186005415  0.0031533460523058386326773
  Gauss  quadrature  points  and  weights  order  2n-1=95

64
-0.9993050417357721394569056  0.0017832807216964329472961
-0.9963401167719552793469245  0.0041470332605624676352875
-0.9910133714767443207393824  0.0065044579689783628561174
-0.9833362538846259569312993  0.0088467598263639477230309
-0.9733268277899109637418535  0.0111681394601311288185905
-0.9610087996520537189186141  0.0134630478967186425980608
-0.9464113748584028160624815  0.0157260304760247193219660
-0.9295691721319395758214902  0.0179517157756973430850453
-0.9105221370785028057563807  0.0201348231535302093723403
-0.8893154459951141058534040  0.0222701738083832541592983
-0.8659993981540928197607834  0.0243527025687108733381776
-0.8406292962525803627516915  0.0263774697150546586716918
-0.8132653151227975597419233  0.0283396726142594832275113
-0.7839723589433414076102205  0.0302346570724024788679741
-0.7528199072605318966118638  0.0320579283548515535854675
-0.7198818501716108268489402  0.0338051618371416093915655
-0.6852363130542332425635584  0.0354722132568823838106931
-0.6489654712546573398577612  0.0370551285402400460404151
-0.6111553551723932502488530  0.0385501531786156291289625
-0.5718956462026340342838781  0.0399537411327203413866569
-0.5312794640198945456580139  0.0412625632426235286101563
-0.4894031457070529574785263  0.0424735151236535890073398
-0.4463660172534640879849477  0.0435837245293234533768279
-0.4022701579639916036957668  0.0445905581637565630601347
-0.3572201583376681159504426  0.0454916279274181444797710
-0.3113228719902109561575127  0.0462847965813144172959532
-0.2646871622087674163739642  0.0469681828162100173253263
-0.2174236437400070841496487  0.0475401657148303086622822
-0.1696444204239928180373136  0.0479993885964583077281262
-0.1214628192961205544703765  0.0483447622348029571697695
-0.0729931217877990394495429  0.0485754674415034269347991
-0.0243502926634244325089558  0.0486909570091397203833654
0.0243502926634244325089558  0.0486909570091397203833654
0.0729931217877990394495429  0.0485754674415034269347991
0.1214628192961205544703765  0.0483447622348029571697695
0.1696444204239928180373136  0.0479993885964583077281262
0.2174236437400070841496487  0.0475401657148303086622822
0.2646871622087674163739642  0.0469681828162100173253263
0.3113228719902109561575127  0.0462847965813144172959532
0.3572201583376681159504426  0.0454916279274181444797710
0.4022701579639916036957668  0.0445905581637565630601347
0.4463660172534640879849477  0.0435837245293234533768279
0.4894031457070529574785263  0.0424735151236535890073398
0.5312794640198945456580139  0.0412625632426235286101563
0.5718956462026340342838781  0.0399537411327203413866569
0.6111553551723932502488530  0.0385501531786156291289625
0.6489654712546573398577612  0.0370551285402400460404151
0.6852363130542332425635584  0.0354722132568823838106931
0.7198818501716108268489402  0.0338051618371416093915655
0.7528199072605318966118638  0.0320579283548515535854675
0.7839723589433414076102205  0.0302346570724024788679741
0.8132653151227975597419233  0.0283396726142594832275113
0.8406292962525803627516915  0.0263774697150546586716918
0.8659993981540928197607834  0.0243527025687108733381776
0.8893154459951141058534040  0.0222701738083832541592983
0.9105221370785028057563807  0.0201348231535302093723403
0.9295691721319395758214902  0.0179517157756973430850453
0.9464113748584028160624815  0.0157260304760247193219660
0.9610087996520537189186141  0.0134630478967186425980608
0.9733268277899109637418535  0.0111681394601311288185900
0.9833362538846259569312993  0.0088467598263639477230309
0.9910133714767443207393824  0.0065044579689783628561174
0.9963401167719552793469245  0.0041470332605624676352875
0.9993050417357721394569056  0.0017832807216964329472961
  Gauss  quadrature  points  and  weights  order  2n-1=127
"""


economical_gauss_quadrilateral_table = """\
1
0.000000000000000  0.000000000000000  4.000000000000000
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=0  1

4
0.577350269189626  0.577350269189626  1.000000000000000
0.577350269189626  -0.577350269189626  1.000000000000000
-0.577350269189626  0.577350269189626  1.000000000000000
-0.577350269189626  -0.577350269189626  1.000000000000000
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=2  3

8
0.683130051063973  0.000000000000000  0.816326530612245
-0.683130051063973  0.000000000000000  0.816326530612245
0.000000000000000  0.683130051063973  0.816326530612245
0.000000000000000  -0.683130051063973  0.816326530612245
0.881917103688197  0.881917103688197  0.183673469387755
0.881917103688197  -0.881917103688197  0.183673469387755
-0.881917103688197  0.881917103688197  0.183673469387755
-0.881917103688197  -0.881917103688197  0.183673469387755
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=4  5

12
0.925820099772551  0.000000000000000  0.241975308641975
-0.925820099772551  0.000000000000000  0.241975308641975
0.000000000000000  0.925820099772551  0.241975308641975
0.000000000000000  -0.925820099772551  0.241975308641975
0.805979782918599  0.805979782918599  0.237431774690630
0.805979782918599  -0.805979782918599  0.237431774690630
-0.805979782918599  0.805979782918599  0.237431774690630
-0.805979782918599  -0.805979782918599  0.237431774690630
0.380554433208316  0.380554433208316  0.520592916667394
0.380554433208316  -0.380554433208316  0.520592916667394
-0.380554433208316  0.380554433208316  0.520592916667394
-0.380554433208316  -0.380554433208316  0.520592916667394
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=6  7

20
1.121225763866564  0.000000000000000  0.018475842507491
-1.121225763866564  0.000000000000000  0.018475842507491
0.000000000000000  1.121225763866564  0.018475842507491
0.000000000000000  -1.121225763866564  0.018475842507491
0.451773049920657  0.000000000000000  0.390052939160735
-0.451773049920657  0.000000000000000  0.390052939160735
0.000000000000000  0.451773049920657  0.390052939160735
0.000000000000000  -0.451773049920657  0.390052939160735
0.891849420851512  0.891849420851512  0.083095178026482
0.891849420851512  -0.891849420851512  0.083095178026482
-0.891849420851512  0.891849420851512  0.083095178026482
-0.891849420851512  -0.891849420851512  0.083095178026482
0.824396370749276  0.411623426336542  0.254188020152646
0.824396370749276  -0.411623426336542  0.254188020152646
-0.824396370749276  0.411623426336542  0.254188020152646
-0.824396370749276  -0.411623426336542  0.254188020152646
0.411623426336542  0.824396370749276  0.254188020152646
0.411623426336542  -0.824396370749276  0.254188020152646
-0.411623426336542  0.824396370749276  0.254188020152646
-0.411623426336542  -0.824396370749276  0.254188020152646
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=8  9

25
0.000000000000000  0.000000000000000  0.365379525585903
1.044402915409813  0.000000000000000  0.027756165564204
-1.044402915409813  0.000000000000000  0.027756165564204
0.000000000000000  1.044402915409813  0.027756165564204
0.000000000000000  -1.044402915409813  0.027756165564204
0.769799068396649  0.000000000000000  0.244272057751754
-0.769799068396649  0.000000000000000  0.244272057751754
0.000000000000000  0.769799068396649  0.244272057751754
0.000000000000000  -0.769799068396649  0.244272057751754
0.935787012440540  0.935787012440540  0.034265103851229
0.935787012440540  -0.935787012440540  0.034265103851229
-0.935787012440540  0.935787012440540  0.034265103851229
-0.935787012440540  -0.935787012440540  0.034265103851229
0.413491953449114  0.413491953449114  0.308993036133713
0.413491953449114  -0.413491953449114  0.308993036133713
-0.413491953449114  0.413491953449114  0.308993036133713
-0.413491953449114  -0.413491953449114  0.308993036133713
0.883025508525690  0.575653595840465  0.146684377651312
0.883025508525690  -0.575653595840465  0.146684377651312
-0.883025508525690  0.575653595840465  0.146684377651312
-0.883025508525690  -0.575653595840465  0.146684377651312
0.575653595840465  0.883025508525690  0.146684377651312
0.575653595840465  -0.883025508525690  0.146684377651312
-0.575653595840465  0.883025508525690  0.146684377651312
-0.575653595840465  -0.883025508525690  0.146684377651312
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=10  11

36
1.086056158573971  0.000000000000000  0.005656169693764
-1.086056158573971  0.000000000000000  0.005656169693764
0.000000000000000  1.086056158573971  0.005656169693764
0.000000000000000  -1.086056158573971  0.005656169693764
0.658208197042585  0.000000000000000  0.192443967470396
-0.658208197042585  0.000000000000000  0.192443967470396
0.000000000000000  0.658208197042585  0.192443967470396
0.000000000000000  -0.658208197042585  0.192443967470396
1.001300602991729  1.001300602991729  0.005166832979773
1.001300602991729  -1.001300602991729  0.005166832979773
-1.001300602991729  1.001300602991729  0.005166832979773
-1.001300602991729  -1.001300602991729  0.005166832979773
0.584636168775946  0.584636168775946  0.200302559622138
0.584636168775946  -0.584636168775946  0.200302559622138
-0.584636168775946  0.584636168775946  0.200302559622138
-0.584636168775946  -0.584636168775946  0.200302559622138
0.246795612720261  0.246795612720261  0.228125175912536
0.246795612720261  -0.246795612720261  0.228125175912536
-0.246795612720261  0.246795612720261  0.228125175912536
-0.246795612720261  -0.246795612720261  0.228125175912536
0.900258815287201  0.304720678579870  0.117496926974491
0.900258815287201  -0.304720678579870  0.117496926974491
-0.900258815287201  0.304720678579870  0.117496926974491
-0.900258815287201  -0.304720678579870  0.117496926974491
0.304720678579870  0.900258815287201  0.117496926974491
0.304720678579870  -0.900258815287201  0.117496926974491
-0.304720678579870  0.900258815287201  0.117496926974491
-0.304720678579870  -0.900258815287201  0.117496926974491
0.929866705560780  0.745052720131169  0.066655770186205
0.929866705560780  -0.745052720131169  0.066655770186205
-0.929866705560780  0.745052720131169  0.066655770186205
-0.929866705560780  -0.745052720131169  0.066655770186205
0.745052720131169  0.929866705560780  0.066655770186205
0.745052720131169  -0.929866705560780  0.066655770186205
-0.745052720131169  0.929866705560780  0.066655770186205
-0.745052720131169  -0.929866705560780  0.066655770186205
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=12  13

45
0.000000000000000  0.000000000000000  -0.001768979827207
1.027314357719367  0.000000000000000  0.012816726617512
-1.027314357719367  0.000000000000000  0.012816726617512
0.000000000000000  1.027314357719367  0.012816726617512
0.000000000000000  -1.027314357719367  0.012816726617512
0.856766776147643  0.000000000000000  0.119897873101347
-0.856766776147643  0.000000000000000  0.119897873101347
0.000000000000000  0.856766776147643  0.119897873101347
0.000000000000000  -0.856766776147643  0.119897873101347
0.327332998189723  0.000000000000000  0.210885452208801
-0.327332998189723  0.000000000000000  0.210885452208801
0.000000000000000  0.327332998189723  0.210885452208801
0.000000000000000  -0.327332998189723  0.210885452208801
0.967223740028505  0.967223740028505  0.006392720128215
0.967223740028505  -0.967223740028505  0.006392720128215
-0.967223740028505  0.967223740028505  0.006392720128215
-0.967223740028505  -0.967223740028505  0.006392720128215
0.732168901749711  0.732168901749711  0.104415680788580
0.732168901749711  -0.732168901749711  0.104415680788580
-0.732168901749711  0.732168901749711  0.104415680788580
-0.732168901749711  -0.732168901749711  0.104415680788580
0.621974427996805  0.321696694921009  0.168053047203816
0.621974427996805  -0.321696694921009  0.168053047203816
-0.621974427996805  0.321696694921009  0.168053047203816
-0.621974427996805  -0.321696694921009  0.168053047203816
0.321696694921009  0.621974427996805  0.168053047203816
0.321696694921009  -0.621974427996805  0.168053047203816
-0.321696694921009  0.621974427996805  0.168053047203816
-0.321696694921009  -0.621974427996805  0.168053047203816
0.928618480068352  0.455124178121179  0.076169694452294
0.928618480068352  -0.455124178121179  0.076169694452294
-0.928618480068352  0.455124178121179  0.076169694452294
-0.928618480068352  -0.455124178121179  0.076169694452294
0.455124178121179  0.928618480068352  0.076169694452294
0.455124178121179  -0.928618480068352  0.076169694452294
-0.455124178121179  0.928618480068352  0.076169694452294
-0.455124178121179  -0.928618480068352  0.076169694452294
0.960457474887516  0.809863684081217  0.028794154400064
0.960457474887516  -0.809863684081217  0.028794154400064
-0.960457474887516  0.809863684081217  0.028794154400064
-0.960457474887516  -0.809863684081217  0.028794154400064
0.809863684081217  0.960457474887516  0.028794154400064
0.809863684081217  -0.960457474887516  0.028794154400064
-0.809863684081217  0.960457474887516  0.028794154400064
-0.809863684081217  -0.960457474887516  0.028794154400064
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=14  15

60
0.989353074512600  0.000000000000000  0.020614915919991
-0.989353074512600  0.000000000000000  0.020614915919991
0.000000000000000  0.989353074512600  0.020614915919991
0.000000000000000  -0.989353074512600  0.020614915919991
0.376285207157973  0.000000000000000  0.128025716179910
-0.376285207157973  0.000000000000000  0.128025716179910
0.000000000000000  0.376285207157973  0.128025716179910
0.000000000000000  -0.376285207157973  0.128025716179910
0.978848279262233  0.978848279262233  0.005511739534032
0.978848279262233  -0.978848279262233  0.005511739534032
-0.978848279262233  0.978848279262233  0.005511739534032
-0.978848279262233  -0.978848279262233  0.005511739534032
0.885794729164116  0.885794729164116  0.039207712457142
0.885794729164116  -0.885794729164116  0.039207712457142
-0.885794729164116  0.885794729164116  0.039207712457142
-0.885794729164116  -0.885794729164116  0.039207712457142
0.171756123838348  0.171756123838348  0.076396945079863
0.171756123838348  -0.171756123838348  0.076396945079863
-0.171756123838348  0.171756123838348  0.076396945079863
-0.171756123838348  -0.171756123838348  0.076396945079863
0.590499273806002  0.319505036634574  0.141513729949972
0.590499273806002  -0.319505036634574  0.141513729949972
-0.590499273806002  0.319505036634574  0.141513729949972
-0.590499273806002  -0.319505036634574  0.141513729949972
0.319505036634574  0.590499273806002  0.141513729949972
0.319505036634574  -0.590499273806002  0.141513729949972
-0.319505036634574  0.590499273806002  0.141513729949972
-0.319505036634574  -0.590499273806002  0.141513729949972
0.799079131916863  0.597972451929457  0.083903279363798
0.799079131916863  -0.597972451929457  0.083903279363798
-0.799079131916863  0.597972451929457  0.083903279363798
-0.799079131916863  -0.597972451929457  0.083903279363798
0.597972451929457  0.799079131916863  0.083903279363798
0.597972451929457  -0.799079131916863  0.083903279363798
-0.597972451929457  0.799079131916863  0.083903279363798
-0.597972451929457  -0.799079131916863  0.083903279363798
0.803743962958745  0.058344481776551  0.060394163649685
0.803743962958745  -0.058344481776551  0.060394163649685
-0.803743962958745  0.058344481776551  0.060394163649685
-0.803743962958745  -0.058344481776551  0.060394163649685
0.058344481776551  0.803743962958745  0.060394163649685
0.058344481776551  -0.803743962958745  0.060394163649685
-0.058344481776551  0.803743962958745  0.060394163649685
-0.058344481776551  -0.803743962958745  0.060394163649685
0.936506276127495  0.347386316166203  0.057387752969213
0.936506276127495  -0.347386316166203  0.057387752969213
-0.936506276127495  0.347386316166203  0.057387752969213
-0.936506276127495  -0.347386316166203  0.057387752969213
0.347386316166203  0.936506276127495  0.057387752969213
0.347386316166203  -0.936506276127495  0.057387752969213
-0.347386316166203  0.936506276127495  0.057387752969213
-0.347386316166203  -0.936506276127495  0.057387752969213
0.981321179805452  0.706000287798646  0.021922559481864
0.981321179805452  -0.706000287798646  0.021922559481864
-0.981321179805452  0.706000287798646  0.021922559481864
-0.981321179805452  -0.706000287798646  0.021922559481864
0.706000287798646  0.981321179805452  0.021922559481864
0.706000287798646  -0.981321179805452  0.021922559481864
-0.706000287798646  0.981321179805452  0.021922559481864
-0.706000287798646  -0.981321179805452  0.021922559481864
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=16  17

72
0.943962831808239  0.000000000000000  0.038205406871462
-0.943962831808239  0.000000000000000  0.038205406871462
0.000000000000000  0.943962831808239  0.038205406871462
0.000000000000000  -0.943962831808239  0.038205406871462
0.536918434376013  0.000000000000000  0.135368502976521
-0.536918434376013  0.000000000000000  0.135368502976521
0.000000000000000  0.536918434376013  0.135368502976521
0.000000000000000  -0.536918434376013  0.135368502976521
0.973981076394170  0.973981076394170  0.005773708558664
0.973981076394170  -0.973981076394170  0.005773708558664
-0.973981076394170  0.973981076394170  0.005773708558664
-0.973981076394170  -0.973981076394170  0.005773708558664
0.742995535327609  0.742995535327609  0.067460759759473
0.742995535327609  -0.742995535327609  0.067460759759473
-0.742995535327609  0.742995535327609  0.067460759759473
-0.742995535327609  -0.742995535327609  0.067460759759473
0.285010052188916  0.285010052188916  0.140899115227892
0.285010052188916  -0.285010052188916  0.140899115227892
-0.285010052188916  0.285010052188916  0.140899115227892
-0.285010052188916  -0.285010052188916  0.140899115227892
0.068354569272491  0.068354569272491  0.047466627685662
0.068354569272491  -0.068354569272491  0.047466627685662
-0.068354569272491  0.068354569272491  0.047466627685662
-0.068354569272491  -0.068354569272491  0.047466627685662
0.802952004398543  0.203345534163332  0.078619467342982
0.802952004398543  -0.203345534163332  0.078619467342982
-0.802952004398543  0.203345534163332  0.078619467342982
-0.802952004398543  -0.203345534163332  0.078619467342982
0.203345534163332  0.802952004398543  0.078619467342982
0.203345534163332  -0.802952004398543  0.078619467342982
-0.203345534163332  0.802952004398543  0.078619467342982
-0.203345534163332  -0.802952004398543  0.078619467342982
0.634244672807882  0.426572172992877  0.094979169511394
0.634244672807882  -0.426572172992877  0.094979169511394
-0.634244672807882  0.426572172992877  0.094979169511394
-0.634244672807882  -0.426572172992877  0.094979169511394
0.426572172992877  0.634244672807882  0.094979169511394
0.426572172992877  -0.634244672807882  0.094979169511394
-0.426572172992877  0.634244672807882  0.094979169511394
-0.426572172992877  -0.634244672807882  0.094979169511394
0.978350706908227  0.295830776620995  0.022331162356015
0.978350706908227  -0.295830776620995  0.022331162356015
-0.978350706908227  0.295830776620995  0.022331162356015
-0.978350706908227  -0.295830776620995  0.022331162356015
0.295830776620995  0.978350706908227  0.022331162356015
0.295830776620995  -0.978350706908227  0.022331162356015
-0.295830776620995  0.978350706908227  0.022331162356015
-0.295830776620995  -0.978350706908227  0.022331162356015
0.901672717710389  0.541983037327871  0.055594877793785
0.901672717710389  -0.541983037327871  0.055594877793785
-0.901672717710389  0.541983037327871  0.055594877793785
-0.901672717710389  -0.541983037327871  0.055594877793785
0.541983037327871  0.901672717710389  0.055594877793785
0.541983037327871  -0.901672717710389  0.055594877793785
-0.541983037327871  0.901672717710389  0.055594877793785
-0.541983037327871  -0.901672717710389  0.055594877793785
1.007018449383116  0.669414798783936  0.006049054506376
1.007018449383116  -0.669414798783936  0.006049054506376
-1.007018449383116  0.669414798783936  0.006049054506376
-1.007018449383116  -0.669414798783936  0.006049054506376
0.669414798783936  1.007018449383116  0.006049054506376
0.669414798783936  -1.007018449383116  0.006049054506376
-0.669414798783936  1.007018449383116  0.006049054506376
-0.669414798783936  -1.007018449383116  0.006049054506376
0.945161453573471  0.829501421477824  0.024839207949609
0.945161453573471  -0.829501421477824  0.024839207949609
-0.945161453573471  0.829501421477824  0.024839207949609
-0.945161453573471  -0.829501421477824  0.024839207949609
0.829501421477824  0.945161453573471  0.024839207949609
0.829501421477824  -0.945161453573471  0.024839207949609
-0.829501421477824  0.945161453573471  0.024839207949609
-0.829501421477824  -0.945161453573471  0.024839207949609
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=18  19

88
0.980883148832881  0.000000000000000  0.019503841092684
-0.980883148832881  0.000000000000000  0.019503841092684
0.000000000000000  0.980883148832881  0.019503841092684
0.000000000000000  -0.980883148832881  0.019503841092684
0.678152700336576  0.000000000000000  0.089012127744268
-0.678152700336576  0.000000000000000  0.089012127744268
0.000000000000000  0.678152700336576  0.089012127744268
0.000000000000000  -0.678152700336576  0.089012127744268
0.240599282275864  0.000000000000000  0.114568584702749
-0.240599282275864  0.000000000000000  0.114568584702749
0.000000000000000  0.240599282275864  0.114568584702749
0.000000000000000  -0.240599282275864  0.114568584702749
0.965176994929162  0.965176994929162  0.007463627359106
0.965176994929162  -0.965176994929162  0.007463627359106
-0.965176994929162  0.965176994929162  0.007463627359106
-0.965176994929162  -0.965176994929162  0.007463627359106
0.749698539312765  0.749698539312765  0.050585943594705
0.749698539312765  -0.749698539312765  0.050585943594705
-0.749698539312765  0.749698539312765  0.050585943594705
-0.749698539312765  -0.749698539312765  0.050585943594705
0.568983925500818  0.568983925500818  0.074613865184212
0.568983925500818  -0.568983925500818  0.074613865184212
-0.568983925500818  0.568983925500818  0.074613865184212
-0.568983925500818  -0.568983925500818  0.074613865184212
0.971086142843168  0.355832132274584  0.023501091310143
0.971086142843168  -0.355832132274584  0.023501091310143
-0.971086142843168  0.355832132274584  0.023501091310143
-0.971086142843168  -0.355832132274584  0.023501091310143
0.355832132274584  0.971086142843168  0.023501091310143
0.355832132274584  -0.971086142843168  0.023501091310143
-0.355832132274584  0.971086142843168  0.023501091310143
-0.355832132274584  -0.971086142843168  0.023501091310143
0.983453947854968  0.645588139196562  0.011588562644144
0.983453947854968  -0.645588139196562  0.011588562644144
-0.983453947854968  0.645588139196562  0.011588562644144
-0.983453947854968  -0.645588139196562  0.011588562644144
0.645588139196562  0.983453947854968  0.011588562644144
0.645588139196562  -0.983453947854968  0.011588562644144
-0.645588139196562  0.983453947854968  0.011588562644144
-0.645588139196562  -0.983453947854968  0.011588562644144
0.933927707027213  0.821920249234369  0.023073245798171
0.933927707027213  -0.821920249234369  0.023073245798171
-0.933927707027213  0.821920249234369  0.023073245798171
-0.933927707027213  -0.821920249234369  0.023073245798171
0.821920249234369  0.933927707027213  0.023073245798171
0.821920249234369  -0.933927707027213  0.023073245798171
-0.821920249234369  0.933927707027213  0.023073245798171
-0.821920249234369  -0.933927707027213  0.023073245798171
1.014086498915039  0.862185099566557  0.001570221774472
1.014086498915039  -0.862185099566557  0.001570221774472
-1.014086498915039  0.862185099566557  0.001570221774472
-1.014086498915039  -0.862185099566557  0.001570221774472
0.862185099566557  1.014086498915039  0.001570221774472
0.862185099566557  -1.014086498915039  0.001570221774472
-0.862185099566557  1.014086498915039  0.001570221774472
-0.862185099566557  -1.014086498915039  0.001570221774472
0.877914842155496  0.168914072450263  0.049102258016227
0.877914842155496  -0.168914072450263  0.049102258016227
-0.877914842155496  0.168914072450263  0.049102258016227
-0.877914842155496  -0.168914072450263  0.049102258016227
0.168914072450263  0.877914842155496  0.049102258016227
0.168914072450263  -0.877914842155496  0.049102258016227
-0.168914072450263  0.877914842155496  0.049102258016227
-0.168914072450263  -0.877914842155496  0.049102258016227
0.882246882640128  0.568113580166780  0.042512352239126
0.882246882640128  -0.568113580166780  0.042512352239126
-0.882246882640128  0.568113580166780  0.042512352239126
-0.882246882640128  -0.568113580166780  0.042512352239126
0.568113580166780  0.882246882640128  0.042512352239126
0.568113580166780  -0.882246882640128  0.042512352239126
-0.568113580166780  0.882246882640128  0.042512352239126
-0.568113580166780  -0.882246882640128  0.042512352239126
0.741324453314596  0.371360260002223  0.067270936863160
0.741324453314596  -0.371360260002223  0.067270936863160
-0.741324453314596  0.371360260002223  0.067270936863160
-0.741324453314596  -0.371360260002223  0.067270936863160
0.371360260002223  0.741324453314596  0.067270936863160
0.371360260002223  -0.741324453314596  0.067270936863160
-0.371360260002223  0.741324453314596  0.067270936863160
-0.371360260002223  -0.741324453314596  0.067270936863160
0.469570217710647  0.237333359193547  0.103507336515645
0.469570217710647  -0.237333359193547  0.103507336515645
-0.469570217710647  0.237333359193547  0.103507336515645
-0.469570217710647  -0.237333359193547  0.103507336515645
0.237333359193547  0.469570217710647  0.103507336515645
0.237333359193547  -0.469570217710647  0.103507336515645
-0.237333359193547  0.469570217710647  0.103507336515645
-0.237333359193547  -0.469570217710647  0.103507336515645
  Gauss  quadrature  points  and  weights  on  the  reference  quadrilateral  order  p=20  21
"""


economical_gauss_hexahedron_table = """\
1
0.000000000000000  0.000000000000000  0.000000000000000  8.000000000000000
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=0  1

6
1.000000000000000  0.000000000000000  0.000000000000000  1.333333333333333
-1.000000000000000  0.000000000000000  0.000000000000000  1.333333333333333
0.000000000000000  1.000000000000000  0.000000000000000  1.333333333333333
0.000000000000000  -1.000000000000000  0.000000000000000  1.333333333333333
0.000000000000000  0.000000000000000  1.000000000000000  1.333333333333333
0.000000000000000  0.000000000000000  -1.000000000000000  1.333333333333333
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=2  3

14
0.795822425754221  0.000000000000000  0.000000000000000  0.886426592797784
-0.795822425754221  0.000000000000000  0.000000000000000  0.886426592797784
0.000000000000000  0.795822425754221  0.000000000000000  0.886426592797784
0.000000000000000  -0.795822425754221  0.000000000000000  0.886426592797784
0.000000000000000  0.000000000000000  0.795822425754221  0.886426592797784
0.000000000000000  0.000000000000000  -0.795822425754221  0.886426592797784
0.758786910639328  0.758786910639328  0.758786910639328  0.335180055401662
0.758786910639328  -0.758786910639328  0.758786910639328  0.335180055401662
0.758786910639328  0.758786910639328  -0.758786910639328  0.335180055401662
0.758786910639328  -0.758786910639328  -0.758786910639328  0.335180055401662
-0.758786910639328  0.758786910639328  0.758786910639328  0.335180055401662
-0.758786910639328  -0.758786910639328  0.758786910639328  0.335180055401662
-0.758786910639328  0.758786910639328  -0.758786910639328  0.335180055401662
-0.758786910639328  -0.758786910639328  -0.758786910639328  0.335180055401662
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=4  5

27
0.000000000000000  0.000000000000000  0.000000000000000  0.788073482744211
0.848418011472252  0.000000000000000  0.000000000000000  0.499369002307720
-0.848418011472252  0.000000000000000  0.000000000000000  0.499369002307720
0.000000000000000  0.848418011472252  0.000000000000000  0.499369002307720
0.000000000000000  -0.848418011472252  0.000000000000000  0.499369002307720
0.000000000000000  0.000000000000000  0.848418011472252  0.499369002307720
0.000000000000000  0.000000000000000  -0.848418011472252  0.499369002307720
0.652816472101691  0.652816472101691  0.652816472101691  0.478508449425127
0.652816472101691  -0.652816472101691  0.652816472101691  0.478508449425127
0.652816472101691  0.652816472101691  -0.652816472101691  0.478508449425127
0.652816472101691  -0.652816472101691  -0.652816472101691  0.478508449425127
-0.652816472101691  0.652816472101691  0.652816472101691  0.478508449425127
-0.652816472101691  -0.652816472101691  0.652816472101691  0.478508449425127
-0.652816472101691  0.652816472101691  -0.652816472101691  0.478508449425127
-0.652816472101691  -0.652816472101691  -0.652816472101691  0.478508449425127
0.000000000000000  1.106412898626718  1.106412898626718  0.032303742334037
0.000000000000000  -1.106412898626718  1.106412898626718  0.032303742334037
0.000000000000000  1.106412898626718  -1.106412898626718  0.032303742334037
0.000000000000000  -1.106412898626718  -1.106412898626718  0.032303742334037
1.106412898626718  0.000000000000000  1.106412898626718  0.032303742334037
-1.106412898626718  0.000000000000000  1.106412898626718  0.032303742334037
1.106412898626718  0.000000000000000  -1.106412898626718  0.032303742334037
-1.106412898626718  0.000000000000000  -1.106412898626718  0.032303742334037
1.106412898626718  1.106412898626718  0.000000000000000  0.032303742334037
-1.106412898626718  1.106412898626718  0.000000000000000  0.032303742334037
1.106412898626718  -1.106412898626718  0.000000000000000  0.032303742334037
-1.106412898626718  -1.106412898626718  0.000000000000000  0.032303742334037
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=6  7

53
0.000000000000000  0.000000000000000  0.000000000000000  0.588405321380412
1.064082230328777  0.000000000000000  0.000000000000000  -0.152097068487023
-1.064082230328777  0.000000000000000  0.000000000000000  -0.152097068487023
0.000000000000000  1.064082230328777  0.000000000000000  -0.152097068487023
0.000000000000000  -1.064082230328777  0.000000000000000  -0.152097068487023
0.000000000000000  0.000000000000000  1.064082230328777  -0.152097068487023
0.000000000000000  0.000000000000000  -1.064082230328777  -0.152097068487023
0.905830033000216  0.000000000000000  0.000000000000000  0.369012523996709
-0.905830033000216  0.000000000000000  0.000000000000000  0.369012523996709
0.000000000000000  0.905830033000216  0.000000000000000  0.369012523996709
0.000000000000000  -0.905830033000216  0.000000000000000  0.369012523996709
0.000000000000000  0.000000000000000  0.905830033000216  0.369012523996709
0.000000000000000  0.000000000000000  -0.905830033000216  0.369012523996709
0.817286490798906  0.817286490798906  0.817286490798906  0.104007450974435
0.817286490798906  -0.817286490798906  0.817286490798906  0.104007450974435
0.817286490798906  0.817286490798906  -0.817286490798906  0.104007450974435
0.817286490798906  -0.817286490798906  -0.817286490798906  0.104007450974435
-0.817286490798906  0.817286490798906  0.817286490798906  0.104007450974435
-0.817286490798906  -0.817286490798906  0.817286490798906  0.104007450974435
-0.817286490798906  0.817286490798906  -0.817286490798906  0.104007450974435
-0.817286490798906  -0.817286490798906  -0.817286490798906  0.104007450974435
0.501292956337400  0.501292956337400  0.501292956337400  0.380660357224238
0.501292956337400  -0.501292956337400  0.501292956337400  0.380660357224238
0.501292956337400  0.501292956337400  -0.501292956337400  0.380660357224238
0.501292956337400  -0.501292956337400  -0.501292956337400  0.380660357224238
-0.501292956337400  0.501292956337400  0.501292956337400  0.380660357224238
-0.501292956337400  -0.501292956337400  0.501292956337400  0.380660357224238
-0.501292956337400  0.501292956337400  -0.501292956337400  0.380660357224238
-0.501292956337400  -0.501292956337400  -0.501292956337400  0.380660357224238
1.017168937265364  0.650007853956632  0.000000000000000  0.0930316449988371
1.017168937265364  -0.650007853956632  0.000000000000000  0.0930316449988371
0.650007853956632  1.017168937265364  0.000000000000000  0.0930316449988371
-0.650007853956632  1.017168937265364  0.000000000000000  0.0930316449988371
-1.017168937265364  0.650007853956632  0.000000000000000  0.0930316449988371
-1.017168937265364  -0.650007853956632  0.000000000000000  0.0930316449988371
0.650007853956632  -1.017168937265364  0.000000000000000  0.0930316449988371
-0.650007853956632  -1.017168937265364  0.000000000000000  0.0930316449988371
0.000000000000000  0.650007853956632  1.017168937265364  0.0930316449988371
0.000000000000000  -0.650007853956632  1.017168937265364  0.0930316449988371
0.000000000000000  1.017168937265364  0.650007853956632  0.0930316449988371
0.000000000000000  1.017168937265364  -0.650007853956632  0.0930316449988371
0.000000000000000  0.650007853956632  -1.017168937265364  0.0930316449988371
0.000000000000000  -0.650007853956632  -1.017168937265364  0.0930316449988371
0.000000000000000  -1.017168937265364  0.650007853956632  0.0930316449988371
0.000000000000000  -1.017168937265364  -0.650007853956632  0.0930316449988371
1.017168937265364  0.000000000000000  0.650007853956632  0.0930316449988371
1.017168937265364  0.000000000000000  -0.650007853956632  0.0930316449988371
0.650007853956632  0.000000000000000  1.017168937265364  0.0930316449988371
-0.650007853956632  0.000000000000000  1.017168937265364  0.0930316449988371
-1.017168937265364  0.000000000000000  0.650007853956632  0.0930316449988371
-1.017168937265364  0.000000000000000  -0.650007853956632  0.0930316449988371
0.650007853956632  0.000000000000000  -1.017168937265364  0.0930316449988371
-0.650007853956632  0.000000000000000  -1.017168937265364  0.0930316449988371
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=8  9

89
0.000000000000000  0.000000000000000  0.000000000000000  -0.729841346362385
1.044892251909493  0.000000000000000  0.000000000000000  -0.295542895787525
-1.044892251909493  0.000000000000000  0.000000000000000  -0.295542895787525
0.000000000000000  1.044892251909493  0.000000000000000  -0.295542895787525
0.000000000000000  -1.044892251909493  0.000000000000000  -0.295542895787525
0.000000000000000  0.000000000000000  1.044892251909493  -0.295542895787525
0.000000000000000  0.000000000000000  -1.044892251909493  -0.295542895787525
1.128826539781045  0.000000000000000  0.000000000000000  0.075281719493673
-1.128826539781045  0.000000000000000  0.000000000000000  0.075281719493673
0.000000000000000  1.128826539781045  0.000000000000000  0.075281719493673
0.000000000000000  -1.128826539781045  0.000000000000000  0.075281719493673
0.000000000000000  0.000000000000000  1.128826539781045  0.075281719493673
0.000000000000000  0.000000000000000  -1.128826539781045  0.075281719493673
0.965386862750349  0.000000000000000  0.000000000000000  0.288865143683389
-0.965386862750349  0.000000000000000  0.000000000000000  0.288865143683389
0.000000000000000  0.965386862750349  0.000000000000000  0.288865143683389
0.000000000000000  -0.965386862750349  0.000000000000000  0.288865143683389
0.000000000000000  0.000000000000000  0.965386862750349  0.288865143683389
0.000000000000000  0.000000000000000  -0.965386862750349  0.288865143683389
0.398089853865140  0.000000000000000  0.000000000000000  0.393648801431496
-0.398089853865140  0.000000000000000  0.000000000000000  0.393648801431496
0.000000000000000  0.398089853865140  0.000000000000000  0.393648801431496
0.000000000000000  -0.398089853865140  0.000000000000000  0.393648801431496
0.000000000000000  0.000000000000000  0.398089853865140  0.393648801431496
0.000000000000000  0.000000000000000  -0.398089853865140  0.393648801431496
0.888733872960214  0.888733872960214  0.888733872960214  0.014156042088181
0.888733872960214  -0.888733872960214  0.888733872960214  0.014156042088181
0.888733872960214  0.888733872960214  -0.888733872960214  0.014156042088181
0.888733872960214  -0.888733872960214  -0.888733872960214  0.014156042088181
-0.888733872960214  0.888733872960214  0.888733872960214  0.014156042088181
-0.888733872960214  -0.888733872960214  0.888733872960214  0.014156042088181
-0.888733872960214  0.888733872960214  -0.888733872960214  0.014156042088181
-0.888733872960214  -0.888733872960214  -0.888733872960214  0.014156042088181
0.554161513623834  0.554161513623834  0.554161513623834  0.232318705969944
0.554161513623834  -0.554161513623834  0.554161513623834  0.232318705969944
0.554161513623834  0.554161513623834  -0.554161513623834  0.232318705969944
0.554161513623834  -0.554161513623834  -0.554161513623834  0.232318705969944
-0.554161513623834  0.554161513623834  0.554161513623834  0.232318705969944
-0.554161513623834  -0.554161513623834  0.554161513623834  0.232318705969944
-0.554161513623834  0.554161513623834  -0.554161513623834  0.232318705969944
-0.554161513623834  -0.554161513623834  -0.554161513623834  0.232318705969944
0.920049645350334  0.544394886959702  0.000000000000000  0.118935816985260
0.920049645350334  -0.544394886959702  0.000000000000000  0.118935816985260
0.544394886959702  0.920049645350334  0.000000000000000  0.118935816985260
-0.544394886959702  0.920049645350334  0.000000000000000  0.118935816985260
-0.920049645350334  0.544394886959702  0.000000000000000  0.118935816985260
-0.920049645350334  -0.544394886959702  0.000000000000000  0.118935816985260
0.544394886959702  -0.920049645350334  0.000000000000000  0.118935816985260
-0.544394886959702  -0.920049645350334  0.000000000000000  0.118935816985260
0.000000000000000  0.544394886959702  0.920049645350334  0.118935816985260
0.000000000000000  -0.544394886959702  0.920049645350334  0.118935816985260
0.000000000000000  0.920049645350334  0.544394886959702  0.118935816985260
0.000000000000000  0.920049645350334  -0.544394886959702  0.118935816985260
0.000000000000000  0.544394886959702  -0.920049645350334  0.118935816985260
0.000000000000000  -0.544394886959702  -0.920049645350334  0.118935816985260
0.000000000000000  -0.920049645350334  0.544394886959702  0.118935816985260
0.000000000000000  -0.920049645350334  -0.544394886959702  0.118935816985260
0.920049645350334  0.000000000000000  0.544394886959702  0.118935816985260
0.920049645350334  0.000000000000000  -0.544394886959702  0.118935816985260
0.544394886959702  0.000000000000000  0.920049645350334  0.118935816985260
-0.544394886959702  0.000000000000000  0.920049645350334  0.118935816985260
-0.920049645350334  0.000000000000000  0.544394886959702  0.118935816985260
-0.920049645350334  0.000000000000000  -0.544394886959702  0.118935816985260
0.544394886959702  0.000000000000000  -0.920049645350334  0.118935816985260
-0.544394886959702  0.000000000000000  -0.920049645350334  0.118935816985260
0.904534033733291  0.904534033733291  0.496742636335202  0.047086130965273
0.904534033733291  -0.904534033733291  0.496742636335202  0.047086130965273
0.904534033733291  0.904534033733291  -0.496742636335202  0.047086130965273
0.904534033733291  -0.904534033733291  -0.496742636335202  0.047086130965273
-0.904534033733291  0.904534033733291  0.496742636335202  0.047086130965273
-0.904534033733291  -0.904534033733291  0.496742636335202  0.047086130965273
-0.904534033733291  0.904534033733291  -0.496742636335202  0.047086130965273
-0.904534033733291  -0.904534033733291  -0.496742636335202  0.047086130965273
0.904534033733291  0.496742636335202  0.904534033733291  0.047086130965273
0.904534033733291  0.496742636335202  -0.904534033733291  0.047086130965273
0.904534033733291  -0.496742636335202  0.904534033733291  0.047086130965273
0.904534033733291  -0.496742636335202  -0.904534033733291  0.047086130965273
-0.904534033733291  0.496742636335202  0.904534033733291  0.047086130965273
-0.904534033733291  0.496742636335202  -0.904534033733291  0.047086130965273
-0.904534033733291  -0.496742636335202  0.904534033733291  0.047086130965273
-0.904534033733291  -0.496742636335202  -0.904534033733291  0.047086130965273
0.496742636335202  0.904534033733291  0.904534033733291  0.047086130965273
0.496742636335202  0.904534033733291  -0.904534033733291  0.047086130965273
-0.496742636335202  0.904534033733291  0.904534033733291  0.047086130965273
-0.496742636335202  0.904534033733291  -0.904534033733291  0.047086130965273
0.496742636335202  -0.904534033733291  0.904534033733291  0.047086130965273
0.496742636335202  -0.904534033733291  -0.904534033733291  0.047086130965273
-0.496742636335202  -0.904534033733291  0.904534033733291  0.047086130965273
-0.496742636335202  -0.904534033733291  -0.904534033733291  0.047086130965273
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=10  11

151
0.000000000000000  0.000000000000000  0.000000000000000  0.120671303848159
0.957697664927095  0.000000000000000  0.000000000000000  0.0827155895731376
-0.957697664927095  0.000000000000000  0.000000000000000  0.0827155895731376
0.000000000000000  0.957697664927095  0.000000000000000  0.0827155895731376
0.000000000000000  -0.957697664927095  0.000000000000000  0.0827155895731376
0.000000000000000  0.000000000000000  0.957697664927095  0.0827155895731376
0.000000000000000  0.000000000000000  -0.957697664927095  0.0827155895731376
0.799168794176284  0.000000000000000  0.000000000000000  -0.200338143812084
-0.799168794176284  0.000000000000000  0.000000000000000  -0.200338143812084
0.000000000000000  0.799168794176284  0.000000000000000  -0.200338143812084
0.000000000000000  -0.799168794176284  0.000000000000000  -0.200338143812084
0.000000000000000  0.000000000000000  0.799168794176284  -0.200338143812084
0.000000000000000  0.000000000000000  -0.799168794176284  -0.200338143812084
0.478190772481903  0.000000000000000  0.000000000000000  0.119014821875897
-0.478190772481903  0.000000000000000  0.000000000000000  0.119014821875897
0.000000000000000  0.478190772481903  0.000000000000000  0.119014821875897
0.000000000000000  -0.478190772481903  0.000000000000000  0.119014821875897
0.000000000000000  0.000000000000000  0.478190772481903  0.119014821875897
0.000000000000000  0.000000000000000  -0.478190772481903  0.119014821875897
0.863636428036401  0.863636428036401  0.863636428036401  0.033682130047600
0.863636428036401  -0.863636428036401  0.863636428036401  0.033682130047600
0.863636428036401  0.863636428036401  -0.863636428036401  0.033682130047600
0.863636428036401  -0.863636428036401  -0.863636428036401  0.033682130047600
-0.863636428036401  0.863636428036401  0.863636428036401  0.033682130047600
-0.863636428036401  -0.863636428036401  0.863636428036401  0.033682130047600
-0.863636428036401  0.863636428036401  -0.863636428036401  0.033682130047600
-0.863636428036401  -0.863636428036401  -0.863636428036401  0.033682130047600
0.625548513457035  0.625548513457035  0.625548513457035  0.0991539279234894
0.625548513457035  -0.625548513457035  0.625548513457035  0.0991539279234894
0.625548513457035  0.625548513457035  -0.625548513457035  0.0991539279234894
0.625548513457035  -0.625548513457035  -0.625548513457035  0.0991539279234894
-0.625548513457035  0.625548513457035  0.625548513457035  0.0991539279234894
-0.625548513457035  -0.625548513457035  0.625548513457035  0.0991539279234894
-0.625548513457035  0.625548513457035  -0.625548513457035  0.0991539279234894
-0.625548513457035  -0.625548513457035  -0.625548513457035  0.0991539279234894
0.367859232104514  0.367859232104514  0.367859232104514  0.0883375530928565
0.367859232104514  -0.367859232104514  0.367859232104514  0.0883375530928565
0.367859232104514  0.367859232104514  -0.367859232104514  0.0883375530928565
0.367859232104514  -0.367859232104514  -0.367859232104514  0.0883375530928565
-0.367859232104514  0.367859232104514  0.367859232104514  0.0883375530928565
-0.367859232104514  -0.367859232104514  0.367859232104514  0.0883375530928565
-0.367859232104514  0.367859232104514  -0.367859232104514  0.0883375530928565
-0.367859232104514  -0.367859232104514  -0.367859232104514  0.0883375530928565
0.869451316547931  0.869451316547931  0.000000000000000  0.038932959951030
0.869451316547931  -0.869451316547931  0.000000000000000  0.038932959951030
-0.869451316547931  0.869451316547931  0.000000000000000  0.038932959951030
-0.869451316547931  -0.869451316547931  0.000000000000000  0.038932959951030
0.000000000000000  0.869451316547931  0.869451316547931  0.038932959951030
0.000000000000000  0.869451316547931  -0.869451316547931  0.038932959951030
0.000000000000000  -0.869451316547931  0.869451316547931  0.038932959951030
0.000000000000000  -0.869451316547931  -0.869451316547931  0.038932959951030
0.869451316547931  0.000000000000000  0.869451316547931  0.038932959951030
-0.869451316547931  0.000000000000000  0.869451316547931  0.038932959951030
0.869451316547931  0.000000000000000  -0.869451316547931  0.038932959951030
-0.869451316547931  0.000000000000000  -0.869451316547931  0.038932959951030
0.767220340241960  0.354778664757558  0.000000000000000  0.154603530012075
0.767220340241960  -0.354778664757558  0.000000000000000  0.154603530012075
0.767220340241960  0.000000000000000  0.354778664757558  0.154603530012075
0.767220340241960  0.000000000000000  -0.354778664757558  0.154603530012075
-0.767220340241960  0.354778664757558  0.000000000000000  0.154603530012075
-0.767220340241960  -0.354778664757558  0.000000000000000  0.154603530012075
-0.767220340241960  0.000000000000000  0.354778664757558  0.154603530012075
-0.767220340241960  0.000000000000000  -0.354778664757558  0.154603530012075
0.354778664757558  0.767220340241960  0.000000000000000  0.154603530012075
-0.354778664757558  0.767220340241960  0.000000000000000  0.154603530012075
0.000000000000000  0.767220340241960  0.354778664757558  0.154603530012075
0.000000000000000  0.767220340241960  -0.354778664757558  0.154603530012075
0.354778664757558  -0.767220340241960  0.000000000000000  0.154603530012075
-0.354778664757558  -0.767220340241960  0.000000000000000  0.154603530012075
0.000000000000000  -0.767220340241960  0.354778664757558  0.154603530012075
0.000000000000000  -0.767220340241960  -0.354778664757558  0.154603530012075
0.000000000000000  0.354778664757558  0.767220340241960  0.154603530012075
0.000000000000000  -0.354778664757558  0.767220340241960  0.154603530012075
0.354778664757558  0.000000000000000  0.767220340241960  0.154603530012075
-0.354778664757558  0.000000000000000  0.767220340241960  0.154603530012075
0.000000000000000  0.354778664757558  -0.767220340241960  0.154603530012075
0.000000000000000  -0.354778664757558  -0.767220340241960  0.154603530012075
0.354778664757558  0.000000000000000  -0.767220340241960  0.154603530012075
-0.354778664757558  0.000000000000000  -0.767220340241960  0.154603530012075
1.012782207207705  1.012782207207705  0.537923170210472  0.00380152555202625
1.012782207207705  1.012782207207705  -0.537923170210472  0.00380152555202625
1.012782207207705  -1.012782207207705  0.537923170210472  0.00380152555202625
1.012782207207705  -1.012782207207705  -0.537923170210472  0.00380152555202625
-1.012782207207705  1.012782207207705  0.537923170210472  0.00380152555202625
-1.012782207207705  1.012782207207705  -0.537923170210472  0.00380152555202625
-1.012782207207705  -1.012782207207705  0.537923170210472  0.00380152555202625
-1.012782207207705  -1.012782207207705  -0.537923170210472  0.00380152555202625
0.537923170210472  1.012782207207705  1.012782207207705  0.00380152555202625
-0.537923170210472  1.012782207207705  1.012782207207705  0.00380152555202625
0.537923170210472  1.012782207207705  -1.012782207207705  0.00380152555202625
-0.537923170210472  1.012782207207705  -1.012782207207705  0.00380152555202625
0.537923170210472  -1.012782207207705  1.012782207207705  0.00380152555202625
-0.537923170210472  -1.012782207207705  1.012782207207705  0.00380152555202625
0.537923170210472  -1.012782207207705  -1.012782207207705  0.00380152555202625
-0.537923170210472  -1.012782207207705  -1.012782207207705  0.00380152555202625
1.012782207207705  0.537923170210472  1.012782207207705  0.00380152555202625
1.012782207207705  -0.537923170210472  1.012782207207705  0.00380152555202625
-1.012782207207705  0.537923170210472  1.012782207207705  0.00380152555202625
-1.012782207207705  -0.537923170210472  1.012782207207705  0.00380152555202625
1.012782207207705  0.537923170210472  -1.012782207207705  0.00380152555202625
1.012782207207705  -0.537923170210472  -1.012782207207705  0.00380152555202625
-1.012782207207705  0.537923170210472  -1.012782207207705  0.00380152555202625
-1.012782207207705  -0.537923170210472  -1.012782207207705  0.00380152555202625
0.944392525641525  0.688378084301308  0.372427809150424  0.0381806114347457
0.944392525641525  0.688378084301308  -0.372427809150424  0.0381806114347457
0.944392525641525  -0.688378084301308  0.372427809150424  0.0381806114347457
0.944392525641525  -0.688378084301308  -0.372427809150424  0.0381806114347457
0.944392525641525  0.372427809150424  0.688378084301308  0.0381806114347457
0.944392525641525  -0.372427809150424  0.688378084301308  0.0381806114347457
0.944392525641525  0.372427809150424  -0.688378084301308  0.0381806114347457
0.944392525641525  -0.372427809150424  -0.688378084301308  0.0381806114347457
-0.944392525641525  0.688378084301308  0.372427809150424  0.0381806114347457
-0.944392525641525  0.688378084301308  -0.372427809150424  0.0381806114347457
-0.944392525641525  -0.688378084301308  0.372427809150424  0.0381806114347457
-0.944392525641525  -0.688378084301308  -0.372427809150424  0.0381806114347457
-0.944392525641525  0.372427809150424  0.688378084301308  0.0381806114347457
-0.944392525641525  -0.372427809150424  0.688378084301308  0.0381806114347457
-0.944392525641525  0.372427809150424  -0.688378084301308  0.0381806114347457
-0.944392525641525  -0.372427809150424  -0.688378084301308  0.0381806114347457
0.372427809150424  0.944392525641525  0.688378084301308  0.0381806114347457
-0.372427809150424  0.944392525641525  0.688378084301308  0.0381806114347457
0.372427809150424  0.944392525641525  -0.688378084301308  0.0381806114347457
-0.372427809150424  0.944392525641525  -0.688378084301308  0.0381806114347457
0.688378084301308  0.944392525641525  0.372427809150424  0.0381806114347457
0.688378084301308  0.944392525641525  -0.372427809150424  0.0381806114347457
-0.688378084301308  0.944392525641525  0.372427809150424  0.0381806114347457
-0.688378084301308  0.944392525641525  -0.372427809150424  0.0381806114347457
0.372427809150424  -0.944392525641525  0.688378084301308  0.0381806114347457
-0.372427809150424  -0.944392525641525  0.688378084301308  0.0381806114347457
0.372427809150424  -0.944392525641525  -0.688378084301308  0.0381806114347457
-0.372427809150424  -0.944392525641525  -0.688378084301308  0.0381806114347457
0.688378084301308  -0.944392525641525  0.372427809150424  0.0381806114347457
0.688378084301308  -0.944392525641525  -0.372427809150424  0.0381806114347457
-0.688378084301308  -0.944392525641525  0.372427809150424  0.0381806114347457
-0.688378084301308  -0.944392525641525  -0.372427809150424  0.0381806114347457
0.688378084301308  0.372427809150424  0.944392525641525  0.0381806114347457
0.688378084301308  -0.372427809150424  0.944392525641525  0.0381806114347457
-0.688378084301308  0.372427809150424  0.944392525641525  0.0381806114347457
-0.688378084301308  -0.372427809150424  0.944392525641525  0.0381806114347457
0.372427809150424  0.688378084301308  0.944392525641525  0.0381806114347457
-0.372427809150424  0.688378084301308  0.944392525641525  0.0381806114347457
0.372427809150424  -0.688378084301308  0.944392525641525  0.0381806114347457
-0.372427809150424  -0.688378084301308  0.944392525641525  0.0381806114347457
0.688378084301308  0.372427809150424  -0.944392525641525  0.0381806114347457
0.688378084301308  -0.372427809150424  -0.944392525641525  0.0381806114347457
-0.688378084301308  0.372427809150424  -0.944392525641525  0.0381806114347457
-0.688378084301308  -0.372427809150424  -0.944392525641525  0.0381806114347457
0.372427809150424  0.688378084301308  -0.944392525641525  0.0381806114347457
-0.372427809150424  0.688378084301308  -0.944392525641525  0.0381806114347457
0.372427809150424  -0.688378084301308  -0.944392525641525  0.0381806114347457
-0.372427809150424  -0.688378084301308  -0.944392525641525  0.0381806114347457
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=12  13

235
0.000000000000000  0.000000000000000  0.000000000000000  0.034732518560372
0.955412599148645  0.000000000000000  0.000000000000000  0.060620835444102
-0.955412599148645  0.000000000000000  0.000000000000000  0.060620835444102
0.000000000000000  0.955412599148645  0.000000000000000  0.060620835444102
0.000000000000000  -0.955412599148645  0.000000000000000  0.060620835444102
0.000000000000000  0.000000000000000  0.955412599148645  0.060620835444102
0.000000000000000  0.000000000000000  -0.955412599148645  0.060620835444102
0.851429576022840  0.000000000000000  0.000000000000000  -0.092661266731244
-0.851429576022840  0.000000000000000  0.000000000000000  -0.092661266731244
0.000000000000000  0.851429576022840  0.000000000000000  -0.092661266731244
0.000000000000000  -0.851429576022840  0.000000000000000  -0.092661266731244
0.000000000000000  0.000000000000000  0.851429576022840  -0.092661266731244
0.000000000000000  0.000000000000000  -0.851429576022840  -0.092661266731244
0.339914995753808  0.000000000000000  0.000000000000000  0.076730128936029
-0.339914995753808  0.000000000000000  0.000000000000000  0.076730128936029
0.000000000000000  0.339914995753808  0.000000000000000  0.076730128936029
0.000000000000000  -0.339914995753808  0.000000000000000  0.076730128936029
0.000000000000000  0.000000000000000  0.339914995753808  0.076730128936029
0.000000000000000  0.000000000000000  -0.339914995753808  0.076730128936029
1.122550051350066  1.122550051350066  1.122550051350066  0.000035467306353
1.122550051350066  -1.122550051350066  1.122550051350066  0.000035467306353
1.122550051350066  1.122550051350066  -1.122550051350066  0.000035467306353
1.122550051350066  -1.122550051350066  -1.122550051350066  0.000035467306353
-1.122550051350066  1.122550051350066  1.122550051350066  0.000035467306353
-1.122550051350066  -1.122550051350066  1.122550051350066  0.000035467306353
-1.122550051350066  1.122550051350066  -1.122550051350066  0.000035467306353
-1.122550051350066  -1.122550051350066  -1.122550051350066  0.000035467306353
0.683321227375114  0.683321227375114  0.683321227375114  0.061339172795445
0.683321227375114  -0.683321227375114  0.683321227375114  0.061339172795445
0.683321227375114  0.683321227375114  -0.683321227375114  0.061339172795445
0.683321227375114  -0.683321227375114  -0.683321227375114  0.061339172795445
-0.683321227375114  0.683321227375114  0.683321227375114  0.061339172795445
-0.683321227375114  -0.683321227375114  0.683321227375114  0.061339172795445
-0.683321227375114  0.683321227375114  -0.683321227375114  0.061339172795445
-0.683321227375114  -0.683321227375114  -0.683321227375114  0.061339172795445
0.398466735636029  0.398466735636029  0.398466735636029  0.123406122314294
0.398466735636029  -0.398466735636029  0.398466735636029  0.123406122314294
0.398466735636029  0.398466735636029  -0.398466735636029  0.123406122314294
0.398466735636029  -0.398466735636029  -0.398466735636029  0.123406122314294
-0.398466735636029  0.398466735636029  0.398466735636029  0.123406122314294
-0.398466735636029  -0.398466735636029  0.398466735636029  0.123406122314294
-0.398466735636029  0.398466735636029  -0.398466735636029  0.123406122314294
-0.398466735636029  -0.398466735636029  -0.398466735636029  0.123406122314294
1.031795356557355  0.870614907692603  0.000000000000000  0.005042040991613
1.031795356557355  -0.870614907692603  0.000000000000000  0.0050420409916130
1.031795356557355  0.000000000000000  0.870614907692603  0.005042040991613
1.031795356557355  0.000000000000000  -0.870614907692603  0.005042040991613
-1.031795356557355  0.870614907692603  0.000000000000000  0.005042040991613
-1.031795356557355  -0.870614907692603  0.000000000000000  0.005042040991613
-1.031795356557355  0.000000000000000  0.870614907692603  0.005042040991613
-1.031795356557355  0.000000000000000  -0.870614907692603  0.005042040991613
0.870614907692603  1.031795356557355  0.000000000000000  0.005042040991613
-0.870614907692603  1.031795356557355  0.000000000000000  0.005042040991613
0.000000000000000  1.031795356557355  0.870614907692603  0.005042040991613
0.000000000000000  1.031795356557355  -0.870614907692603  0.005042040991613
0.870614907692603  -1.031795356557355  0.000000000000000  0.005042040991613
-0.870614907692603  -1.031795356557355  0.000000000000000  0.005042040991613
0.000000000000000  -1.031795356557355  0.870614907692603  0.005042040991613
0.000000000000000  -1.031795356557355  -0.870614907692603  0.005042040991613
0.870614907692603  0.000000000000000  1.031795356557355  0.005042040991613
-0.870614907692603  0.000000000000000  1.031795356557355  0.005042040991613
0.000000000000000  0.870614907692603  1.031795356557355  0.005042040991613
0.000000000000000  -0.870614907692603  1.031795356557355  0.005042040991613
0.870614907692603  0.000000000000000  -1.031795356557355  0.005042040991613
-0.870614907692603  0.000000000000000  -1.031795356557355  0.005042040991613
0.000000000000000  0.870614907692603  -1.031795356557355  0.005042040991613
0.000000000000000  -0.870614907692603  -1.031795356557355  0.005042040991613
0.859754056650996  0.417236245841784  0.000000000000000  0.072704572667140
0.859754056650996  -0.417236245841784  0.000000000000000  0.072704572667140
0.859754056650996  0.000000000000000  0.417236245841784  0.072704572667140
0.859754056650996  0.000000000000000  -0.417236245841784  0.072704572667140
-0.859754056650996  0.417236245841784  0.000000000000000  0.072704572667140
-0.859754056650996  -0.417236245841784  0.000000000000000  0.072704572667140
-0.859754056650996  0.000000000000000  0.417236245841784  0.072704572667140
-0.859754056650996  0.000000000000000  -0.417236245841784  0.072704572667140
0.417236245841784  0.859754056650996  0.000000000000000  0.072704572667140
-0.417236245841784  0.859754056650996  0.000000000000000  0.072704572667140
0.000000000000000  0.859754056650996  0.417236245841784  0.072704572667140
0.000000000000000  0.859754056650996  -0.417236245841784  0.072704572667140
0.417236245841784  -0.859754056650996  0.000000000000000  0.072704572667140
-0.417236245841784  -0.859754056650996  0.000000000000000  0.072704572667140
0.000000000000000  -0.859754056650996  0.417236245841784  0.072704572667140
0.000000000000000  -0.859754056650996  -0.417236245841784  0.072704572667140
0.417236245841784  0.000000000000000  0.859754056650996  0.072704572667140
-0.417236245841784  0.000000000000000  0.859754056650996  0.072704572667140
0.000000000000000  0.417236245841784  0.859754056650996  0.072704572667140
0.000000000000000  -0.417236245841784  0.859754056650996  0.072704572667140
0.417236245841784  0.000000000000000  -0.859754056650996  0.072704572667140
-0.417236245841784  0.000000000000000  -0.859754056650996  0.072704572667140
0.000000000000000  0.417236245841784  -0.859754056650996  0.072704572667140
0.000000000000000  -0.417236245841784  -0.859754056650996  0.072704572667140
0.677373166268794  0.234354064403370  0.000000000000000  0.055578436206074
0.677373166268794  -0.234354064403370  0.000000000000000  0.055578436206074
0.677373166268794  0.000000000000000  0.234354064403370  0.055578436206074
0.677373166268794  0.000000000000000  -0.234354064403370  0.055578436206074
-0.677373166268794  0.234354064403370  0.000000000000000  0.055578436206074
-0.677373166268794  -0.234354064403370  0.000000000000000  0.055578436206074
-0.677373166268794  0.000000000000000  0.234354064403370  0.055578436206074
-0.677373166268794  0.000000000000000  -0.234354064403370  0.055578436206074
0.234354064403370  0.677373166268794  0.000000000000000  0.055578436206074
-0.234354064403370  0.677373166268794  0.000000000000000  0.055578436206074
0.000000000000000  0.677373166268794  0.234354064403370  0.055578436206074
0.000000000000000  0.677373166268794  -0.234354064403370  0.055578436206074
0.234354064403370  -0.677373166268794  0.000000000000000  0.055578436206074
-0.234354064403370  -0.677373166268794  0.000000000000000  0.055578436206074
0.000000000000000  -0.677373166268794  0.234354064403370  0.055578436206074
0.000000000000000  -0.677373166268794  -0.234354064403370  0.055578436206074
0.234354064403370  0.000000000000000  0.677373166268794  0.055578436206074
-0.234354064403370  0.000000000000000  0.677373166268794  0.055578436206074
0.000000000000000  0.234354064403370  0.677373166268794  0.055578436206074
0.000000000000000  -0.234354064403370  0.677373166268794  0.055578436206074
0.234354064403370  0.000000000000000  -0.677373166268794  0.055578436206074
-0.234354064403370  0.000000000000000  -0.677373166268794  0.055578436206074
0.000000000000000  0.234354064403370  -0.677373166268794  0.055578436206074
0.000000000000000  -0.234354064403370  -0.677373166268794  0.055578436206074
0.931914101559387  0.931914101559387  0.710695749151767  0.015152147947642
0.931914101559387  0.931914101559387  -0.710695749151767  0.015152147947642
0.931914101559387  -0.931914101559387  0.710695749151767  0.015152147947642
0.931914101559387  -0.931914101559387  -0.710695749151767  0.015152147947642
-0.931914101559387  0.931914101559387  0.710695749151767  0.015152147947642
-0.931914101559387  0.931914101559387  -0.710695749151767  0.015152147947642
-0.931914101559387  -0.931914101559387  0.710695749151767  0.015152147947642
-0.931914101559387  -0.931914101559387  -0.710695749151767  0.015152147947642
0.931914101559387  0.710695749151767  0.931914101559387  0.015152147947642
0.931914101559387  -0.710695749151767  0.931914101559387  0.015152147947642
0.931914101559387  0.710695749151767  -0.931914101559387  0.015152147947642
0.931914101559387  -0.710695749151767  -0.931914101559387  0.015152147947642
-0.931914101559387  0.710695749151767  0.931914101559387  0.015152147947642
-0.931914101559387  -0.710695749151767  0.931914101559387  0.015152147947642
-0.931914101559387  0.710695749151767  -0.931914101559387  0.015152147947642
-0.931914101559387  -0.710695749151767  -0.931914101559387  0.015152147947642
0.710695749151767  0.931914101559387  0.931914101559387  0.015152147947642
-0.710695749151767  0.931914101559387  0.931914101559387  0.015152147947642
0.710695749151767  0.931914101559387  -0.931914101559387  0.015152147947642
-0.710695749151767  0.931914101559387  -0.931914101559387  0.015152147947642
0.710695749151767  -0.931914101559387  0.931914101559387  0.015152147947642
-0.710695749151767  -0.931914101559387  0.931914101559387  0.015152147947642
0.710695749151767  -0.931914101559387  -0.931914101559387  0.015152147947642
-0.710695749151767  -0.931914101559387  -0.931914101559387  0.015152147947642
0.871331764342344  0.871331764342344  0.277186077584005  0.027062050257263
0.871331764342344  0.871331764342344  -0.277186077584005  0.027062050257263
0.871331764342344  0.277186077584005  0.871331764342344  0.027062050257263
0.871331764342344  -0.277186077584005  0.871331764342344  0.027062050257263
0.871331764342344  -0.871331764342344  0.277186077584005  0.027062050257263
0.871331764342344  -0.871331764342344  -0.277186077584005  0.027062050257263
0.871331764342344  0.277186077584005  -0.871331764342344  0.027062050257263
0.871331764342344  -0.277186077584005  -0.871331764342344  0.027062050257263
-0.871331764342344  0.871331764342344  0.277186077584005  0.027062050257263
-0.871331764342344  0.871331764342344  -0.277186077584005  0.027062050257263
-0.871331764342344  0.277186077584005  0.871331764342344  0.027062050257263
-0.871331764342344  -0.277186077584005  0.871331764342344  0.027062050257263
-0.871331764342344  -0.871331764342344  0.277186077584005  0.027062050257263
-0.871331764342344  -0.871331764342344  -0.277186077584005  0.027062050257263
-0.871331764342344  0.277186077584005  -0.871331764342344  0.027062050257263
-0.871331764342344  -0.277186077584005  -0.871331764342344  0.027062050257263
0.277186077584005  0.871331764342344  0.871331764342344  0.027062050257263
0.277186077584005  0.871331764342344  -0.871331764342344  0.027062050257263
0.277186077584005  -0.871331764342344  0.871331764342344  0.027062050257263
0.277186077584005  -0.871331764342344  -0.871331764342344  0.027062050257263
-0.277186077584005  0.871331764342344  0.871331764342344  0.027062050257263
-0.277186077584005  0.871331764342344  -0.871331764342344  0.027062050257263
-0.277186077584005  -0.871331764342344  0.871331764342344  0.027062050257263
-0.277186077584005  -0.871331764342344  -0.871331764342344  0.027062050257263
0.693611775581488  0.693611775581488  0.258751904925954  0.041892267526622
0.693611775581488  0.693611775581488  -0.258751904925954  0.041892267526622
0.693611775581488  0.258751904925954  0.693611775581488  0.041892267526622
0.693611775581488  -0.258751904925954  0.693611775581488  0.041892267526622
0.693611775581488  -0.693611775581488  0.258751904925954  0.041892267526622
0.693611775581488  -0.693611775581488  -0.258751904925954  0.041892267526622
0.693611775581488  0.258751904925954  -0.693611775581488  0.041892267526622
0.693611775581488  -0.258751904925954  -0.693611775581488  0.041892267526622
-0.693611775581488  0.693611775581488  0.258751904925954  0.041892267526622
-0.693611775581488  0.693611775581488  -0.258751904925954  0.041892267526622
-0.693611775581488  0.258751904925954  0.693611775581488  0.041892267526622
-0.693611775581488  -0.258751904925954  0.693611775581488  0.041892267526622
-0.693611775581488  -0.693611775581488  0.258751904925954  0.041892267526622
-0.693611775581488  -0.693611775581488  -0.258751904925954  0.041892267526622
-0.693611775581488  0.258751904925954  -0.693611775581488  0.041892267526622
-0.693611775581488  -0.258751904925954  -0.693611775581488  0.041892267526622
0.258751904925954  0.693611775581488  0.693611775581488  0.041892267526622
0.258751904925954  0.693611775581488  -0.693611775581488  0.041892267526622
0.258751904925954  -0.693611775581488  0.693611775581488  0.041892267526622
0.258751904925954  -0.693611775581488  -0.693611775581488  0.041892267526622
-0.258751904925954  0.693611775581488  0.693611775581488  0.041892267526622
-0.258751904925954  0.693611775581488  -0.693611775581488  0.041892267526622
-0.258751904925954  -0.693611775581488  0.693611775581488  0.041892267526622
-0.258751904925954  -0.693611775581488  -0.693611775581488  0.041892267526622
0.959113457650743  0.622683549553047  0.406412429958087  0.020844308789689
0.959113457650743  0.622683549553047  -0.406412429958087  0.020844308789689
0.959113457650743  0.406412429958087  0.622683549553047  0.020844308789689
0.959113457650743  -0.406412429958087  0.622683549553047  0.020844308789689
0.959113457650743  -0.622683549553047  0.406412429958087  0.020844308789689
0.959113457650743  -0.622683549553047  -0.406412429958087  0.020844308789689
0.959113457650743  0.406412429958087  -0.622683549553047  0.020844308789689
0.959113457650743  -0.406412429958087  -0.622683549553047  0.020844308789689
-0.959113457650743  0.622683549553047  0.406412429958087  0.020844308789689
-0.959113457650743  0.622683549553047  -0.406412429958087  0.020844308789689
-0.959113457650743  0.406412429958087  0.622683549553047  0.020844308789689
-0.959113457650743  -0.406412429958087  0.622683549553047  0.020844308789689
-0.959113457650743  -0.622683549553047  0.406412429958087  0.020844308789689
-0.959113457650743  -0.622683549553047  -0.406412429958087  0.020844308789689
-0.959113457650743  0.406412429958087  -0.622683549553047  0.020844308789689
-0.959113457650743  -0.406412429958087  -0.622683549553047  0.020844308789689
0.622683549553047  0.959113457650743  0.406412429958087  0.020844308789689
0.622683549553047  0.959113457650743  -0.406412429958087  0.020844308789689
0.406412429958087  0.959113457650743  0.622683549553047  0.020844308789689
-0.406412429958087  0.959113457650743  0.622683549553047  0.020844308789689
-0.622683549553047  0.959113457650743  0.406412429958087  0.020844308789689
-0.622683549553047  0.959113457650743  -0.406412429958087  0.020844308789689
0.406412429958087  0.959113457650743  -0.622683549553047  0.020844308789689
-0.406412429958087  0.959113457650743  -0.622683549553047  0.020844308789689
0.622683549553047  -0.959113457650743  0.406412429958087  0.020844308789689
0.622683549553047  -0.959113457650743  -0.406412429958087  0.020844308789689
0.406412429958087  -0.959113457650743  0.622683549553047  0.020844308789689
-0.406412429958087  -0.959113457650743  0.622683549553047  0.020844308789689
-0.622683549553047  -0.959113457650743  0.406412429958087  0.020844308789689
-0.622683549553047  -0.959113457650743  -0.406412429958087  0.020844308789689
0.406412429958087  -0.959113457650743  -0.622683549553047  0.020844308789689
-0.406412429958087  -0.959113457650743  -0.622683549553047  0.0208443087896892
0.622683549553047  0.406412429958087  0.959113457650743  0.020844308789689
0.622683549553047  -0.406412429958087  0.959113457650743  0.020844308789689
0.406412429958087  0.622683549553047  0.959113457650743  0.020844308789689
-0.406412429958087  0.622683549553047  0.959113457650743  0.020844308789689
-0.622683549553047  0.406412429958087  0.959113457650743  0.020844308789689
-0.622683549553047  -0.406412429958087  0.959113457650743  0.020844308789689
0.406412429958087  -0.622683549553047  0.959113457650743  0.020844308789689
-0.406412429958087  -0.622683549553047  0.959113457650743  0.020844308789689
0.622683549553047  0.406412429958087  -0.959113457650743  0.020844308789689
0.622683549553047  -0.406412429958087  -0.959113457650743  0.020844308789689
0.406412429958087  0.622683549553047  -0.959113457650743  0.020844308789689
-0.406412429958087  0.622683549553047  -0.959113457650743  0.020844308789689
-0.622683549553047  0.406412429958087  -0.959113457650743  0.020844308789689
-0.622683549553047  -0.406412429958087  -0.959113457650743  0.020844308789689
0.406412429958087  -0.622683549553047  -0.959113457650743  0.020844308789689
-0.406412429958087  -0.622683549553047  -0.959113457650743  0.020844308789689
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=14  15

307
0.000000000000000  0.000000000000000  0.000000000000000  0.127375023441727
1.086788874756602  0.000000000000000  0.000000000000000  -0.006413377967131
-1.086788874756602  0.000000000000000  0.000000000000000  -0.006413377967131
0.000000000000000  1.086788874756602  0.000000000000000  -0.006413377967131
0.000000000000000  -1.086788874756602  0.000000000000000  -0.006413377967131
0.000000000000000  0.000000000000000  1.086788874756602  -0.006413377967131
0.000000000000000  0.000000000000000  -1.086788874756602  -0.006413377967131
0.922008663747819  0.000000000000000  0.000000000000000  0.050925999450554
-0.922008663747819  0.000000000000000  0.000000000000000  0.050925999450554
0.000000000000000  0.922008663747819  0.000000000000000  0.050925999450554
0.000000000000000  -0.922008663747819  0.000000000000000  0.050925999450554
0.000000000000000  0.000000000000000  0.922008663747819  0.050925999450554
0.000000000000000  0.000000000000000  -0.922008663747819  0.050925999450554
0.641870173734099  0.000000000000000  0.000000000000000  -0.113461087562205
-0.641870173734099  0.000000000000000  0.000000000000000  -0.113461087562205
0.000000000000000  0.641870173734099  0.000000000000000  -0.113461087562205
0.000000000000000  -0.641870173734099  0.000000000000000  -0.113461087562205
0.000000000000000  0.000000000000000  0.641870173734099  -0.113461087562205
0.000000000000000  0.000000000000000  -0.641870173734099  -0.113461087562205
0.891607017924525  0.891607017924525  0.891607017924525  0.011115161935589
0.891607017924525  -0.891607017924525  0.891607017924525  0.011115161935589
0.891607017924525  0.891607017924525  -0.891607017924525  0.011115161935589
0.891607017924525  -0.891607017924525  -0.891607017924525  0.011115161935589
-0.891607017924525  0.891607017924525  0.891607017924525  0.011115161935589
-0.891607017924525  -0.891607017924525  0.891607017924525  0.011115161935589
-0.891607017924525  0.891607017924525  -0.891607017924525  0.011115161935589
-0.891607017924525  -0.891607017924525  -0.891607017924525  0.011115161935589
0.734488336558120  0.734488336558120  0.734488336558120  0.037186427226959
0.734488336558120  -0.734488336558120  0.734488336558120  0.037186427226959
0.734488336558120  0.734488336558120  -0.734488336558120  0.037186427226959
0.734488336558120  -0.734488336558120  -0.734488336558120  0.037186427226959
-0.734488336558120  0.734488336558120  0.734488336558120  0.037186427226959
-0.734488336558120  -0.734488336558120  0.734488336558120  0.037186427226959
-0.734488336558120  0.734488336558120  -0.734488336558120  0.037186427226959
-0.734488336558120  -0.734488336558120  -0.734488336558120  0.037186427226959
0.518312376398533  0.518312376398533  0.518312376398533  0.077218674032468
0.518312376398533  -0.518312376398533  0.518312376398533  0.077218674032468
0.518312376398533  0.518312376398533  -0.518312376398533  0.077218674032468
0.518312376398533  -0.518312376398533  -0.518312376398533  0.077218674032468
-0.518312376398533  0.518312376398533  0.518312376398533  0.077218674032468
-0.518312376398533  -0.518312376398533  0.518312376398533  0.077218674032468
-0.518312376398533  0.518312376398533  -0.518312376398533  0.077218674032468
-0.518312376398533  -0.518312376398533  -0.518312376398533  0.077218674032468
0.989312892186012  0.989312892186012  0.000000000000000  0.002134417397357
0.989312892186012  0.000000000000000  0.989312892186012  0.002134417397357
0.989312892186012  -0.989312892186012  0.000000000000000  0.002134417397357
0.989312892186012  0.000000000000000  -0.989312892186012  0.002134417397357
0.000000000000000  0.989312892186012  -0.989312892186012  0.002134417397357
0.000000000000000  -0.989312892186012  0.989312892186012  0.002134417397357
0.000000000000000  -0.989312892186012  -0.989312892186012  0.002134417397357
0.000000000000000  0.989312892186012  0.989312892186012  0.002134417397357
-0.989312892186012  0.989312892186012  0.000000000000000  0.002134417397357
-0.989312892186012  0.000000000000000  0.989312892186012  0.002134417397357
-0.989312892186012  -0.989312892186012  0.000000000000000  0.002134417397357
-0.989312892186012  0.000000000000000  -0.989312892186012  0.002134417397357
0.364474560584746  0.364474560584746  0.000000000000000  0.080804712012006
0.364474560584746  0.000000000000000  0.364474560584746  0.080804712012006
0.364474560584746  -0.364474560584746  0.000000000000000  0.080804712012006
0.364474560584746  0.000000000000000  -0.364474560584746  0.080804712012006
0.000000000000000  0.364474560584746  0.364474560584746  0.080804712012006
0.000000000000000  -0.364474560584746  0.364474560584746  0.080804712012006
0.000000000000000  0.364474560584746  -0.364474560584746  0.080804712012006
0.000000000000000  -0.364474560584746  -0.364474560584746  0.080804712012006
-0.364474560584746  0.000000000000000  0.364474560584746  0.080804712012006
-0.364474560584746  0.364474560584746  0.000000000000000  0.080804712012006
-0.364474560584746  0.000000000000000  -0.364474560584746  0.080804712012006
-0.364474560584746  -0.364474560584746  0.000000000000000  0.080804712012006
0.948247130849454  0.780127958605308  0.000000000000000  0.015903500138507
0.948247130849454  0.000000000000000  0.780127958605308  0.015903500138507
0.948247130849454  -0.780127958605308  0.000000000000000  0.015903500138507
0.948247130849454  0.000000000000000  -0.780127958605308  0.015903500138507
-0.948247130849454  0.780127958605308  0.000000000000000  0.015903500138507
-0.948247130849454  0.000000000000000  0.780127958605308  0.015903500138507
-0.948247130849454  -0.780127958605308  0.000000000000000  0.015903500138507
-0.948247130849454  0.000000000000000  -0.780127958605308  0.015903500138507
0.780127958605308  0.948247130849454  0.000000000000000  0.015903500138507
0.000000000000000  0.948247130849454  0.780127958605308  0.015903500138507
-0.780127958605308  0.948247130849454  0.000000000000000  0.015903500138507
0.000000000000000  0.948247130849454  -0.780127958605308  0.015903500138507
0.780127958605308  -0.948247130849454  0.000000000000000  0.015903500138507
0.000000000000000  -0.948247130849454  0.780127958605308  0.015903500138507
-0.780127958605308  -0.948247130849454  0.000000000000000  0.015903500138507
0.000000000000000  -0.948247130849454  -0.780127958605308  0.015903500138507
0.780127958605308  0.000000000000000  0.948247130849454  0.015903500138507
0.000000000000000  0.780127958605308  0.948247130849454  0.015903500138507
-0.780127958605308  0.000000000000000  0.948247130849454  0.015903500138507
0.000000000000000  -0.780127958605308  0.948247130849454  0.015903500138507
0.780127958605308  0.000000000000000  -0.948247130849454  0.015903500138507
0.000000000000000  0.780127958605308  -0.948247130849454  0.015903500138507
-0.780127958605308  0.000000000000000  -0.948247130849454  0.015903500138507
0.000000000000000  -0.780127958605308  -0.948247130849454  0.015903500138507
1.045702906353664  0.348818446769418  0.000000000000000  0.007654491886780
1.045702906353664  0.000000000000000  0.348818446769418  0.007654491886780
1.045702906353664  -0.348818446769418  0.000000000000000  0.007654491886780
1.045702906353664  0.000000000000000  -0.348818446769418  0.007654491886780
-1.045702906353664  0.348818446769418  0.000000000000000  0.007654491886780
-1.045702906353664  0.000000000000000  0.348818446769418  0.007654491886780
-1.045702906353664  -0.348818446769418  0.000000000000000  0.007654491886780
-1.045702906353664  0.000000000000000  -0.348818446769418  0.007654491886780
0.348818446769418  1.045702906353664  0.000000000000000  0.007654491886780
0.000000000000000  1.045702906353664  0.348818446769418  0.007654491886780
-0.348818446769418  1.045702906353664  0.000000000000000  0.007654491886780
0.000000000000000  1.045702906353664  -0.348818446769418  0.007654491886780
0.348818446769418  -1.045702906353664  0.000000000000000  0.007654491886780
0.000000000000000  -1.045702906353664  0.348818446769418  0.007654491886780
-0.348818446769418  -1.045702906353664  0.000000000000000  0.007654491886780
0.000000000000000  -1.045702906353664  -0.348818446769418  0.007654491886780
0.348818446769418  0.000000000000000  1.045702906353664  0.007654491886780
0.000000000000000  0.348818446769418  1.045702906353664  0.007654491886780
-0.348818446769418  0.000000000000000  1.045702906353664  0.007654491886780
0.000000000000000  -0.348818446769418  1.045702906353664  0.007654491886780
0.348818446769418  0.000000000000000  -1.045702906353664  0.007654491886780
0.000000000000000  0.348818446769418  -1.045702906353664  0.007654491886780
-0.348818446769418  0.000000000000000  -1.045702906353664  0.007654491886780
0.000000000000000  -0.348818446769418  -1.045702906353664  0.007654491886780
0.660531051473545  0.241437081938889  0.000000000000000  0.085528406261081
0.660531051473545  0.000000000000000  0.241437081938889  0.085528406261081
0.660531051473545  -0.241437081938889  0.000000000000000  0.085528406261081
0.660531051473545  0.000000000000000  -0.241437081938889  0.085528406261081
-0.660531051473545  0.241437081938889  0.000000000000000  0.085528406261081
-0.660531051473545  0.000000000000000  0.241437081938889  0.085528406261081
-0.660531051473545  -0.241437081938889  0.000000000000000  0.085528406261081
-0.660531051473545  0.000000000000000  -0.241437081938889  0.085528406261081
0.241437081938889  0.660531051473545  0.000000000000000  0.085528406261081
0.000000000000000  0.660531051473545  0.241437081938889  0.085528406261081
-0.241437081938889  0.660531051473545  0.000000000000000  0.085528406261081
0.000000000000000  0.660531051473545  -0.241437081938889  0.085528406261081
0.241437081938889  -0.660531051473545  0.000000000000000  0.085528406261081
0.000000000000000  -0.660531051473545  0.241437081938889  0.085528406261081
-0.241437081938889  -0.660531051473545  0.000000000000000  0.085528406261081
0.000000000000000  -0.660531051473545  -0.241437081938889  0.085528406261081
0.241437081938889  0.000000000000000  0.660531051473545  0.085528406261081
0.000000000000000  0.241437081938889  0.660531051473545  0.085528406261081
-0.241437081938889  0.000000000000000  0.660531051473545  0.085528406261081
0.000000000000000  -0.241437081938889  0.660531051473545  0.085528406261081
0.241437081938889  0.000000000000000  -0.660531051473545  0.085528406261081
0.000000000000000  0.241437081938889  -0.660531051473545  0.085528406261081
-0.241437081938889  0.000000000000000  -0.660531051473545  0.085528406261081
0.000000000000000  -0.241437081938889  -0.660531051473545  0.085528406261081
0.990502583320736  0.990502583320736  0.757226655088057  0.001979912538349
0.990502583320736  0.990502583320736  -0.757226655088057  0.001979912538349
0.990502583320736  0.757226655088057  0.990502583320736  0.001979912538349
0.990502583320736  -0.757226655088057  0.990502583320736  0.001979912538349
0.990502583320736  -0.990502583320736  0.757226655088057  0.001979912538349
0.990502583320736  -0.990502583320736  -0.757226655088057  0.001979912538349
0.990502583320736  0.757226655088057  -0.990502583320736  0.001979912538349
0.990502583320736  -0.757226655088057  -0.990502583320736  0.001979912538349
-0.990502583320736  0.990502583320736  0.757226655088057  0.001979912538349
-0.990502583320736  0.990502583320736  -0.757226655088057  0.001979912538349
-0.990502583320736  0.757226655088057  0.990502583320736  0.001979912538349
-0.990502583320736  -0.757226655088057  0.990502583320736  0.001979912538349
-0.990502583320736  -0.990502583320736  0.757226655088057  0.001979912538349
-0.990502583320736  -0.990502583320736  -0.757226655088057  0.001979912538349
-0.990502583320736  0.757226655088057  -0.990502583320736  0.001979912538349
-0.990502583320736  -0.757226655088057  -0.990502583320736  0.001979912538349
0.757226655088057  0.990502583320736  0.990502583320736  0.001979912538349
0.757226655088057  0.990502583320736  -0.990502583320736  0.001979912538349
0.757226655088057  -0.990502583320736  0.990502583320736  0.001979912538349
0.757226655088057  -0.990502583320736  -0.990502583320736  0.001979912538349
-0.757226655088057  0.990502583320736  0.990502583320736  0.001979912538349
-0.757226655088057  0.990502583320736  -0.990502583320736  0.001979912538349
-0.757226655088057  -0.990502583320736  0.990502583320736  0.001979912538349
-0.757226655088057  -0.990502583320736  -0.990502583320736  0.001979912538349
0.926857250551066  0.926857250551066  0.315061197287156  0.008143142918418
0.926857250551066  0.926857250551066  -0.315061197287156  0.008143142918418
0.926857250551066  0.315061197287156  0.926857250551066  0.008143142918418
0.926857250551066  -0.315061197287156  0.926857250551066  0.008143142918418
0.926857250551066  -0.926857250551066  0.315061197287156  0.008143142918418
0.926857250551066  -0.926857250551066  -0.315061197287156  0.008143142918418
0.926857250551066  0.315061197287156  -0.926857250551066  0.008143142918418
0.926857250551066  -0.315061197287156  -0.926857250551066  0.008143142918418
-0.926857250551066  0.926857250551066  0.315061197287156  0.008143142918418
-0.926857250551066  0.926857250551066  -0.315061197287156  0.008143142918418
-0.926857250551066  0.315061197287156  0.926857250551066  0.008143142918418
-0.926857250551066  -0.315061197287156  0.926857250551066  0.008143142918418
-0.926857250551066  -0.926857250551066  0.315061197287156  0.008143142918418
-0.926857250551066  -0.926857250551066  -0.315061197287156  0.008143142918418
-0.926857250551066  0.315061197287156  -0.926857250551066  0.008143142918418
-0.926857250551066  -0.315061197287156  -0.926857250551066  0.008143142918418
0.315061197287156  0.926857250551066  0.926857250551066  0.008143142918418
0.315061197287156  0.926857250551066  -0.926857250551066  0.008143142918418
0.315061197287156  -0.926857250551066  0.926857250551066  0.008143142918418
0.315061197287156  -0.926857250551066  -0.926857250551066  0.008143142918418
-0.315061197287156  0.926857250551066  0.926857250551066  0.008143142918418
-0.315061197287156  0.926857250551066  -0.926857250551066  0.008143142918418
-0.315061197287156  -0.926857250551066  0.926857250551066  0.008143142918418
-0.315061197287156  -0.926857250551066  -0.926857250551066  0.008143142918418
0.710700783405400  0.710700783405400  0.208469339137832  0.035763262755692
0.710700783405400  0.710700783405400  -0.208469339137832  0.035763262755692
0.710700783405400  0.208469339137832  0.710700783405400  0.035763262755692
0.710700783405400  -0.208469339137832  0.710700783405400  0.035763262755692
0.710700783405400  -0.710700783405400  0.208469339137832  0.035763262755692
0.710700783405400  -0.710700783405400  -0.208469339137832  0.035763262755692
0.710700783405400  0.208469339137832  -0.710700783405400  0.035763262755692
0.710700783405400  -0.208469339137832  -0.710700783405400  0.035763262755692
-0.710700783405400  0.710700783405400  0.208469339137832  0.035763262755692
-0.710700783405400  0.710700783405400  -0.208469339137832  0.035763262755692
-0.710700783405400  0.208469339137832  0.710700783405400  0.035763262755692
-0.710700783405400  -0.208469339137832  0.710700783405400  0.035763262755692
-0.710700783405400  -0.710700783405400  0.208469339137832  0.035763262755692
-0.710700783405400  -0.710700783405400  -0.208469339137832  0.035763262755692
-0.710700783405400  0.208469339137832  -0.710700783405400  0.035763262755692
-0.710700783405400  -0.208469339137832  -0.710700783405400  0.035763262755692
0.208469339137832  0.710700783405400  0.710700783405400  0.035763262755692
0.208469339137832  0.710700783405400  -0.710700783405400  0.035763262755692
0.208469339137832  -0.710700783405400  0.710700783405400  0.035763262755692
0.208469339137832  -0.710700783405400  -0.710700783405400  0.035763262755692
-0.208469339137832  0.710700783405400  0.710700783405400  0.035763262755692
-0.208469339137832  0.710700783405400  -0.710700783405400  0.035763262755692
-0.208469339137832  -0.710700783405400  0.710700783405400  0.035763262755692
-0.208469339137832  -0.710700783405400  -0.710700783405400  0.035763262755692
0.957873006911044  0.780040197522315  0.527111921367462  0.0155500497623716
0.957873006911044  0.780040197522315  -0.527111921367462  0.0155500497623716
0.957873006911044  0.527111921367462  0.780040197522315  0.0155500497623716
0.957873006911044  -0.527111921367462  0.780040197522315  0.0155500497623716
0.957873006911044  -0.780040197522315  0.527111921367462  0.0155500497623716
0.957873006911044  -0.780040197522315  -0.527111921367462  0.0155500497623716
0.957873006911044  0.527111921367462  -0.780040197522315  0.0155500497623716
0.957873006911044  -0.527111921367462  -0.780040197522315  0.0155500497623716
-0.957873006911044  0.780040197522315  0.527111921367462  0.0155500497623716
-0.957873006911044  0.780040197522315  -0.527111921367462  0.0155500497623716
-0.957873006911044  0.527111921367462  0.780040197522315  0.0155500497623716
-0.957873006911044  -0.527111921367462  0.780040197522315  0.0155500497623716
-0.957873006911044  -0.780040197522315  0.527111921367462  0.0155500497623716
-0.957873006911044  -0.780040197522315  -0.527111921367462  0.0155500497623716
-0.957873006911044  0.527111921367462  -0.780040197522315  0.0155500497623716
-0.957873006911044  -0.527111921367462  -0.780040197522315  0.0155500497623716
0.780040197522315  0.957873006911044  0.527111921367462  0.0155500497623716
0.780040197522315  0.957873006911044  -0.527111921367462  0.0155500497623716
0.780040197522315  0.527111921367462  0.957873006911044  0.0155500497623716
0.780040197522315  -0.527111921367462  0.957873006911044  0.0155500497623716
0.780040197522315  -0.957873006911044  0.527111921367462  0.0155500497623716
0.780040197522315  -0.957873006911044  -0.527111921367462  0.0155500497623716
0.780040197522315  0.527111921367462  -0.957873006911044  0.0155500497623716
0.780040197522315  -0.527111921367462  -0.957873006911044  0.0155500497623716
-0.780040197522315  0.957873006911044  0.527111921367462  0.0155500497623716
-0.780040197522315  0.957873006911044  -0.527111921367462  0.0155500497623716
-0.780040197522315  0.527111921367462  0.957873006911044  0.0155500497623716
-0.780040197522315  -0.527111921367462  0.957873006911044  0.0155500497623716
-0.780040197522315  -0.957873006911044  0.527111921367462  0.0155500497623716
-0.780040197522315  -0.957873006911044  -0.527111921367462  0.0155500497623716
-0.780040197522315  0.527111921367462  -0.957873006911044  0.0155500497623716
-0.780040197522315  -0.527111921367462  -0.957873006911044  0.0155500497623716
0.527111921367462  0.957873006911044  0.780040197522315  0.0155500497623716
0.527111921367462  0.957873006911044  -0.780040197522315  0.0155500497623716
0.527111921367462  0.780040197522315  0.957873006911044  0.0155500497623716
0.527111921367462  -0.780040197522315  0.957873006911044  0.0155500497623716
0.527111921367462  -0.957873006911044  0.780040197522315  0.0155500497623716
0.527111921367462  -0.957873006911044  -0.780040197522315  0.0155500497623716
0.527111921367462  0.780040197522315  -0.957873006911044  0.0155500497623716
0.527111921367462  -0.780040197522315  -0.957873006911044  0.0155500497623716
-0.527111921367462  0.957873006911044  0.780040197522315  0.0155500497623716
-0.527111921367462  0.957873006911044  -0.780040197522315  0.0155500497623716
-0.527111921367462  0.780040197522315  0.957873006911044  0.0155500497623716
-0.527111921367462  -0.780040197522315  0.957873006911044  0.0155500497623716
-0.527111921367462  -0.957873006911044  0.780040197522315  0.0155500497623716
-0.527111921367462  -0.957873006911044  -0.780040197522315  0.0155500497623716
-0.527111921367462  0.780040197522315  -0.957873006911044  0.0155500497623716
-0.527111921367462  -0.780040197522315  -0.957873006911044  0.0155500497623716
0.880972079148717  0.513930736899189  0.269433491346948  0.037940344374850
0.880972079148717  0.513930736899189  -0.269433491346948  0.037940344374850
0.880972079148717  0.269433491346948  0.513930736899189  0.037940344374850
0.880972079148717  -0.269433491346948  0.513930736899189  0.037940344374850
0.880972079148717  -0.513930736899189  0.269433491346948  0.037940344374850
0.880972079148717  -0.513930736899189  -0.269433491346948  0.037940344374850
0.880972079148717  0.269433491346948  -0.513930736899189  0.037940344374850
0.880972079148717  -0.269433491346948  -0.513930736899189  0.037940344374850
-0.880972079148717  0.513930736899189  0.269433491346948  0.037940344374850
-0.880972079148717  0.513930736899189  -0.269433491346948  0.037940344374850
-0.880972079148717  0.269433491346948  0.513930736899189  0.037940344374850
-0.880972079148717  -0.269433491346948  0.513930736899189  0.037940344374850
-0.880972079148717  -0.513930736899189  0.269433491346948  0.037940344374850
-0.880972079148717  -0.513930736899189  -0.269433491346948  0.037940344374850
-0.880972079148717  0.269433491346948  -0.513930736899189  0.037940344374850
-0.880972079148717  -0.269433491346948  -0.513930736899189  0.037940344374850
0.513930736899189  0.880972079148717  0.269433491346948  0.037940344374850
0.513930736899189  0.880972079148717  -0.269433491346948  0.037940344374850
0.513930736899189  0.269433491346948  0.880972079148717  0.037940344374850
0.513930736899189  -0.269433491346948  0.880972079148717  0.037940344374850
0.513930736899189  -0.880972079148717  0.269433491346948  0.037940344374850
0.513930736899189  -0.880972079148717  -0.269433491346948  0.037940344374850
0.513930736899189  0.269433491346948  -0.880972079148717  0.037940344374850
0.513930736899189  -0.269433491346948  -0.880972079148717  0.037940344374850
-0.513930736899189  0.880972079148717  0.269433491346948  0.037940344374850
-0.513930736899189  0.880972079148717  -0.269433491346948  0.037940344374850
-0.513930736899189  0.269433491346948  0.880972079148717  0.037940344374850
-0.513930736899189  -0.269433491346948  0.880972079148717  0.037940344374850
-0.513930736899189  -0.880972079148717  0.269433491346948  0.037940344374850
-0.513930736899189  -0.880972079148717  -0.269433491346948  0.037940344374850
-0.513930736899189  0.269433491346948  -0.880972079148717  0.037940344374850
-0.513930736899189  -0.269433491346948  -0.880972079148717  0.037940344374850
0.269433491346948  0.880972079148717  0.513930736899189  0.037940344374850
0.269433491346948  0.880972079148717  -0.513930736899189  0.037940344374850
0.269433491346948  0.513930736899189  0.880972079148717  0.037940344374850
0.269433491346948  -0.513930736899189  0.880972079148717  0.037940344374850
0.269433491346948  -0.880972079148717  0.513930736899189  0.037940344374850
0.269433491346948  -0.880972079148717  -0.513930736899189  0.037940344374850
0.269433491346948  0.513930736899189  -0.880972079148717  0.037940344374850
0.269433491346948  -0.513930736899189  -0.880972079148717  0.037940344374850
-0.269433491346948  0.880972079148717  0.513930736899189  0.037940344374850
-0.269433491346948  0.880972079148717  -0.513930736899189  0.037940344374850
-0.269433491346948  0.513930736899189  0.880972079148717  0.037940344374850
-0.269433491346948  -0.513930736899189  0.880972079148717  0.037940344374850
-0.269433491346948  -0.880972079148717  0.513930736899189  0.037940344374850
-0.269433491346948  -0.880972079148717  -0.513930736899189  0.037940344374850
-0.269433491346948  0.513930736899189  -0.880972079148717  0.037940344374850
-0.269433491346948  -0.513930736899189  -0.880972079148717  0.037940344374850
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=16  17

435
0.000000000000000  0.000000000000000  0.000000000000000  -0.296256080038908
0.950006122416513  0.000000000000000  0.000000000000000  -0.031821314899734
-0.950006122416513  0.000000000000000  0.000000000000000  -0.031821314899734
0.000000000000000  0.950006122416513  0.000000000000000  -0.031821314899734
0.000000000000000  -0.950006122416513  0.000000000000000  -0.031821314899734
0.000000000000000  0.000000000000000  0.950006122416513  -0.031821314899734
0.000000000000000  0.000000000000000  -0.950006122416513  -0.031821314899734
0.914965429121309  0.000000000000000  0.000000000000000  0.084749197382812
-0.914965429121309  0.000000000000000  0.000000000000000  0.084749197382812
0.000000000000000  0.914965429121309  0.000000000000000  0.084749197382812
0.000000000000000  -0.914965429121309  0.000000000000000  0.084749197382812
0.000000000000000  0.000000000000000  0.914965429121309  0.084749197382812
0.000000000000000  0.000000000000000  -0.914965429121309  0.084749197382812
0.822592418605602  0.000000000000000  0.000000000000000  -0.186007315013108
-0.822592418605602  0.000000000000000  0.000000000000000  -0.186007315013108
0.000000000000000  0.822592418605602  0.000000000000000  -0.186007315013108
0.000000000000000  -0.822592418605602  0.000000000000000  -0.186007315013108
0.000000000000000  0.000000000000000  0.822592418605602  -0.186007315013108
0.000000000000000  0.000000000000000  -0.822592418605602  -0.186007315013108
0.652114589201603  0.000000000000000  0.000000000000000  0.119234266342962
-0.652114589201603  0.000000000000000  0.000000000000000  0.119234266342962
0.000000000000000  0.652114589201603  0.000000000000000  0.119234266342962
0.000000000000000  -0.652114589201603  0.000000000000000  0.119234266342962
0.000000000000000  0.000000000000000  0.652114589201603  0.119234266342962
0.000000000000000  0.000000000000000  -0.652114589201603  0.119234266342962
0.157242622070114  0.000000000000000  0.000000000000000  0.092710479799536
-0.157242622070114  0.000000000000000  0.000000000000000  0.092710479799536
0.000000000000000  0.157242622070114  0.000000000000000  0.092710479799536
0.000000000000000  -0.157242622070114  0.000000000000000  0.092710479799536
0.000000000000000  0.000000000000000  0.157242622070114  0.092710479799536
0.000000000000000  0.000000000000000  -0.157242622070114  0.092710479799536
1.001394961101233  1.001394961101233  1.001394961101233  0.060022532736429
1.001394961101233  -1.001394961101233  1.001394961101233  0.060022532736429
1.001394961101233  1.001394961101233  -1.001394961101233  0.060022532736429
1.001394961101233  -1.001394961101233  -1.001394961101233  0.060022532736429
-1.001394961101233  1.001394961101233  1.001394961101233  0.060022532736429
-1.001394961101233  -1.001394961101233  1.001394961101233  0.060022532736429
-1.001394961101233  1.001394961101233  -1.001394961101233  0.060022532736429
-1.001394961101233  -1.001394961101233  -1.001394961101233  0.060022532736429
0.949635648752703  0.949635648752703  0.949635648752703  0.006019976234507
0.949635648752703  -0.949635648752703  0.949635648752703  0.006019976234507
0.949635648752703  0.949635648752703  -0.949635648752703  0.006019976234507
0.949635648752703  -0.949635648752703  -0.949635648752703  0.006019976234507
-0.949635648752703  0.949635648752703  0.949635648752703  0.006019976234507
-0.949635648752703  -0.949635648752703  0.949635648752703  0.006019976234507
-0.949635648752703  0.949635648752703  -0.949635648752703  0.006019976234507
-0.949635648752703  -0.949635648752703  -0.949635648752703  0.006019976234507
0.594117176975313  0.594117176975313  0.594117176975313  0.035689510132508
0.594117176975313  -0.594117176975313  0.594117176975313  0.035689510132508
0.594117176975313  0.594117176975313  -0.594117176975313  0.035689510132508
0.594117176975313  -0.594117176975313  -0.594117176975313  0.035689510132508
-0.594117176975313  0.594117176975313  0.594117176975313  0.035689510132508
-0.594117176975313  -0.594117176975313  0.594117176975313  0.035689510132508
-0.594117176975313  0.594117176975313  -0.594117176975313  0.035689510132508
-0.594117176975313  -0.594117176975313  -0.594117176975313  0.035689510132508
0.326702572971493  0.326702572971493  0.326702572971493  0.081790078663352
0.326702572971493  -0.326702572971493  0.326702572971493  0.081790078663352
0.326702572971493  0.326702572971493  -0.326702572971493  0.081790078663352
0.326702572971493  -0.326702572971493  -0.326702572971493  0.081790078663352
-0.326702572971493  0.326702572971493  0.326702572971493  0.081790078663352
-0.326702572971493  -0.326702572971493  0.326702572971493  0.081790078663352
-0.326702572971493  0.326702572971493  -0.326702572971493  0.081790078663352
-0.326702572971493  -0.326702572971493  -0.326702572971493  0.081790078663352
1.003460985202348  1.003460985202348  0.000000000000000  -0.035944843816282
1.003460985202348  0.000000000000000  1.003460985202348  -0.035944843816282
1.003460985202348  -1.003460985202348  0.000000000000000  -0.035944843816282
1.003460985202348  0.000000000000000  -1.003460985202348  -0.035944843816282
0.000000000000000  1.003460985202348  -1.003460985202348  -0.035944843816282
0.000000000000000  -1.003460985202348  1.003460985202348  -0.035944843816282
0.000000000000000  -1.003460985202348  -1.003460985202348  -0.035944843816282
0.000000000000000  1.003460985202348  1.003460985202348  -0.035944843816282
-1.003460985202348  1.003460985202348  0.000000000000000  -0.035944843816282
-1.003460985202348  0.000000000000000  1.003460985202348  -0.035944843816282
-1.003460985202348  -1.003460985202348  0.000000000000000  -0.035944843816282
-1.003460985202348  0.000000000000000  -1.003460985202348  -0.035944843816282
0.893211074627396  0.893211074627396  0.000000000000000  -0.062784001903098
0.893211074627396  0.000000000000000  0.893211074627396  -0.062784001903098
0.893211074627396  -0.893211074627396  0.000000000000000  -0.062784001903098
0.893211074627396  0.000000000000000  -0.893211074627396  -0.062784001903098
0.000000000000000  0.893211074627396  -0.893211074627396  -0.062784001903098
0.000000000000000  -0.893211074627396  0.893211074627396  -0.062784001903098
0.000000000000000  -0.893211074627396  -0.893211074627396  -0.062784001903098
0.000000000000000  0.893211074627396  0.893211074627396  -0.062784001903098
-0.893211074627396  0.893211074627396  0.000000000000000  -0.062784001903098
-0.893211074627396  0.000000000000000  0.893211074627396  -0.062784001903098
-0.893211074627396  -0.893211074627396  0.000000000000000  -0.062784001903098
-0.893211074627396  0.000000000000000  -0.893211074627396  -0.062784001903098
0.488224948121536  0.488224948121536  0.000000000000000  0.0512421306349611
0.488224948121536  0.000000000000000  0.488224948121536  0.0512421306349611
0.488224948121536  -0.488224948121536  0.000000000000000  0.0512421306349611
0.488224948121536  0.000000000000000  -0.488224948121536  0.0512421306349611
0.000000000000000  0.488224948121536  -0.488224948121536  0.0512421306349611
0.000000000000000  -0.488224948121536  0.488224948121536  0.0512421306349611
0.000000000000000  -0.488224948121536  -0.488224948121536  0.0512421306349611
0.000000000000000  0.488224948121536  0.488224948121536  0.0512421306349611
-0.488224948121536  0.488224948121536  0.000000000000000  0.0512421306349611
-0.488224948121536  0.000000000000000  0.488224948121536  0.0512421306349611
-0.488224948121536  -0.488224948121536  0.000000000000000  0.0512421306349611
-0.488224948121536  0.000000000000000  -0.488224948121536  0.0512421306349611
1.010406331416018  0.988440552758072  0.000000000000000  0.020998703607321
1.010406331416018  0.000000000000000  0.988440552758072  0.020998703607321
1.010406331416018  -0.988440552758072  0.000000000000000  0.020998703607321
1.010406331416018  0.000000000000000  -0.988440552758072  0.020998703607321
-1.010406331416018  0.988440552758072  0.000000000000000  0.020998703607321
-1.010406331416018  0.000000000000000  0.988440552758072  0.020998703607321
-1.010406331416018  -0.988440552758072  0.000000000000000  0.020998703607321
-1.010406331416018  0.000000000000000  -0.988440552758072  0.020998703607321
0.988440552758072  1.010406331416018  0.000000000000000  0.020998703607321
0.000000000000000  1.010406331416018  0.988440552758072  0.020998703607321
-0.988440552758072  1.010406331416018  0.000000000000000  0.020998703607321
0.000000000000000  1.010406331416018  -0.988440552758072  0.020998703607321
0.988440552758072  -1.010406331416018  0.000000000000000  0.020998703607321
0.000000000000000  -1.010406331416018  0.988440552758072  0.020998703607321
-0.988440552758072  -1.010406331416018  0.000000000000000  0.020998703607321
0.000000000000000  -1.010406331416018  -0.988440552758072  0.020998703607321
0.988440552758072  0.000000000000000  1.010406331416018  0.020998703607321
0.000000000000000  0.988440552758072  1.010406331416018  0.020998703607321
-0.988440552758072  0.000000000000000  1.010406331416018  0.020998703607321
0.000000000000000  -0.988440552758072  1.010406331416018  0.020998703607321
0.988440552758072  0.000000000000000  -1.010406331416018  0.020998703607321
0.000000000000000  0.988440552758072  -1.010406331416018  0.020998703607321
-0.988440552758072  0.000000000000000  -1.010406331416018  0.020998703607321
0.000000000000000  -0.988440552758072  -1.010406331416018  0.020998703607321
0.989413303740966  0.384711778165505  0.000000000000000  0.012934351826889
0.989413303740966  0.000000000000000  0.384711778165505  0.012934351826889
0.989413303740966  -0.384711778165505  0.000000000000000  0.012934351826889
0.989413303740966  0.000000000000000  -0.384711778165505  0.012934351826889
-0.989413303740966  0.384711778165505  0.000000000000000  0.012934351826889
-0.989413303740966  0.000000000000000  0.384711778165505  0.012934351826889
-0.989413303740966  -0.384711778165505  0.000000000000000  0.012934351826889
-0.989413303740966  0.000000000000000  -0.384711778165505  0.012934351826889
0.384711778165505  0.989413303740966  0.000000000000000  0.012934351826889
0.000000000000000  0.989413303740966  0.384711778165505  0.012934351826889
-0.384711778165505  0.989413303740966  0.000000000000000  0.012934351826889
0.000000000000000  0.989413303740966  -0.384711778165505  0.012934351826889
0.384711778165505  -0.989413303740966  0.000000000000000  0.012934351826889
0.000000000000000  -0.989413303740966  0.384711778165505  0.012934351826889
-0.384711778165505  -0.989413303740966  0.000000000000000  0.012934351826889
0.000000000000000  -0.989413303740966  -0.384711778165505  0.012934351826889
0.384711778165505  0.000000000000000  0.989413303740966  0.012934351826889
0.000000000000000  0.384711778165505  0.989413303740966  0.012934351826889
-0.384711778165505  0.000000000000000  0.989413303740966  0.012934351826889
0.000000000000000  -0.384711778165505  0.989413303740966  0.012934351826889
0.384711778165505  0.000000000000000  -0.989413303740966  0.012934351826889
0.000000000000000  0.384711778165505  -0.989413303740966  0.012934351826889
-0.384711778165505  0.000000000000000  -0.989413303740966  0.012934351826889
0.000000000000000  -0.384711778165505  -0.989413303740966  0.012934351826889
0.907323294169450  0.833136115100468  0.000000000000000  0.050852195792478
0.907323294169450  0.000000000000000  0.833136115100468  0.050852195792478
0.907323294169450  -0.833136115100468  0.000000000000000  0.050852195792478
0.907323294169450  0.000000000000000  -0.833136115100468  0.050852195792478
-0.907323294169450  0.833136115100468  0.000000000000000  0.050852195792478
-0.907323294169450  0.000000000000000  0.833136115100468  0.050852195792478
-0.907323294169450  -0.833136115100468  0.000000000000000  0.050852195792478
-0.907323294169450  0.000000000000000  -0.833136115100468  0.050852195792478
0.833136115100468  0.907323294169450  0.000000000000000  0.050852195792478
0.000000000000000  0.907323294169450  0.833136115100468  0.050852195792478
-0.833136115100468  0.907323294169450  0.000000000000000  0.050852195792478
0.000000000000000  0.907323294169450  -0.833136115100468  0.050852195792478
0.833136115100468  -0.907323294169450  0.000000000000000  0.050852195792478
0.000000000000000  -0.907323294169450  0.833136115100468  0.050852195792478
-0.833136115100468  -0.907323294169450  0.000000000000000  0.050852195792478
0.000000000000000  -0.907323294169450  -0.833136115100468  0.050852195792478
0.833136115100468  0.000000000000000  0.907323294169450  0.050852195792478
0.000000000000000  0.833136115100468  0.907323294169450  0.050852195792478
-0.833136115100468  0.000000000000000  0.907323294169450  0.050852195792478
0.000000000000000  -0.833136115100468  0.907323294169450  0.050852195792478
0.833136115100468  0.000000000000000  -0.907323294169450  0.050852195792478
0.000000000000000  0.833136115100468  -0.907323294169450  0.050852195792478
-0.833136115100468  0.000000000000000  -0.907323294169450  0.050852195792478
0.000000000000000  -0.833136115100468  -0.907323294169450  0.050852195792478
0.831834985558718  0.220277012103345  0.000000000000000  0.062536981939185
0.831834985558718  0.000000000000000  0.220277012103345  0.062536981939185
0.831834985558718  -0.220277012103345  0.000000000000000  0.062536981939185
0.831834985558718  0.000000000000000  -0.220277012103345  0.062536981939185
-0.831834985558718  0.220277012103345  0.000000000000000  0.062536981939185
-0.831834985558718  0.000000000000000  0.220277012103345  0.062536981939185
-0.831834985558718  -0.220277012103345  0.000000000000000  0.062536981939185
-0.831834985558718  0.000000000000000  -0.220277012103345  0.062536981939185
0.220277012103345  0.831834985558718  0.000000000000000  0.062536981939185
0.000000000000000  0.831834985558718  0.220277012103345  0.062536981939185
-0.220277012103345  0.831834985558718  0.000000000000000  0.062536981939185
0.000000000000000  0.831834985558718  -0.220277012103345  0.062536981939185
0.220277012103345  -0.831834985558718  0.000000000000000  0.062536981939185
0.000000000000000  -0.831834985558718  0.220277012103345  0.062536981939185
-0.220277012103345  -0.831834985558718  0.000000000000000  0.062536981939185
0.000000000000000  -0.831834985558718  -0.220277012103345  0.062536981939185
0.220277012103345  0.000000000000000  0.831834985558718  0.062536981939185
0.000000000000000  0.220277012103345  0.831834985558718  0.062536981939185
-0.220277012103345  0.000000000000000  0.831834985558718  0.062536981939185
0.000000000000000  -0.220277012103345  0.831834985558718  0.062536981939185
0.220277012103345  0.000000000000000  -0.831834985558718  0.062536981939185
0.000000000000000  0.220277012103345  -0.831834985558718  0.062536981939185
-0.220277012103345  0.000000000000000  -0.831834985558718  0.062536981939185
0.000000000000000  -0.220277012103345  -0.831834985558718  0.062536981939185
1.012434400912337  1.012434400912337  0.978322332670673  -0.023175821642762
1.012434400912337  1.012434400912337  -0.978322332670673  -0.023175821642762
1.012434400912337  0.978322332670673  1.012434400912337  -0.023175821642762
1.012434400912337  -0.978322332670673  1.012434400912337  -0.023175821642762
1.012434400912337  -1.012434400912337  0.978322332670673  -0.023175821642762
1.012434400912337  -1.012434400912337  -0.978322332670673  -0.023175821642762
1.012434400912337  0.978322332670673  -1.012434400912337  -0.023175821642762
1.012434400912337  -0.978322332670673  -1.012434400912337  -0.023175821642762
-1.012434400912337  1.012434400912337  0.978322332670673  -0.023175821642762
-1.012434400912337  1.012434400912337  -0.978322332670673  -0.023175821642762
-1.012434400912337  0.978322332670673  1.012434400912337  -0.023175821642762
-1.012434400912337  -0.978322332670673  1.012434400912337  -0.023175821642762
-1.012434400912337  -1.012434400912337  0.978322332670673  -0.023175821642762
-1.012434400912337  -1.012434400912337  -0.978322332670673  -0.023175821642762
-1.012434400912337  0.978322332670673  -1.012434400912337  -0.023175821642762
-1.012434400912337  -0.978322332670673  -1.012434400912337  -0.023175821642762
0.978322332670673  1.012434400912337  1.012434400912337  -0.023175821642762
0.978322332670673  1.012434400912337  -1.012434400912337  -0.023175821642762
0.978322332670673  -1.012434400912337  1.012434400912337  -0.023175821642762
0.978322332670673  -1.012434400912337  -1.012434400912337  -0.023175821642762
-0.978322332670673  1.012434400912337  1.012434400912337  -0.023175821642762
-0.978322332670673  1.012434400912337  -1.012434400912337  -0.023175821642762
-0.978322332670673  -1.012434400912337  1.012434400912337  -0.023175821642762
-0.978322332670673  -1.012434400912337  -1.012434400912337  -0.023175821642762
0.940302135044088  0.940302135044088  0.343148808079936  0.009713678246504
0.940302135044088  0.940302135044088  -0.343148808079936  0.009713678246504
0.940302135044088  0.343148808079936  0.940302135044088  0.009713678246504
0.940302135044088  -0.343148808079936  0.940302135044088  0.009713678246504
0.940302135044088  -0.940302135044088  0.343148808079936  0.009713678246504
0.940302135044088  -0.940302135044088  -0.343148808079936  0.009713678246504
0.940302135044088  0.343148808079936  -0.940302135044088  0.009713678246504
0.940302135044088  -0.343148808079936  -0.940302135044088  0.009713678246504
-0.940302135044088  0.940302135044088  0.343148808079936  0.009713678246504
-0.940302135044088  0.940302135044088  -0.343148808079936  0.009713678246504
-0.940302135044088  0.343148808079936  0.940302135044088  0.009713678246504
-0.940302135044088  -0.343148808079936  0.940302135044088  0.009713678246504
-0.940302135044088  -0.940302135044088  0.343148808079936  0.009713678246504
-0.940302135044088  -0.940302135044088  -0.343148808079936  0.009713678246504
-0.940302135044088  0.343148808079936  -0.940302135044088  0.009713678246504
-0.940302135044088  -0.343148808079936  -0.940302135044088  0.009713678246504
0.343148808079936  0.940302135044088  0.940302135044088  0.009713678246504
0.343148808079936  0.940302135044088  -0.940302135044088  0.009713678246504
0.343148808079936  -0.940302135044088  0.940302135044088  0.009713678246504
0.343148808079936  -0.940302135044088  -0.940302135044088  0.009713678246504
-0.343148808079936  0.940302135044088  0.940302135044088  0.009713678246504
-0.343148808079936  0.940302135044088  -0.940302135044088  0.009713678246504
-0.343148808079936  -0.940302135044088  0.940302135044088  0.009713678246504
-0.343148808079936  -0.940302135044088  -0.940302135044088  0.009713678246504
1.036512025425065  1.036512025425065  0.933501389902834  0.002666348244400
1.036512025425065  1.036512025425065  -0.933501389902834  0.002666348244400
1.036512025425065  0.933501389902834  1.036512025425065  0.002666348244400
1.036512025425065  -0.933501389902834  1.036512025425065  0.002666348244400
1.036512025425065  -1.036512025425065  0.933501389902834  0.002666348244400
1.036512025425065  -1.036512025425065  -0.933501389902834  0.002666348244400
1.036512025425065  0.933501389902834  -1.036512025425065  0.002666348244400
1.036512025425065  -0.933501389902834  -1.036512025425065  0.002666348244400
-1.036512025425065  1.036512025425065  0.933501389902834  0.002666348244400
-1.036512025425065  1.036512025425065  -0.933501389902834  0.002666348244400
-1.036512025425065  0.933501389902834  1.036512025425065  0.002666348244400
-1.036512025425065  -0.933501389902834  1.036512025425065  0.002666348244400
-1.036512025425065  -1.036512025425065  0.933501389902834  0.002666348244400
-1.036512025425065  -1.036512025425065  -0.933501389902834  0.002666348244400
-1.036512025425065  0.933501389902834  -1.036512025425065  0.002666348244400
-1.036512025425065  -0.933501389902834  -1.036512025425065  0.002666348244400
0.933501389902834  1.036512025425065  1.036512025425065  0.002666348244400
0.933501389902834  1.036512025425065  -1.036512025425065  0.002666348244400
0.933501389902834  -1.036512025425065  1.036512025425065  0.002666348244400
0.933501389902834  -1.036512025425065  -1.036512025425065  0.002666348244400
-0.933501389902834  1.036512025425065  1.036512025425065  0.002666348244400
-0.933501389902834  1.036512025425065  -1.036512025425065  0.002666348244400
-0.933501389902834  -1.036512025425065  1.036512025425065  0.002666348244400
-0.933501389902834  -1.036512025425065  -1.036512025425065  0.002666348244400
0.827104008787951  0.827104008787951  0.595273916358421  0.0238739030828427
0.827104008787951  0.827104008787951  -0.595273916358421  0.0238739030828427
0.827104008787951  0.595273916358421  0.827104008787951  0.0238739030828427
0.827104008787951  -0.595273916358421  0.827104008787951  0.0238739030828427
0.827104008787951  -0.827104008787951  0.595273916358421  0.0238739030828427
0.827104008787951  -0.827104008787951  -0.595273916358421  0.0238739030828427
0.827104008787951  0.595273916358421  -0.827104008787951  0.0238739030828427
0.827104008787951  -0.595273916358421  -0.827104008787951  0.0238739030828427
-0.827104008787951  0.827104008787951  0.595273916358421  0.0238739030828427
-0.827104008787951  0.827104008787951  -0.595273916358421  0.0238739030828427
-0.827104008787951  0.595273916358421  0.827104008787951  0.0238739030828427
-0.827104008787951  -0.595273916358421  0.827104008787951  0.0238739030828427
-0.827104008787951  -0.827104008787951  0.595273916358421  0.0238739030828427
-0.827104008787951  -0.827104008787951  -0.595273916358421  0.0238739030828427
-0.827104008787951  0.595273916358421  -0.827104008787951  0.0238739030828427
-0.827104008787951  -0.595273916358421  -0.827104008787951  0.0238739030828427
0.595273916358421  0.827104008787951  0.827104008787951  0.0238739030828427
0.595273916358421  0.827104008787951  -0.827104008787951  0.0238739030828427
0.595273916358421  -0.827104008787951  0.827104008787951  0.0238739030828427
0.595273916358421  -0.827104008787951  -0.827104008787951  0.0238739030828427
-0.595273916358421  0.827104008787951  0.827104008787951  0.0238739030828427
-0.595273916358421  0.827104008787951  -0.827104008787951  0.0238739030828427
-0.595273916358421  -0.827104008787951  0.827104008787951  0.0238739030828427
-0.595273916358421  -0.827104008787951  -0.827104008787951  0.0238739030828427
0.853891500333261  0.990340675813395  0.703401752648627  0.004492075519908
0.853891500333261  0.990340675813395  -0.703401752648627  0.004492075519908
0.853891500333261  0.703401752648627  0.990340675813395  0.004492075519908
0.853891500333261  -0.703401752648627  0.990340675813395  0.004492075519908
0.853891500333261  -0.990340675813395  0.703401752648627  0.004492075519908
0.853891500333261  -0.990340675813395  -0.703401752648627  0.004492075519908
0.853891500333261  0.703401752648627  -0.990340675813395  0.004492075519908
0.853891500333261  -0.703401752648627  -0.990340675813395  0.004492075519908
-0.853891500333261  0.990340675813395  0.703401752648627  0.004492075519908
-0.853891500333261  0.990340675813395  -0.703401752648627  0.004492075519908
-0.853891500333261  0.703401752648627  0.990340675813395  0.004492075519908
-0.853891500333261  -0.703401752648627  0.990340675813395  0.004492075519908
-0.853891500333261  -0.990340675813395  0.703401752648627  0.004492075519908
-0.853891500333261  -0.990340675813395  -0.703401752648627  0.004492075519908
-0.853891500333261  0.703401752648627  -0.990340675813395  0.004492075519908
-0.853891500333261  -0.703401752648627  -0.990340675813395  0.004492075519908
0.990340675813395  0.853891500333261  0.703401752648627  0.004492075519908
0.990340675813395  0.853891500333261  -0.703401752648627  0.004492075519908
0.990340675813395  0.703401752648627  0.853891500333261  0.004492075519908
0.990340675813395  -0.703401752648627  0.853891500333261  0.004492075519908
0.990340675813395  -0.853891500333261  0.703401752648627  0.004492075519908
0.990340675813395  -0.853891500333261  -0.703401752648627  0.004492075519908
0.990340675813395  0.703401752648627  -0.853891500333261  0.004492075519908
0.990340675813395  -0.703401752648627  -0.853891500333261  0.004492075519908
-0.990340675813395  0.853891500333261  0.703401752648627  0.004492075519908
-0.990340675813395  0.853891500333261  -0.703401752648627  0.004492075519908
-0.990340675813395  0.703401752648627  0.853891500333261  0.004492075519908
-0.990340675813395  -0.703401752648627  0.853891500333261  0.004492075519908
-0.990340675813395  -0.853891500333261  0.703401752648627  0.004492075519908
-0.990340675813395  -0.853891500333261  -0.703401752648627  0.004492075519908
-0.990340675813395  0.703401752648627  -0.853891500333261  0.004492075519908
-0.990340675813395  -0.703401752648627  -0.853891500333261  0.004492075519908
0.703401752648627  0.853891500333261  0.990340675813395  0.004492075519908
0.703401752648627  0.853891500333261  -0.990340675813395  0.004492075519908
0.703401752648627  0.990340675813395  0.853891500333261  0.004492075519908
0.703401752648627  -0.990340675813395  0.853891500333261  0.004492075519908
0.703401752648627  -0.853891500333261  0.990340675813395  0.004492075519908
0.703401752648627  -0.853891500333261  -0.990340675813395  0.004492075519908
0.703401752648627  0.990340675813395  -0.853891500333261  0.004492075519908
0.703401752648627  -0.990340675813395  -0.853891500333261  0.004492075519908
-0.703401752648627  0.853891500333261  0.990340675813395  0.004492075519908
-0.703401752648627  0.853891500333261  -0.990340675813395  0.004492075519908
-0.703401752648627  0.990340675813395  0.853891500333261  0.004492075519908
-0.703401752648627  -0.990340675813395  0.853891500333261  0.004492075519908
-0.703401752648627  -0.853891500333261  0.990340675813395  0.004492075519908
-0.703401752648627  -0.853891500333261  -0.990340675813395  0.004492075519908
-0.703401752648627  0.990340675813395  -0.853891500333261  0.004492075519908
-0.703401752648627  -0.990340675813395  -0.853891500333261  0.004492075519908
0.946965100974417  0.623429573213959  0.361644847592954  0.020564739563048
0.946965100974417  0.623429573213959  -0.361644847592954  0.020564739563048
0.946965100974417  0.361644847592954  0.623429573213959  0.020564739563048
0.946965100974417  -0.361644847592954  0.623429573213959  0.020564739563048
0.946965100974417  -0.623429573213959  0.361644847592954  0.020564739563048
0.946965100974417  -0.623429573213959  -0.361644847592954  0.020564739563048
0.946965100974417  0.361644847592954  -0.623429573213959  0.020564739563048
0.946965100974417  -0.361644847592954  -0.623429573213959  0.020564739563048
-0.946965100974417  0.623429573213959  0.361644847592954  0.020564739563048
-0.946965100974417  0.623429573213959  -0.361644847592954  0.020564739563048
-0.946965100974417  0.361644847592954  0.623429573213959  0.020564739563048
-0.946965100974417  -0.361644847592954  0.623429573213959  0.020564739563048
-0.946965100974417  -0.623429573213959  0.361644847592954  0.020564739563048
-0.946965100974417  -0.623429573213959  -0.361644847592954  0.020564739563048
-0.946965100974417  0.361644847592954  -0.623429573213959  0.020564739563048
-0.946965100974417  -0.361644847592954  -0.623429573213959  0.020564739563048
0.623429573213959  0.946965100974417  0.361644847592954  0.020564739563048
0.623429573213959  0.946965100974417  -0.361644847592954  0.020564739563048
0.623429573213959  0.361644847592954  0.946965100974417  0.020564739563048
0.623429573213959  -0.361644847592954  0.946965100974417  0.020564739563048
0.623429573213959  -0.946965100974417  0.361644847592954  0.020564739563048
0.623429573213959  -0.946965100974417  -0.361644847592954  0.020564739563048
0.623429573213959  0.361644847592954  -0.946965100974417  0.020564739563048
0.623429573213959  -0.361644847592954  -0.946965100974417  0.020564739563048
-0.623429573213959  0.946965100974417  0.361644847592954  0.020564739563048
-0.623429573213959  0.946965100974417  -0.361644847592954  0.020564739563048
-0.623429573213959  0.361644847592954  0.946965100974417  0.020564739563048
-0.623429573213959  -0.361644847592954  0.946965100974417  0.020564739563048
-0.623429573213959  -0.946965100974417  0.361644847592954  0.020564739563048
-0.623429573213959  -0.946965100974417  -0.361644847592954  0.020564739563048
-0.623429573213959  0.361644847592954  -0.946965100974417  0.020564739563048
-0.623429573213959  -0.361644847592954  -0.946965100974417  0.020564739563048
0.361644847592954  0.946965100974417  0.623429573213959  0.020564739563048
0.361644847592954  0.946965100974417  -0.623429573213959  0.020564739563048
0.361644847592954  0.623429573213959  0.946965100974417  0.020564739563048
0.361644847592954  -0.623429573213959  0.946965100974417  0.020564739563048
0.361644847592954  -0.946965100974417  0.623429573213959  0.020564739563048
0.361644847592954  -0.946965100974417  -0.623429573213959  0.020564739563048
0.361644847592954  0.623429573213959  -0.946965100974417  0.020564739563048
0.361644847592954  -0.623429573213959  -0.946965100974417  0.020564739563048
-0.361644847592954  0.946965100974417  0.623429573213959  0.020564739563048
-0.361644847592954  0.946965100974417  -0.623429573213959  0.020564739563048
-0.361644847592954  0.623429573213959  0.946965100974417  0.020564739563048
-0.361644847592954  -0.623429573213959  0.946965100974417  0.020564739563048
-0.361644847592954  -0.946965100974417  0.623429573213959  0.020564739563048
-0.361644847592954  -0.946965100974417  -0.623429573213959  0.020564739563048
-0.361644847592954  0.623429573213959  -0.946965100974417  0.020564739563048
-0.361644847592954  -0.623429573213959  -0.946965100974417  0.020564739563048
0.754378926224437  0.538990774213829  0.258107746839302  0.039008180977840
0.754378926224437  0.538990774213829  -0.258107746839302  0.039008180977840
0.754378926224437  0.258107746839302  0.538990774213829  0.039008180977840
0.754378926224437  -0.258107746839302  0.538990774213829  0.039008180977840
0.754378926224437  -0.538990774213829  0.258107746839302  0.039008180977840
0.754378926224437  -0.538990774213829  -0.258107746839302  0.039008180977840
0.754378926224437  0.258107746839302  -0.538990774213829  0.039008180977840
0.754378926224437  -0.258107746839302  -0.538990774213829  0.039008180977840
-0.754378926224437  0.538990774213829  0.258107746839302  0.039008180977840
-0.754378926224437  0.538990774213829  -0.258107746839302  0.039008180977840
-0.754378926224437  0.258107746839302  0.538990774213829  0.039008180977840
-0.754378926224437  -0.258107746839302  0.538990774213829  0.039008180977840
-0.754378926224437  -0.538990774213829  0.258107746839302  0.039008180977840
-0.754378926224437  -0.538990774213829  -0.258107746839302  0.039008180977840
-0.754378926224437  0.258107746839302  -0.538990774213829  0.039008180977840
-0.754378926224437  -0.258107746839302  -0.538990774213829  0.039008180977840
0.538990774213829  0.754378926224437  0.258107746839302  0.039008180977840
0.538990774213829  0.754378926224437  -0.258107746839302  0.039008180977840
0.538990774213829  0.258107746839302  0.754378926224437  0.039008180977840
0.538990774213829  -0.258107746839302  0.754378926224437  0.039008180977840
0.538990774213829  -0.754378926224437  0.258107746839302  0.039008180977840
0.538990774213829  -0.754378926224437  -0.258107746839302  0.039008180977840
0.538990774213829  0.258107746839302  -0.754378926224437  0.039008180977840
0.538990774213829  -0.258107746839302  -0.754378926224437  0.039008180977840
-0.538990774213829  0.754378926224437  0.258107746839302  0.039008180977840
-0.538990774213829  0.754378926224437  -0.258107746839302  0.039008180977840
-0.538990774213829  0.258107746839302  0.754378926224437  0.039008180977840
-0.538990774213829  -0.258107746839302  0.754378926224437  0.039008180977840
-0.538990774213829  -0.754378926224437  0.258107746839302  0.039008180977840
-0.538990774213829  -0.754378926224437  -0.258107746839302  0.039008180977840
-0.538990774213829  0.258107746839302  -0.754378926224437  0.039008180977840
-0.538990774213829  -0.258107746839302  -0.754378926224437  0.039008180977840
0.258107746839302  0.754378926224437  0.538990774213829  0.039008180977840
0.258107746839302  0.754378926224437  -0.538990774213829  0.039008180977840
0.258107746839302  0.538990774213829  0.754378926224437  0.039008180977840
0.258107746839302  -0.538990774213829  0.754378926224437  0.039008180977840
0.258107746839302  -0.754378926224437  0.538990774213829  0.039008180977840
0.258107746839302  -0.754378926224437  -0.538990774213829  0.039008180977840
0.258107746839302  0.538990774213829  -0.754378926224437  0.039008180977840
0.258107746839302  -0.538990774213829  -0.754378926224437  0.039008180977840
-0.258107746839302  0.754378926224437  0.538990774213829  0.039008180977840
-0.258107746839302  0.754378926224437  -0.538990774213829  0.039008180977840
-0.258107746839302  0.538990774213829  0.754378926224437  0.039008180977840
-0.258107746839302  -0.538990774213829  0.754378926224437  0.039008180977840
-0.258107746839302  -0.754378926224437  0.538990774213829  0.039008180977840
-0.258107746839302  -0.754378926224437  -0.538990774213829  0.039008180977840
-0.258107746839302  0.538990774213829  -0.754378926224437  0.039008180977840
-0.258107746839302  -0.538990774213829  -0.754378926224437  0.039008180977840
  Gauss  quadrature  points  and  weights  on  the  reference  cube  order  p=18  19
"""


economical_gauss_triangle_table = """\
1
-0.333333333333333  -0.333333333333333  2.000000000000000
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=1

3
-0.666666666666667  -0.666666666666667  0.666666666666667
-0.666666666666667  0.333333333333333  0.666666666666667
0.333333333333333  -0.666666666666667  0.666666666666667
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=2

4
-0.333333333333333  -0.333333333333333  -1.125000000000000
-0.600000000000000  -0.600000000000000  1.041666666666667
-0.600000000000000  0.200000000000000  1.041666666666667
0.200000000000000  -0.600000000000000  1.041666666666667
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=3

6
-0.108103018168070  -0.108103018168070  0.446763179356022
-0.108103018168070  -0.783793963663860  0.446763179356022
-0.783793963663860  -0.108103018168070  0.446763179356022
-0.816847572980458  -0.816847572980458  0.219903487310644
-0.816847572980458  0.633695145960918  0.219903487310644
0.633695145960918  -0.816847572980458  0.219903487310644
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=4

7
-0.333333333333333  -0.333333333333333  0.450000000000000
-0.059715871789770  -0.059715871789770  0.264788305577012
-0.059715871789770  -0.880568256420460  0.264788305577012
-0.880568256420460  -0.059715871789770  0.264788305577012
-0.797426985353088  -0.797426985353088  0.251878361089654
-0.797426985353088  0.594853970706174  0.251878361089654
0.594853970706174  -0.797426985353088  0.251878361089654
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=5

12
-0.501426509658180  -0.501426509658180  0.233572551452758
-0.501426509658180  0.002853019316358  0.233572551452758
0.002853019316358  -0.501426509658180  0.233572551452758
-0.873821971016996  -0.873821971016996  0.101689812740414
-0.873821971016996  0.747643942033992  0.101689812740414
0.747643942033992  -0.873821971016996  0.101689812740414
-0.379295097932432  0.273004998242798  0.165702151236748
0.273004998242798  -0.893709900310366  0.165702151236748
-0.893709900310366  -0.379295097932432  0.165702151236748
-0.379295097932432  -0.893709900310366  0.165702151236748
0.273004998242798  -0.379295097932432  0.165702151236748
-0.893709900310366  0.273004998242798  0.165702151236748
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=6

13
-0.333333333333333  -0.333333333333333  -0.299140088935364
-0.479308067841920  -0.479308067841920  0.351230514866416
-0.479308067841920  -0.041383864316160  0.351230514866416
-0.041383864316160  -0.479308067841920  0.351230514866416
-0.869739794195568  -0.869739794195568  0.106694471217676
-0.869739794195568  0.739479588391136  0.106694471217676
0.739479588391136  -0.869739794195568  0.106694471217676
-0.374269007990252  0.276888377139620  0.154227521780514
0.276888377139620  -0.902619369149368  0.154227521780514
-0.902619369149368  -0.374269007990252  0.154227521780514
-0.374269007990252  -0.902619369149368  0.154227521780514
0.276888377139620  -0.374269007990252  0.154227521780514
-0.902619369149368  0.276888377139620  0.154227521780514
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=7

16
-0.333333333333333  -0.333333333333333  0.288631215355574
-0.081414823414554  -0.081414823414554  0.190183268534570
-0.081414823414554  -0.837170353170892  0.190183268534570
-0.837170353170892  -0.081414823414554  0.190183268534570
-0.658861384496480  -0.658861384496480  0.206434741069436
-0.658861384496480  0.317722768992960  0.206434741069436
0.317722768992960  -0.658861384496480  0.206434741069436
-0.898905543365938  -0.898905543365938  0.064916995246396
-0.898905543365938  0.797811086731876  0.064916995246395
0.797811086731876  -0.898905543365938  0.064916995246396
-0.473774340730724  0.456984785910808  0.054460628348870
0.456984785910808  -0.983210445180084  0.054460628348870
-0.983210445180084  -0.473774340730724  0.054460628348870
-0.473774340730724  -0.983210445180084  0.054460628348870
0.456984785910808  -0.473774340730724  0.054460628348870
-0.983210445180084  0.456984785910808  0.054460628348870
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=8

19
-0.333333333333333  -0.333333333333333  0.194271592565598
-0.020634961602524  -0.020634961602524  0.062669400454278
-0.020634961602524  -0.958730076794950  0.062669400454278
-0.958730076794950  -0.020634961602524  0.062669400454278
-0.125820817014126  -0.125820817014126  0.155655082009548
-0.125820817014126  -0.748358365971746  0.155655082009548
-0.748358365971746  -0.125820817014126  0.155655082009548
-0.623592928761934  -0.623592928761934  0.159295477854420
-0.623592928761934  0.247185857523870  0.159295477854420
0.247185857523870  -0.623592928761934  0.159295477854420
-0.910540973211094  -0.910540973211094  0.051155351317396
-0.910540973211094  0.821081946422190  0.051155351317396
0.821081946422190  -0.910540973211094  0.051155351317396
-0.556074021678468  0.482397197568996  0.086567078754578
0.482397197568996  -0.926323175890528  0.086567078754578
-0.926323175890528  -0.556074021678468  0.086567078754578
-0.556074021678468  -0.926323175890528  0.086567078754578
0.482397197568996  -0.556074021678468  0.086567078754578
-0.926323175890528  0.482397197568996  0.086567078754578
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=9

25
-0.333333333333333  -0.333333333333333  0.181635980765508
-0.028844733232686  -0.028844733232686  0.073451915512934
-0.028844733232686  -0.942310533534630  0.073451915512934
-0.942310533534630  -0.028844733232686  0.073451915512934
-0.781036849029926  -0.781036849029926  0.090642118871056
-0.781036849029926  0.562073698059852  0.090642118871056
0.562073698059852  -0.781036849029926  0.090642118871056
-0.384120322471758  0.100705883641998  0.145515833690840
0.100705883641998  -0.716585561170240  0.145515833690840
-0.716585561170240  -0.384120322471758  0.145515833690840
-0.384120322471758  -0.716585561170240  0.145515833690840
0.100705883641998  -0.384120322471758  0.145515833690840
-0.716585561170240  0.100705883641998  0.145515833690840
-0.506654878720194  0.456647809194822  0.056654485062114
0.456647809194822  -0.949992930474628  0.056654485062114
-0.949992930474628  -0.506654878720194  0.056654485062114
-0.506654878720194  -0.949992930474628  0.056654485062114
0.456647809194822  -0.506654878720194  0.056654485062114
-0.949992930474628  0.456647809194822  0.056654485062114
-0.866393497975600  0.847311867175000  0.018843333927466
0.847311867175000  -0.980918369199402  0.018843333927466
-0.980918369199402  -0.866393497975600  0.018843333927466
-0.866393497975600  -0.980918369199402  0.018843333927466
0.847311867175000  -0.866393497975600  0.018843333927466
-0.980918369199402  0.847311867175000  0.018843333927466
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=10

27
0.069222096541516  0.069222096541516  0.001854012657922
0.069222096541516  -1.138444193083034  0.001854012657922
-1.138444193083034  0.069222096541516  0.001854012657922
-0.202061394068290  -0.202061394068290  0.154299069829626
-0.202061394068290  -0.595877211863420  0.154299069829626
-0.595877211863420  -0.202061394068290  0.154299069829626
-0.593380199137436  -0.593380199137436  0.118645954761548
-0.593380199137436  0.186760398274870  0.118645954761548
0.186760398274870  -0.593380199137436  0.118645954761548
-0.761298175434838  -0.761298175434838  0.072369081006836
-0.761298175434838  0.522596350869674  0.072369081006836
0.522596350869674  -0.761298175434838  0.072369081006836
-0.935270103777448  -0.935270103777448  0.027319462005356
-0.935270103777448  0.870540207554896  0.027319462005356
0.870540207554896  -0.935270103777448  0.027319462005356
-0.286758703477414  0.186402426856426  0.104674223924408
0.186402426856426  -0.899643723379010  0.104674223924408
-0.899643723379010  -0.286758703477414  0.104674223924408
-0.286758703477414  -0.899643723379010  0.104674223924408
0.186402426856426  -0.286758703477414  0.104674223924408
-0.899643723379010  0.186402426856426  0.104674223924408
-0.657022039391916  0.614978006319584  0.041415319278282
0.614978006319584  -0.957955966927668  0.041415319278282
-0.957955966927668  -0.657022039391916  0.041415319278282
-0.657022039391916  -0.957955966927668  0.041415319278282
0.614978006319584  -0.657022039391916  0.041415319278282
-0.957955966927668  0.614978006319584  0.041415319278282
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=11

33
-0.023565220452390  -0.023565220452390  0.051462132880910
-0.023565220452390  -0.952869559095220  0.051462132880910
-0.952869559095220  -0.023565220452390  0.051462132880910
-0.120551215411080  -0.120551215411080  0.087385089076076
-0.120551215411080  -0.758897569177842  0.087385089076076
-0.758897569177842  -0.120551215411080  0.087385089076076
-0.457579229975768  -0.457579229975768  0.125716448435770
-0.457579229975768  -0.084841540048464  0.125716448435770
-0.084841540048464  -0.457579229975768  0.125716448435770
-0.744847708916828  -0.744847708916828  0.069592225861418
-0.744847708916828  0.489695417833656  0.069592225861418
0.489695417833656  -0.744847708916828  0.069592225861418
-0.957365299093580  -0.957365299093580  0.012332522103118
-0.957365299093580  0.914730598187158  0.012332522103118
0.914730598187158  -0.957365299093580  0.012332522103118
-0.448573460628972  0.217886471559576  0.080743115532762
0.217886471559576  -0.769313010930604  0.080743115532762
-0.769313010930604  -0.448573460628972  0.080743115532762
-0.448573460628972  -0.769313010930604  0.080743115532762
0.217886471559576  -0.448573460628972  0.080743115532762
-0.769313010930604  0.217886471559576  0.080743115532762
-0.437348838020120  0.391672173575606  0.044713546404606
0.391672173575606  -0.954323335555486  0.044713546404606
-0.954323335555486  -0.437348838020120  0.044713546404606
-0.437348838020120  -0.954323335555486  0.044713546404606
0.391672173575606  -0.437348838020120  0.044713546404606
-0.954323335555486  0.391672173575606  0.044713546404606
-0.767496168184806  0.716028067088146  0.034632462217318
0.716028067088146  -0.948531898903340  0.034632462217318
-0.948531898903340  -0.767496168184806  0.034632462217318
-0.767496168184806  -0.948531898903340  0.034632462217318
0.716028067088146  -0.767496168184806  0.034632462217318
-0.948531898903340  0.716028067088146  0.034632462217318
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=12

37
-0.333333333333333  -0.333333333333333  0.105041846801604
-0.009903630120590  -0.009903630120590  0.022560290418660
-0.009903630120590  -0.980192739758818  0.022560290418660
-0.980192739758818  -0.009903630120590  0.022560290418660
-0.062566729780852  -0.062566729780852  0.062847036724908
-0.062566729780852  -0.874866540438296  0.062847036724908
-0.874866540438296  -0.062566729780852  0.062847036724908
-0.170957326397446  -0.170957326397446  0.094145005008388
-0.170957326397446  -0.658085347205106  0.094145005008388
-0.658085347205106  -0.170957326397446  0.094145005008388
-0.541200855914338  -0.541200855914338  0.094727173072710
-0.541200855914338  0.082401711828674  0.094727173072710
0.082401711828674  -0.541200855914338  0.094727173072710
-0.771151009607340  -0.771151009607340  0.062335058091588
-0.771151009607340  0.542302019214680  0.062335058091588
0.542302019214680  -0.771151009607340  0.062335058091588
-0.950377217273082  -0.950377217273082  0.015951542930148
-0.950377217273082  0.900754434546164  0.015951542930148
0.900754434546164  -0.950377217273082  0.015951542930148
-0.462410005882478  0.272702349123320  0.073696805457464
0.272702349123320  -0.810292343240842  0.073696805457464
-0.810292343240842  -0.462410005882478  0.073696805457464
-0.462410005882478  -0.810292343240842  0.073696805457464
0.272702349123320  -0.462410005882478  0.073696805457464
-0.810292343240842  0.272702349123320  0.073696805457464
-0.416539866531424  0.380338319973810  0.034802926607644
0.380338319973810  -0.963798453442386  0.034802926607644
-0.963798453442386  -0.416539866531424  0.034802926607644
-0.416539866531424  -0.963798453442386  0.034802926607644
0.380338319973810  -0.416539866531424  0.034802926607644
-0.963798453442386  0.380338319973810  0.034802926607644
-0.747285229016662  0.702819075668482  0.031043573678090
0.702819075668482  -0.955533846651820  0.031043573678090
-0.955533846651820  -0.747285229016662  0.031043573678090
-0.747285229016662  -0.955533846651820  0.031043573678090
0.702819075668482  -0.747285229016662  0.031043573678090
-0.955533846651820  0.702819075668482  0.031043573678090
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=13

42
-0.022072179275642  -0.022072179275642  0.043767162738858
-0.022072179275642  -0.955855641448714  0.043767162738858
-0.955855641448714  -0.022072179275642  0.043767162738858
-0.164710561319092  -0.164710561319092  0.065576707088250
-0.164710561319092  -0.670578877361816  0.065576707088250
-0.670578877361816  -0.164710561319092  0.065576707088250
-0.453044943382322  -0.453044943382322  0.103548209014584
-0.453044943382322  -0.093910113235354  0.103548209014584
-0.093910113235354  -0.453044943382322  0.103548209014584
-0.645588935174914  -0.645588935174914  0.084325177473986
-0.645588935174914  0.291177870349826  0.084325177473986
0.291177870349826  -0.645588935174914  0.084325177473986
-0.876400233818254  -0.876400233818254  0.028867399339554
-0.876400233818254  0.752800467636510  0.028867399339554
0.752800467636510  -0.876400233818254  0.028867399339554
-0.961218077502598  -0.961218077502598  0.009846807204800
-0.961218077502598  0.922436155005196  0.009846807204800
0.9224361550051960  -0.961218077502598  0.009846807204800
-0.655466624357288  0.541217109549992  0.049331506425128
0.541217109549992  -0.885750485192704  0.049331506425128
-0.885750485192704  -0.655466624357288  0.049331506425128
-0.655466624357288  -0.885750485192704  0.049331506425128
0.541217109549992  -0.655466624357288  0.049331506425128
-0.885750485192704  0.541217109549992  0.049331506425128
-0.326277080407310  0.140444581693366  0.077143021574122
0.140444581693366  -0.814167501286056  0.077143021574122
-0.814167501286056  -0.326277080407310  0.077143021574122
-0.326277080407310  -0.814167501286056  0.077143021574122
0.140444581693366  -0.326277080407310  0.077143021574122
-0.814167501286056  0.140444581693366  0.077143021574122
-0.403254235727484  0.373960335616176  0.028872616227068
0.373960335616176  -0.970706099888692  0.028872616227068
-0.970706099888692  -0.403254235727484  0.028872616227068
-0.403254235727484  -0.970706099888692  0.028872616227068
0.373960335616176  -0.403254235727484  0.028872616227068
-0.970706099888692  0.373960335616176  0.028872616227068
-0.762051004606086  0.759514342740342  0.010020457677002
0.759514342740342  -0.997463338134256  0.010020457677002
-0.997463338134256  -0.762051004606086  0.010020457677002
-0.762051004606086  -0.997463338134256  0.010020457677002
0.759514342740342  -0.762051004606086  0.010020457677002
-0.997463338134256  0.759514342740342  0.010020457677002
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=14

48
0.013945833716486  0.013945833716486  0.003833751285698
0.013945833716486  -1.027891667432972  0.003833751285698
-1.027891667432972  0.013945833716486  0.003833751285698
-0.137187291433954  -0.137187291433954  0.088498054542290
-0.137187291433954  -0.725625417132090  0.088498054542290
-0.725625417132090  -0.137187291433954  0.088498054542290
-0.444612710305712  -0.444612710305712  0.102373097437704
-0.444612710305712  -0.110774579388578  0.102373097437704
-0.110774579388578  -0.444612710305712  0.102373097437704
-0.747070217917492  -0.747070217917492  0.047375471741376
-0.747070217917492  0.494140435834984  0.047375471741376
0.494140435834984  -0.747070217917492  0.047375471741376
-0.858383228050628  -0.858383228050628  0.026579551380042
-0.858383228050628  0.716766456101256  0.026579551380042
0.716766456101256  -0.858383228050628  0.026579551380042
-0.962069659517854  -0.962069659517854  0.009497833216384
-0.962069659517854  0.924139319035706  0.009497833216384
0.924139319035706  -0.962069659517854  0.009497833216384
-0.477377257719826  0.209908933786582  0.077100145199186
0.209908933786582  -0.732531676066758  0.077100145199186
-0.732531676066758  -0.477377257719826  0.077100145199186
-0.477377257719826  -0.732531676066758  0.077100145199186
0.209908933786582  -0.477377257719826  0.077100145199186
-0.732531676066758  0.209908933786582  0.077100145199186
-0.223906465819462  0.151173111025628  0.054431628641248
0.151173111025628  -0.927266645206166  0.054431628641248
-0.927266645206166  -0.223906465819462  0.054431628641248
-0.223906465819462  -0.927266645206166  0.054431628641248
0.151173111025628  -0.223906465819462  0.054431628641248
-0.927266645206166  0.151173111025628  0.054431628641248
-0.428575559900168  0.448925326153310  0.004364154733594
0.448925326153310  -1.020349766253142  0.004364154733594
-1.020349766253142  -0.428575559900168  0.004364154733594
-0.428575559900168  -1.020349766253142  0.004364154733594
0.448925326153310  -0.428575559900168  0.004364154733594
-1.020349766253142  0.448925326153310  0.004364154733594
-0.568800671855432  0.495112932103676  0.043010639695462
0.495112932103676  -0.926312260248244  0.043010639695462
-0.926312260248244  -0.568800671855432  0.043010639695462
-0.568800671855432  -0.926312260248244  0.043010639695462
0.495112932103676  -0.568800671855432  0.043010639695462
-0.926312260248244  0.495112932103676  0.043010639695462
-0.792848766847228  0.767929148184832  0.015347885262098
0.767929148184832  -0.975080381337602  0.015347885262098
-0.975080381337602  -0.792848766847228  0.015347885262098
-0.792848766847228  -0.975080381337602  0.015347885262098
0.767929148184832  -0.792848766847228  0.015347885262098
-0.975080381337602  0.767929148184832  0.015347885262098
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=15

52
-0.333333333333333  -0.333333333333333  0.093751394855284
-0.005238916103124  -0.005238916103124  0.012811757157170
-0.005238916103124  -0.989522167793754  0.012811757157170
-0.989522167793754  -0.005238916103124  0.012811757157170
-0.173061122901296  -0.173061122901296  0.083420593478774
-0.173061122901296  -0.653877754197410  0.083420593478774
-0.653877754197410  -0.173061122901296  0.083420593478774
-0.059082801866018  -0.059082801866018  0.053782968500128
-0.059082801866018  -0.881834396267966  0.053782968500128
-0.881834396267966  -0.059082801866018  0.053782968500128
-0.518892500060958  -0.518892500060958  0.084265045523300
-0.518892500060958  0.037785000121916  0.084265045523300
0.037785000121916  -0.518892500060958  0.084265045523300
-0.704068411554854  -0.704068411554854  0.060000533685546
-0.704068411554854  0.408136823109708  0.060000533685546
0.408136823109708  -0.704068411554854  0.060000533685546
-0.849069624685052  -0.849069624685052  0.028400197850048
-0.849069624685052  0.698139249370104  0.028400197850048
0.698139249370104  -0.849069624685052  0.028400197850048
-0.966807194753950  -0.966807194753950  0.007164924702546
-0.966807194753950  0.933614389507900  0.007164924702546
0.933614389507900  -0.966807194753950  0.007164924702546
-0.406888806840226  0.199737422349722  0.065546294921254
0.199737422349722  -0.792848615509496  0.065546294921254
-0.792848615509496  -0.406888806840226  0.065546294921254
-0.406888806840226  -0.792848615509496  0.065546294921254
0.199737422349722  -0.406888806840226  0.065546294921254
-0.792848615509496  0.199737422349722  0.065546294921254
-0.324553873193842  0.284387049883010  0.030596612496882
0.284387049883010  -0.959833176689168  0.030596612496882
-0.959833176689168  -0.324553873193842  0.030596612496882
-0.324553873193842  -0.959833176689168  0.030596612496882
0.284387049883010  -0.324553873193842  0.030596612496882
-0.959833176689168  0.284387049883010  0.030596612496882
-0.590503436714376  0.599185441942654  0.004772488385678
0.599185441942654  -1.008682005228278  0.004772488385678
-1.008682005228278  -0.590503436714376  0.004772488385678
-0.590503436714376  -1.008682005228278  0.004772488385678
0.599185441942654  -0.590503436714376  0.004772488385678
-1.008682005228278  0.599185441942654  0.004772488385678
-0.621283015738754  0.537399442802736  0.038169585511798
0.537399442802736  -0.916116427063980  0.038169585511798
-0.916116427063980  -0.621283015738754  0.038169585511798
-0.621283015738754  -0.916116427063980  0.038169585511798
0.537399442802736  -0.621283015738754  0.038169585511798
-0.916116427063980  0.537399442802736  0.038169585511798
-0.829432768634686  0.800798128173322  0.013700109093084
0.800798128173322  -0.971365359538638  0.013700109093084
-0.971365359538638  -0.829432768634686  0.013700109093084
-0.829432768634686  -0.971365359538638  0.013700109093084
0.800798128173322  -0.829432768634686  0.013700109093084
-0.971365359538638  0.800798128173322  0.013700109093084
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=16

61
-0.333333333333333  -0.333333333333333  0.066874398581606
-0.005658918886452  -0.005658918886452  0.010186830881014
-0.005658918886452  -0.988682162227096  0.010186830881014
-0.988682162227096  -0.005658918886452  0.010186830881014
-0.035647354750750  -0.035647354750750  0.029341729055276
-0.035647354750750  -0.928705290498498  0.029341729055276
-0.928705290498498  -0.035647354750750  0.029341729055276
-0.099520061958436  -0.099520061958436  0.048701756707344
-0.099520061958436  -0.800959876083126  0.048701756707344
-0.800959876083126  -0.099520061958436  0.048701756707344
-0.199467521245206  -0.199467521245206  0.062215101737938
-0.199467521245206  -0.601064957509588  0.062215101737938
-0.601064957509588  -0.199467521245206  0.062215101737938
-0.495717464058094  -0.495717464058094  0.062514222437240
-0.495717464058094  -0.008565071883810  0.062514222437240
-0.008565071883810  -0.495717464058094  0.062514222437240
-0.675905990683078  -0.675905990683078  0.049631308679330
-0.675905990683078  0.351811981366154  0.049631308679330
0.351811981366154  -0.675905990683078  0.049631308679330
-0.848248235478508  -0.848248235478508  0.028112146141114
-0.848248235478508  0.696496470957016  0.028112146141114
0.696496470957016  -0.848248235478508  0.028112146141114
-0.968690546064356  -0.968690546064356  0.006389352347558
-0.968690546064356  0.937381092128712  0.006389352347558
0.937381092128712  -0.968690546064356  0.006389352347558
-0.331360265272684  0.310986407618846  0.016239310637986
0.310986407618846  -0.979626142346162  0.016239310637986
-0.979626142346162  -0.331360265272684  0.016239310637986
-0.331360265272684  -0.979626142346162  0.016239310637986
0.310986407618846  -0.331360265272684  0.016239310637986
-0.979626142346162  0.310986407618846  0.016239310637986
-0.415556924406112  0.144675181064040  0.053611484566326
0.144675181064040  -0.729118256657928  0.053611484566326
-0.729118256657928  -0.415556924406112  0.053611484566326
-0.415556924406112  -0.729118256657928  0.053611484566326
0.144675181064040  -0.415556924406112  0.053611484566326
-0.729118256657928  0.144675181064040  0.053611484566326
-0.360850229153620  0.252002380572456  0.036919986421644
0.252002380572456  -0.891152151418834  0.036919986421644
-0.891152151418834  -0.360850229153620  0.036919986421644
-0.360850229153620  -0.891152151418834  0.036919986421644
0.252002380572456  -0.360850229153620  0.036919986421644
-0.891152151418834  0.252002380572456  0.036919986421644
-0.618591551615416  0.592854429948142  0.016953737068656
0.592854429948142  -0.974262878332726  0.016953737068656
-0.974262878332726  -0.618591551615416  0.016953737068656
-0.618591551615416  -0.974262878332726  0.016953737068656
0.592854429948142  -0.618591551615416  0.016953737068656
-0.974262878332726  0.592854429948142  0.016953737068656
-0.639033576702508  0.504702011875458  0.036585593540050
0.504702011875458  -0.865668435172952  0.036585593540050
-0.865668435172952  -0.639033576702508  0.036585593540050
-0.639033576702508  -0.865668435172952  0.036585593540050
0.504702011875458  -0.639033576702508  0.036585593540050
-0.865668435172952  0.504702011875458  0.036585593540050
-0.838577372640872  0.809251008191216  0.013331264008330
0.809251008191216  -0.970673635550344  0.013331264008330
-0.970673635550344  -0.838577372640872  0.013331264008330
-0.838577372640872  -0.970673635550344  0.013331264008330
0.809251008191216  -0.838577372640872  0.013331264008330
-0.970673635550344  0.809251008191216  0.013331264008330
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=17

70
-0.333333333333333  -0.333333333333333  0.061619879875294
-0.013310382738158  -0.013310382738158  0.018144873358808
-0.013310382738158  -0.973379234523686  0.018144873358808
-0.973379234523686  -0.013310382738158  0.018144873358808
-0.061578811516086  -0.061578811516086  0.037522633879188
-0.061578811516086  -0.876842376967828  0.037522633879188
-0.876842376967828  -0.061578811516086  0.037522633879188
-0.127437208225988  -0.127437208225988  0.038882195970954
-0.127437208225988  -0.745125583548022  0.038882195970954
-0.745125583548022  -0.127437208225988  0.038882195970954
-0.210307658653168  -0.210307658653168  0.055507897221620
-0.210307658653168  -0.579384682693664  0.055507897221620
-0.579384682693664  -0.210307658653168  0.055507897221620
-0.500410862393686  -0.500410862393686  0.064512450702914
-0.500410862393686  0.000821724787372  0.064512450702914
0.000821724787372  -0.500410862393686  0.064512450702914
-0.677135612512314  -0.677135612512314  0.050148065233844
-0.677135612512314  0.354271225024630  0.050148065233844
0.354271225024630  -0.677135612512314  0.050148065233844
-0.846803545029258  -0.846803545029258  0.030543855943664
-0.846803545029258  0.693607090058514  0.030543855943664
0.693607090058514  -0.846803545029258  0.030543855943664
-0.951495121293100  -0.951495121293100  0.013587844045926
-0.951495121293100  0.902990242586200  0.013587844045926
0.902990242586200  -0.951495121293100  0.013587844045926
-0.913707265566070  -0.913707265566070  -0.004446197459840
-0.913707265566070  0.827414531132142  -0.004446197459840
0.827414531132142  -0.913707265566070  -0.004446197459840
-0.282177010118112  0.265315937713272  0.012663828152812
0.265315937713272  -0.983138927595160  0.012663828152812
-0.983138927595160  -0.282177010118112  0.012663828152812
-0.282177010118112  -0.983138927595160  0.012663828152812
0.265315937713272  -0.282177010118112  0.012663828152812
-0.983138927595160  0.265315937713272  0.012663828152812
-0.411195046496086  0.148821943021710  0.054515076098276
0.148821943021710  -0.737626896525624  0.054515076098276
-0.737626896525624  -0.411195046496086  0.054515076098276
-0.411195046496086  -0.737626896525624  0.054515076098276
0.148821943021710  -0.411195046496086  0.054515076098276
-0.737626896525624  0.148821943021710  0.054515076098276
-0.349964396716372  0.249558093585024  0.035353571298930
0.249558093585024  -0.899593696868650  0.035353571298930
-0.899593696868650  -0.349964396716372  0.035353571298930
-0.349964396716372  -0.899593696868650  0.035353571298930
0.249558093585024  -0.349964396716372  0.035353571298930
-0.899593696868650  0.249558093585024  0.035353571298930
-0.630524880667908  0.497866353046074  0.036758969276140
0.497866353046074  -0.867341472378168  0.036758969276140
-0.867341472378168  -0.630524880667908  0.036758969276140
-0.630524880667908  -0.867341472378168  0.036758969276140
0.497866353046074  -0.630524880667908  0.036758969276140
-0.867341472378168  0.497866353046074  0.036758969276140
-0.562406399973358  0.538414010840886  0.016209465616384
0.538414010840886  -0.976007610867528  0.016209465616384
-0.976007610867528  -0.562406399973358  0.016209465616384
-0.562406399973358  -0.976007610867528  0.016209465616384
0.538414010840886  -0.562406399973358  0.016209465616384
-0.976007610867528  0.538414010840886  0.016209465616384
-0.797640805727184  0.767924604546934  0.015268258141450
0.767924604546934  -0.970283798819750  0.015268258141450
-0.970283798819750  -0.797640805727184  0.015268258141450
-0.797640805727184  -0.970283798819750  0.015268258141450
0.767924604546934  -0.797640805727184  0.015268258141450
-0.970283798819750  0.767924604546934  0.015268258141450
-0.958250489434828  1.028694520010726  0.000092375321588
1.028694520010726  -1.070444030575898  0.000092375321588
-1.070444030575898  -0.9582504894348281  0.000092375321588
-0.958250489434828  -1.070444030575898  0.000092375321588
1.028694520010726  -0.958250489434828  0.000092375321588
-1.070444030575898    1.028694520010726  0.000092375321588
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=18

73
-0.333333333333333  -0.333333333333333  0.065812662777838
-0.020780025853988  -0.020780025853988  0.020661463782544
-0.020780025853988  -0.958439948292026  0.020661463782544
-0.958439948292026  -0.020780025853988  0.020661463782544
-0.090926214604214  -0.090926214604214  0.044774494526032
-0.090926214604214  -0.818147570791570  0.044774494526032
-0.818147570791570  -0.090926214604214  0.044774494526032
-0.197166638701138  -0.197166638701138  0.060532251738936
-0.197166638701138  -0.605666722597724  0.060532251738936
-0.605666722597724  -0.197166638701138  0.060532251738936
-0.488896691193804  -0.488896691193804  0.060981935604396
-0.488896691193804  -0.022206617612390  0.060981935604396
-0.022206617612390  -0.488896691193804  0.060981935604396
-0.645844115695740  -0.645844115695740  0.048318425483282
-0.645844115695740  0.291688231391482  0.048318425483282
0.291688231391482  -0.645844115695740  0.048318425483282
-0.779877893544096  -0.779877893544096  0.032101607173602
-0.779877893544096  0.559755787088192  0.032101607173602
0.559755787088192  -0.779877893544096  0.032101607173602
-0.888942751496320  -0.888942751496320  0.016169160523568
-0.888942751496320  0.777885502992642  0.016169160523568
0.777885502992642  -0.888942751496320  0.016169160523568
-0.974756272445542  -0.974756272445542  0.004158724054970
-0.974756272445542  0.949512544891086  0.004158724054970
0.949512544891086  -0.974756272445542  0.004158724054970
-0.208490425286114  0.201267589589290  0.007769753809962
0.201267589589290  -0.992777164303176  0.007769753809962
-0.992777164303176  -0.208490425286114  0.007769753809962
-0.208490425286114  -0.992777164303176  0.007769753809962
0.201267589589290  -0.208490425286114  0.007769753809962
-0.992777164303176  0.201267589589286  0.007769753809962
-0.384140032239128  0.115206523177568  0.051148321224044
0.115206523177568  -0.731066490938440  0.051148321224044
-0.731066490938440  -0.384140032239128  0.051148321224044
-0.384140032239128  -0.731066490938440  0.051148321224044
0.115206523177568  -0.384140032239128  0.051148321224044
-0.731066490938440  0.115206523177568  0.051148321224044
-0.470866103186960  0.441974051634730  0.017761807146676
0.441974051634730  -0.971107948447770  0.017761807146676
-0.971107948447770  -0.470866103186960  0.017761807146676
-0.470866103186960  -0.971107948447770  0.017761807146676
0.441974051634730  -0.470866103186960  0.017761807146676
-0.971107948447770  0.441974051634730  0.017761807146676
-0.282921295588098  0.189054137911742  0.032249093523462
0.189054137911742  -0.906132842323644  0.032249093523462
-0.906132842323644  -0.282921295588098  0.032249093523462
-0.282921295588098  -0.906132842323644  0.032249093523462
0.189054137911742  -0.282921295588098  0.032249093523462
-0.906132842323644  0.189054137911742  0.032249093523462
-0.684385188062810  0.678662947361678  0.004983883634982
0.678662947361678  -0.994277759298866  0.004983883634982
-0.994277759298866  -0.684385188062810  0.004983883634982
-0.684385188062810  -0.994277759298866  0.004983883634982
0.678662947361678  -0.684385188062810  0.004983883634982
-0.994277759298866  0.678662947361678  0.004983883634982
-0.849898806048178  0.402175957852346  0.036485680237902
0.402175957852346  -0.552277151804168  0.036485680237902
-0.552277151804168  -0.849898806048178  0.036485680237902
-0.849898806048178  -0.552277151804168  0.036485680237902
0.402175957852346  -0.849898806048178  0.036485680237902
-0.552277151804168  0.402175957852346  0.036485680237902
-0.715156797773234  0.645862648139714  0.020517127472398
0.645862648139714  -0.930705850366480  0.020517127472398
-0.930705850366480  -0.715156797773234  0.020517127472398
-0.715156797773234  -0.930705850366480  0.020517127472398
0.645862648139714  -0.715156797773234  0.020517127472398
-0.930705850366480  0.645862648139714  0.020517127472398
-0.869010743834124  0.848688505241568  0.007599857710604
0.848688505241568  -0.979677761407444  0.007599857710604
-0.979677761407444  -0.869010743834124  0.007599857710604
-0.869010743834124  -0.979677761407444  0.007599857710604
0.848688505241568  -0.869010743834124  0.007599857710604
-0.979677761407444  0.848688505241568  0.007599857710604
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=19

79
-0.333333333333333  -0.333333333333333  0.066114111083248
0.001900928704400  0.001900928704400  0.001734038371326
0.001900928704400  -1.003801857408800  0.001734038371326
-1.003801857408800  0.001900928704400  0.001734038371326
-0.023574084130543  -0.023574084130543  0.023320105432896
-0.023574084130543  -0.952851831738914  0.023320105432896
-0.952851831738914  -0.023574084130543  0.023320105432896
-0.089726626099435  -0.089726626099435  0.045753872712842
-0.089726626099435  -0.820546727801130  0.045753872712842
-0.820546727801130  -0.089726626099435  0.045753872712842
-0.196007481363421  -0.196007481363421  0.060897965347876
-0.196007481363421  -0.607985037273158  0.060897965347876
-0.607985037273158  -0.196007481363421  0.060897965347876
-0.488214180481157  -0.488214180481157  0.061249783450710
-0.488214180481157  -0.023571639037686  0.061249783450710
-0.023571639037686  -0.488214180481157  0.061249783450710
-0.647023488009788  -0.647023488009788  0.048736115353600
-0.647023488009788  0.294046976019576  0.048736115353600
0.294046976019576  -0.647023488009788  0.048736115353600
-0.791658289326483  -0.791658289326483  0.031994864064048
-0.791658289326483  0.583316578652966  0.031994864064048
0.583316578652966  -0.791658289326483  0.031994864064048
-0.893862072318140  -0.893862072318140  0.015396603631204
-0.893862072318140  0.787724144636280  0.015396603631204
0.787724144636280  -0.893862072318140  0.015396603631204
-0.916762569607942  -0.916762569607942  -0.001264120994976
-0.916762569607942  0.833525139215884  -0.001264120994976
0.833525139215884  -0.916762569607942  -0.001264120994976
-0.976836157186356  -0.976836157186356  0.003502268602386
-0.976836157186356  0.953672314372712  0.003502268602386
0.953672314372712  -0.976836157186356  0.003502268602386
-0.310288459541998  0.212805292212320  0.032931678379152
0.212805292212320  -0.902516832670322  0.032931678379152
-0.902516832670322  -0.310288459541998  0.032931678379152
-0.310288459541998  -0.902516832670322  0.032931678379152
0.212805292212320  -0.310288459541998  0.032931678379152
-0.902516832670322  0.212805292212320  0.032931678379152
-0.244313460810292  0.231685228913082  0.009678067080970
0.231685228913082  -0.987371768102790  0.009678067080970
-0.987371768102790  -0.244313460810292  0.009678067080970
-0.244313460810292  -0.987371768102790  0.009678067080970
0.231685228913082  -0.244313460810292  0.009678067080970
-0.987371768102790  0.231685228913082  0.009678067080970
-0.386729041875286  0.118096000780590  0.051609813069300
0.118096000780590  -0.731366958905304  0.051609813069300
-0.731366958905304  -0.386729041875286  0.051609813069300
-0.386729041875286  -0.731366958905304  0.051609813069300
0.118096000780590  -0.386729041875286  0.051609813069300
-0.731366958905304  0.118096000780590  0.051609813069300
-0.501161274450516  0.473213486525732  0.016942182108882
0.473213486525732  -0.972052212075216  0.016942182108882
-0.972052212075216  -0.501161274450516  0.016942182108882
-0.501161274450516  -0.972052212075216  0.016942182108882
0.473213486525732  -0.501161274450516  0.016942182108882
-0.972052212075216  0.473213486525732  0.016942182108882
-0.574448550394396  0.423350284574868  0.036709828212560
0.423350284574868  -0.848901734180472  0.036709828212560
-0.848901734180472  -0.574448550394396  0.036709828212560
-0.574448550394396  -0.848901734180472  0.036709828212560
0.423350284574868  -0.574448550394396  0.036709828212560
-0.848901734180472  0.423350284574868  0.036709828212560
-0.706069127893522  0.722805434309974  0.001408809355816
0.722805434309974  -1.016736306416454  0.001408809355816
-1.016736306416454  -0.706069127893522  0.001408809355816
-0.706069127893522  -1.016736306416454  0.001408809355816
0.722805434309974  -0.706069127893522  0.001408809355816
-1.016736306416454  0.722805434309974  0.001408809355816
-0.724546042342154  0.671173915824726  0.020225369854924
0.671173915824726  -0.946627873482572  0.020225369854924
-0.946627873482572  -0.724546042342154  0.020225369854924
-0.724546042342154  -0.946627873482572  0.020225369854924
0.671173915824726  -0.724546042342154  0.020225369854924
-0.946627873482572  0.671173915824726  0.020225369854924
-0.880607781701986  0.859512343113706  0.007147818771900
0.859512343113706  -0.978904561411718  0.007147818771900
-0.978904561411718  -0.880607781701986  0.007147818771900
-0.880607781701986  -0.978904561411718  0.007147818771900
0.859512343113706  -0.880607781701986  0.007147818771900
-0.978904561411718  0.859512343113706  0.007147818771900
  Gauss  quadrature  points  and  weights  on  the  reference  triangle  order  p=20
"""


economical_gauss_tetrahedron_table = """\
1
-0.500000000000000  -0.500000000000000  -0.500000000000000  1.333333333333333
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=1

4
-0.723606797749979  -0.723606797749979  -0.723606797749979  0.333333333333333
0.170820393249937  -0.723606797749979  -0.723606797749979  0.333333333333333
-0.723606797749979  0.170820393249937  -0.723606797749979  0.333333333333333
-0.723606797749979  -0.723606797749979  0.170820393249937  0.333333333333333
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=2

5
-0.500000000000000  -0.500000000000000  -0.500000000000000  -1.066666666666667
-0.666666666666667  -0.666666666666667  -0.666666666666667  0.600000000000000
-0.666666666666667  -0.666666666666667  0.000000000000000  0.600000000000000
-0.666666666666667  0.000000000000000  -0.666666666666667  0.600000000000000
0.000000000000000  -0.666666666666667  -0.666666666666667  0.600000000000000
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=3

11
-0.500000000000000  -0.500000000000000  -0.500000000000000  -0.105244444444444
-0.857142857142857  -0.857142857142857  -0.857142857142857  0.060977777777778
-0.857142857142857  -0.857142857142857  0.571428571428571  0.060977777777778
-0.857142857142857  0.571428571428571  -0.857142857142857  0.060977777777778
0.571428571428571  -0.857142857142857  -0.857142857142857  0.060977777777778
-0.201192847666402  -0.201192847666402  -0.798807152333598  0.199111111111111
-0.201192847666402  -0.798807152333598  -0.201192847666402  0.199111111111111
-0.798807152333598  -0.201192847666402  -0.201192847666402  0.199111111111111
-0.201192847666402  -0.798807152333598  -0.798807152333598  0.199111111111111
-0.798807152333598  -0.201192847666402  -0.798807152333598  0.199111111111111
-0.798807152333598  -0.798807152333598  -0.201192847666402  0.199111111111111
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=4

14
-0.814529499378218  -0.814529499378218  -0.814529499378218  0.097990724155149
0.443588498134653  -0.814529499378218  -0.814529499378218  0.097990724155149
-0.814529499378218  0.443588498134653  -0.814529499378218  0.097990724155149
-0.814529499378218  -0.814529499378218  0.443588498134653  0.097990724155149
-0.378228161473399  -0.378228161473399  -0.378228161473399  0.150250567624021
-0.865315515579804  -0.378228161473399  -0.378228161473399  0.150250567624021
-0.378228161473399  -0.865315515579804  -0.378228161473399  0.150250567624021
-0.378228161473399  -0.378228161473399  -0.865315515579804  0.150250567624021
-0.091007408251299  -0.091007408251299  -0.908992591748701  0.056728027702775
-0.091007408251299  -0.908992591748701  -0.091007408251299  0.056728027702775
-0.908992591748701  -0.091007408251299  -0.091007408251299  0.056728027702775
-0.091007408251299  -0.908992591748701  -0.908992591748701  0.056728027702775
-0.908992591748701  -0.091007408251299  -0.908992591748701  0.056728027702775
-0.908992591748701  -0.908992591748701  -0.091007408251299  0.056728027702775
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=5

24
-0.570794257481696  -0.570794257481696  -0.570794257481696  0.053230333677557
-0.287617227554912  -0.570794257481696  -0.570794257481696  0.053230333677557
-0.570794257481696  -0.287617227554912  -0.570794257481696  0.053230333677557
-0.570794257481696  -0.570794257481696  -0.287617227554912  0.053230333677557
-0.918652082930777  -0.918652082930777  -0.918652082930777  0.013436281407094
0.755956248792332  -0.918652082930777  -0.918652082930777  0.013436281407094
-0.918652082930777  0.755956248792332  -0.918652082930777  0.013436281407094
-0.918652082930777  -0.918652082930777  0.755956248792332  0.013436281407094
-0.355324219715449  -0.355324219715449  -0.355324219715449  0.073809575391540
-0.934027340853653  -0.355324219715449  -0.355324219715449  0.073809575391540
-0.355324219715449  -0.934027340853653  -0.355324219715449  0.073809575391540
-0.355324219715449  -0.355324219715449  -0.934027340853653  0.073809575391540
-0.872677996249965  -0.872677996249965  -0.460655337083368  0.064285714285714
-0.872677996249965  -0.460655337083368  -0.872677996249965  0.064285714285714
-0.872677996249965  -0.872677996249965  0.206011329583298  0.064285714285714
-0.872677996249965  0.206011329583298  -0.872677996249965  0.064285714285714
-0.872677996249965  -0.460655337083368  0.206011329583298  0.064285714285714
-0.872677996249965  0.206011329583298  -0.460655337083368  0.064285714285714
-0.460655337083368  -0.872677996249965  -0.872677996249965  0.064285714285714
-0.460655337083368  -0.872677996249965  0.206011329583298  0.064285714285714
-0.460655337083368  0.206011329583298  -0.872677996249965  0.064285714285714
0.206011329583298  -0.872677996249965  -0.460655337083368  0.064285714285714
0.206011329583298  -0.872677996249965  -0.872677996249965  0.064285714285714
0.206011329583298  -0.460655337083368  -0.872677996249965  0.064285714285714
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=6

31
0.000000000000000  0.000000000000000  -1.000000000000000  0.007760141093474
0.000000000000000  -1.000000000000000  0.000000000000000  0.007760141093474
-1.000000000000000  0.000000000000000  0.000000000000000  0.007760141093474
-1.000000000000000  -1.000000000000000  0.000000000000000  0.007760141093474
-1.000000000000000  0.000000000000000  -1.000000000000000  0.007760141093474
0.000000000000000  -1.000000000000000  -1.000000000000000  0.007760141093474
-0.500000000000000  -0.500000000000000  -0.500000000000000  0.146113787728871
-0.843573615339364  -0.843573615339364  -0.843573615339364  0.084799532195309
-0.843573615339364  -0.843573615339364  0.530720846018092  0.084799532195309
-0.843573615339364  0.530720846018092  -0.843573615339364  0.084799532195309
0.530720846018092  -0.843573615339364  -0.843573615339364  0.084799532195309
-0.756313566672190  -0.756313566672190  -0.756313566672190  -0.500141920914655
-0.756313566672190  -0.756313566672190  0.268940700016569  -0.500141920914655
-0.756313566672190  0.268940700016569  -0.756313566672190  -0.500141920914655
0.268940700016569  -0.756313566672190  -0.756313566672190  -0.500141920914655
-0.334921671107159  -0.334921671107159  -0.334921671107159  0.039131402104588
-0.334921671107159  -0.334921671107159  -0.995234986678524  0.039131402104588
-0.334921671107159  -0.995234986678524  -0.334921671107159  0.039131402104588
-0.995234986678524  -0.334921671107159  -0.334921671107159  0.039131402104588
-0.800000000000000  -0.800000000000000  -0.600000000000000  0.220458553791887
-0.800000000000000  -0.600000000000000  -0.800000000000000  0.220458553791887
-0.800000000000000  -0.800000000000000  0.200000000000000  0.220458553791887
-0.800000000000000  0.200000000000000  -0.800000000000000  0.220458553791887
-0.800000000000000  -0.600000000000000  0.200000000000000  0.220458553791887
-0.800000000000000  0.200000000000000  -0.600000000000000  0.220458553791887
-0.600000000000000  -0.800000000000000  -0.800000000000000  0.220458553791887
-0.600000000000000  -0.800000000000000  0.200000000000000  0.220458553791887
-0.600000000000000  0.200000000000000  -0.800000000000000  0.220458553791887
0.200000000000000  -0.800000000000000  -0.600000000000000  0.220458553791887
0.200000000000000  -0.800000000000000  -0.800000000000000  0.220458553791887
0.200000000000000  -0.600000000000000  -0.800000000000000  0.220458553791887
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=7

43
-0.500000000000000  -0.500000000000000  -0.500000000000000  -0.164001509269119
-0.586340136778654  -0.586340136778654  -0.586340136778654  0.114002446582935
-0.586340136778654  -0.586340136778654  -0.240979589664039  0.114002446582935
-0.586340136778654  -0.240979589664039  -0.586340136778654  0.114002446582935
-0.240979589664039  -0.586340136778654  -0.586340136778654  0.114002446582935
-0.835792823378907  -0.835792823378907  -0.835792823378907  0.015736266505071
-0.835792823378907  -0.835792823378907  0.507378470136720  0.015736266505071
-0.835792823378907  0.507378470136720  -0.835792823378907  0.015736266505071
0.507378470136720  -0.835792823378907  -0.835792823378907  0.015736266505071
-0.988436098989604  -0.988436098989604  -0.988436098989604  0.001358672872743
-0.988436098989604  -0.988436098989604  0.965308296968812  0.001358672872743
-0.988436098989604  0.965308296968812  -0.988436098989604  0.001358672872743
0.965308296968812  -0.988436098989604  -0.988436098989604  0.001358672872743
-0.898934519962212  -0.898934519962212  -0.101065480037788  0.036637470595738
-0.898934519962212  -0.101065480037788  -0.898934519962212  0.036637470595738
-0.101065480037788  -0.898934519962212  -0.898934519962212  0.036637470595738
-0.898934519962212  -0.101065480037788  -0.101065480037788  0.036637470595738
-0.101065480037788  -0.898934519962212  -0.101065480037788  0.036637470595738
-0.101065480037788  -0.101065480037788  -0.898934519962212  0.036637470595738
-0.541866927766378  -0.541866927766378  -0.928720834422932  0.045635886469455
-0.541866927766378  -0.928720834422932  -0.541866927766378  0.045635886469455
-0.541866927766378  -0.541866927766378  0.012454689955687  0.045635886469455
-0.541866927766378  0.012454689955687  -0.541866927766378  0.045635886469455
-0.541866927766378  -0.928720834422932  0.012454689955687  0.045635886469455
-0.541866927766378  0.012454689955687  -0.928720834422932  0.045635886469455
-0.928720834422932  -0.541866927766378  -0.541866927766378  0.045635886469455
-0.928720834422932  -0.541866927766378  0.012454689955687  0.045635886469455
-0.928720834422932  0.012454689955687  -0.541866927766378  0.045635886469455
0.012454689955687  -0.541866927766378  -0.928720834422932  0.045635886469455
0.012454689955687  -0.541866927766378  -0.541866927766378  0.045635886469455
0.012454689955687  -0.928720834422932  -0.541866927766378  0.045635886469455
-0.926784500893605  -0.926784500893605  -0.619027916130733  0.017124153129297
-0.926784500893605  -0.619027916130733  -0.926784500893605  0.017124153129297
-0.926784500893605  -0.926784500893605  0.472596917917943  0.017124153129297
-0.926784500893605  0.472596917917943  -0.926784500893605  0.017124153129297
-0.926784500893605  -0.619027916130733  0.472596917917943  0.017124153129297
-0.926784500893605  0.472596917917943  -0.619027916130733  0.017124153129297
-0.619027916130733  -0.926784500893605  -0.926784500893605  0.017124153129297
-0.619027916130733  -0.926784500893605  0.472596917917943  0.017124153129297
-0.619027916130733  0.472596917917943  -0.926784500893605  0.017124153129297
0.472596917917943  -0.926784500893605  -0.619027916130733  0.017124153129297
0.472596917917943  -0.926784500893605  -0.926784500893605  0.017124153129297
0.472596917917943  -0.619027916130733  -0.926784500893605  0.017124153129297
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=8

53
-0.500000000000000  -0.500000000000000  -0.500000000000000  -1.102392306608869
-0.903297922900526  -0.903297922900526  -0.903297922900526  0.014922692552682
-0.903297922900526  -0.903297922900526  0.709893768701580  0.014922692552682
-0.903297922900526  0.709893768701580  -0.903297922900526  0.014922692552682
0.709893768701580  -0.903297922900526  -0.903297922900526  0.014922692552682
-0.350841439764235  -0.350841439764235  -0.350841439764235  0.034475391755947
-0.350841439764235  -0.350841439764235  -0.947475680707294  0.034475391755947
-0.350841439764235  -0.947475680707294  -0.350841439764235  0.034475391755947
-0.947475680707294  -0.350841439764235  -0.350841439764235  0.034475391755947
-0.770766919552010  -0.770766919552010  -0.770766919552010  -0.721478131849612
-0.770766919552010  -0.770766919552010  0.312300758656029  -0.721478131849612
-0.770766919552010  0.312300758656029  -0.770766919552010  -0.721478131849612
0.312300758656029  -0.770766919552010  -0.770766919552010  -0.721478131849612
-0.549020096176972  -0.549020096176972  -0.549020096176972  0.357380609620092
-0.549020096176972  -0.549020096176972  -0.352939711469084  0.357380609620092
-0.549020096176972  -0.352939711469084  -0.549020096176972  0.357380609620092
-0.352939711469084  -0.549020096176972  -0.549020096176972  0.357380609620092
-0.736744381506260  -0.736744381506260  -0.832670596765630  0.277603247076406
-0.736744381506260  -0.832670596765630  -0.736744381506260  0.277603247076406
-0.736744381506260  -0.736744381506260  0.306159359778151  0.277603247076406
-0.736744381506260  0.306159359778151  -0.736744381506260  0.277603247076406
-0.736744381506260  -0.832670596765630  0.306159359778151  0.277603247076406
-0.736744381506260  0.306159359778151  -0.832670596765630  0.277603247076406
-0.832670596765630  -0.736744381506260  -0.736744381506260  0.277603247076406
-0.832670596765630  -0.736744381506260  0.306159359778151  0.277603247076406
-0.832670596765630  0.306159359778151  -0.736744381506260  0.277603247076406
0.306159359778151  -0.736744381506260  -0.832670596765630  0.277603247076406
0.306159359778151  -0.736744381506260  -0.736744381506260  0.277603247076406
0.306159359778151  -0.832670596765630  -0.736744381506260  0.277603247076406
-0.132097077177186  -0.132097077177186  -0.784460280901143  0.026820671221285
-0.132097077177186  -0.784460280901143  -0.132097077177186  0.026820671221285
-0.132097077177186  -0.132097077177186  -0.951345564744484  0.026820671221285
-0.132097077177186  -0.951345564744484  -0.132097077177186  0.026820671221285
-0.132097077177186  -0.784460280901143  -0.951345564744484  0.026820671221285
-0.132097077177186  -0.951345564744484  -0.784460280901143  0.026820671221285
-0.784460280901143  -0.132097077177186  -0.132097077177186  0.026820671221285
-0.784460280901143  -0.132097077177186  -0.951345564744484  0.026820671221285
-0.784460280901143  -0.951345564744484  -0.132097077177186  0.026820671221285
-0.951345564744484  -0.132097077177186  -0.784460280901143  0.026820671221285
-0.951345564744484  -0.132097077177186  -0.132097077177186  0.026820671221285
-0.951345564744484  -0.784460280901143  -0.132097077177186  0.026820671221285
-1.002752554636276  -1.002752554636276  -0.446893054726385  0.003453031004456
-1.002752554636276  -0.446893054726385  -1.002752554636276  0.003453031004456
-1.002752554636276  -1.002752554636276  0.452398163998938  0.003453031004456
-1.002752554636276  0.452398163998938  -1.002752554636276  0.003453031004456
-1.002752554636276  -0.446893054726385  0.452398163998938  0.003453031004456
-1.002752554636276  0.452398163998938  -0.446893054726385  0.003453031004456
-0.446893054726385  -1.002752554636276  -1.002752554636276  0.003453031004456
-0.446893054726385  -1.002752554636276  0.452398163998938  0.003453031004456
-0.446893054726385  0.452398163998938  -1.002752554636276  0.003453031004456
0.452398163998938  -1.002752554636276  -0.446893054726385  0.003453031004456
0.452398163998938  -1.002752554636276  -1.002752554636276  0.003453031004456
0.452398163998938  -0.446893054726385  -1.002752554636276  0.003453031004456
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=9

126
-0.857142857142857  -0.857142857142857  0.571428571428571  0.362902592520648
-0.857142857142857  -0.571428571428571  0.285714285714286  0.362902592520648
-0.857142857142857  -0.285714285714286  0.000000000000000  0.362902592520648
-0.857142857142857  0.000000000000000  -0.285714285714286  0.362902592520648
-0.857142857142857  0.285714285714286  -0.571428571428571  0.362902592520648
-0.857142857142857  0.571428571428571  -0.857142857142857  0.362902592520648
-0.571428571428571  -0.857142857142857  0.285714285714286  0.362902592520648
-0.571428571428571  -0.571428571428571  0.000000000000000  0.362902592520648
-0.571428571428571  -0.285714285714286  -0.285714285714286  0.362902592520648
-0.571428571428571  0.000000000000000  -0.571428571428571  0.362902592520648
-0.571428571428571  0.285714285714286  -0.857142857142857  0.362902592520648
-0.285714285714286  -0.857142857142857  0.000000000000000  0.362902592520648
-0.285714285714286  -0.571428571428571  -0.285714285714286  0.362902592520648
-0.285714285714286  -0.285714285714286  -0.571428571428571  0.362902592520648
-0.285714285714286  0.000000000000000  -0.857142857142857  0.362902592520648
0.000000000000000  -0.857142857142857  -0.285714285714286  0.362902592520648
0.000000000000000  -0.571428571428571  -0.571428571428571  0.362902592520648
0.000000000000000  -0.285714285714286  -0.857142857142857  0.362902592520648
0.285714285714286  -0.857142857142857  -0.571428571428571  0.362902592520648
0.285714285714286  -0.571428571428571  -0.857142857142857  0.362902592520648
0.571428571428571  -0.857142857142857  -0.857142857142857  0.362902592520648
-0.857142857142857  -0.857142857142857  0.285714285714286  0.362902592520648
-0.857142857142857  -0.571428571428571  0.000000000000000  0.362902592520648
-0.857142857142857  -0.285714285714286  -0.285714285714286  0.362902592520648
-0.857142857142857  0.000000000000000  -0.571428571428571  0.362902592520648
-0.857142857142857  0.285714285714286  -0.857142857142857  0.362902592520648
-0.571428571428571  -0.857142857142857  0.000000000000000  0.362902592520648
-0.571428571428571  -0.571428571428571  -0.285714285714286  0.362902592520648
-0.571428571428571  -0.285714285714286  -0.571428571428571  0.362902592520648
-0.571428571428571  0.000000000000000  -0.857142857142857  0.362902592520648
-0.285714285714286  -0.857142857142857  -0.285714285714286  0.362902592520648
-0.285714285714286  -0.571428571428571  -0.571428571428571  0.362902592520648
-0.285714285714286  -0.285714285714286  -0.857142857142857  0.362902592520648
0.000000000000000  -0.857142857142857  -0.571428571428571  0.362902592520648
0.000000000000000  -0.571428571428571  -0.857142857142857  0.362902592520648
0.285714285714286  -0.857142857142857  -0.857142857142857  0.362902592520648
-0.857142857142857  -0.857142857142857  0.000000000000000  0.362902592520648
-0.857142857142857  -0.571428571428571  -0.285714285714286  0.362902592520648
-0.857142857142857  -0.285714285714286  -0.571428571428571  0.362902592520648
-0.857142857142857  0.000000000000000  -0.857142857142857  0.362902592520648
-0.571428571428571  -0.857142857142857  -0.285714285714286  0.362902592520648
-0.571428571428571  -0.571428571428571  -0.571428571428571  0.362902592520648
-0.571428571428571  -0.285714285714286  -0.857142857142857  0.362902592520648
-0.285714285714286  -0.857142857142857  -0.571428571428571  0.362902592520648
-0.285714285714286  -0.571428571428571  -0.857142857142857  0.362902592520648
0.000000000000000  -0.857142857142857  -0.857142857142857  0.362902592520648
-0.857142857142857  -0.857142857142857  -0.285714285714286  0.362902592520648
-0.857142857142857  -0.571428571428571  -0.571428571428571  0.362902592520648
-0.857142857142857  -0.285714285714286  -0.857142857142857  0.362902592520648
-0.571428571428571  -0.857142857142857  -0.571428571428571  0.362902592520648
-0.571428571428571  -0.571428571428571  -0.857142857142857  0.362902592520648
-0.285714285714286  -0.857142857142857  -0.857142857142857  0.362902592520648
-0.857142857142857  -0.857142857142857  -0.571428571428571  0.362902592520648
-0.857142857142857  -0.571428571428571  -0.857142857142857  0.362902592520648
-0.571428571428571  -0.857142857142857  -0.857142857142857  0.362902592520648
-0.857142857142857  -0.857142857142857  -0.857142857142857  0.362902592520648
-0.833333333333333  -0.833333333333333  0.500000000000000  -0.932187812187812
-0.833333333333333  -0.500000000000000  0.166666666666667  -0.932187812187812
-0.833333333333333  -0.166666666666667  -0.166666666666667  -0.932187812187812
-0.833333333333333  0.166666666666667  -0.500000000000000  -0.932187812187812
-0.833333333333333  0.500000000000000  -0.833333333333333  -0.932187812187812
-0.500000000000000  -0.833333333333333  0.166666666666667  -0.932187812187812
-0.500000000000000  -0.500000000000000  -0.166666666666667  -0.932187812187812
-0.500000000000000  -0.166666666666667  -0.500000000000000  -0.932187812187812
-0.500000000000000  0.166666666666667  -0.833333333333333  -0.932187812187812
-0.166666666666667  -0.833333333333333  -0.166666666666667  -0.932187812187812
-0.166666666666667  -0.500000000000000  -0.500000000000000  -0.932187812187812
-0.166666666666667  -0.166666666666667  -0.833333333333333  -0.932187812187812
0.166666666666667  -0.833333333333333  -0.500000000000000  -0.932187812187812
0.166666666666667  -0.500000000000000  -0.833333333333333  -0.932187812187812
0.500000000000000  -0.833333333333333  -0.833333333333333  -0.932187812187812
-0.833333333333333  -0.833333333333333  0.166666666666667  -0.932187812187812
-0.833333333333333  -0.500000000000000  -0.166666666666667  -0.932187812187812
-0.833333333333333  -0.166666666666667  -0.500000000000000  -0.932187812187812
-0.833333333333333  0.166666666666667  -0.833333333333333  -0.932187812187812
-0.500000000000000  -0.833333333333333  -0.166666666666667  -0.932187812187812
-0.500000000000000  -0.500000000000000  -0.500000000000000  -0.932187812187812
-0.500000000000000  -0.166666666666667  -0.833333333333333  -0.932187812187812
-0.166666666666667  -0.833333333333333  -0.500000000000000  -0.932187812187812
-0.166666666666667  -0.500000000000000  -0.833333333333333  -0.932187812187812
0.166666666666667  -0.833333333333333  -0.833333333333333  -0.932187812187812
-0.833333333333333  -0.833333333333333  -0.166666666666667  -0.932187812187812
-0.833333333333333  -0.500000000000000  -0.500000000000000  -0.932187812187812
-0.833333333333333  -0.166666666666667  -0.833333333333333  -0.932187812187812
-0.500000000000000  -0.833333333333333  -0.500000000000000  -0.932187812187812
-0.500000000000000  -0.500000000000000  -0.833333333333333  -0.932187812187812
-0.166666666666667  -0.833333333333333  -0.833333333333333  -0.932187812187812
-0.833333333333333  -0.833333333333333  -0.500000000000000  -0.932187812187812
-0.833333333333333  -0.500000000000000  -0.833333333333333  -0.932187812187812
-0.500000000000000  -0.833333333333333  -0.833333333333333  -0.932187812187812
-0.833333333333333  -0.833333333333333  -0.833333333333333  -0.932187812187812
-0.800000000000000  -0.800000000000000  0.400000000000000  0.815498319838598
-0.800000000000000  -0.400000000000000  0.000000000000000  0.815498319838598
-0.800000000000000  0.000000000000000  -0.400000000000000  0.815498319838598
-0.800000000000000  0.400000000000000  -0.800000000000000  0.815498319838598
-0.400000000000000  -0.800000000000000  0.000000000000000  0.815498319838598
-0.400000000000000  -0.400000000000000  -0.400000000000000  0.815498319838598
-0.400000000000000  0.000000000000000  -0.800000000000000  0.815498319838598
0.000000000000000  -0.800000000000000  -0.400000000000000  0.815498319838598
0.000000000000000  -0.400000000000000  -0.800000000000000  0.815498319838598
0.400000000000000  -0.800000000000000  -0.800000000000000  0.815498319838598
-0.800000000000000  -0.800000000000000  0.000000000000000  0.815498319838598
-0.800000000000000  -0.400000000000000  -0.400000000000000  0.815498319838598
-0.800000000000000  0.000000000000000  -0.800000000000000  0.815498319838598
-0.400000000000000  -0.800000000000000  -0.400000000000000  0.815498319838598
-0.400000000000000  -0.400000000000000  -0.800000000000000  0.815498319838598
0.000000000000000  -0.800000000000000  -0.800000000000000  0.815498319838598
-0.800000000000000  -0.800000000000000  -0.400000000000000  0.815498319838598
-0.800000000000000  -0.400000000000000  -0.800000000000000  0.815498319838598
-0.400000000000000  -0.800000000000000  -0.800000000000000  0.815498319838598
-0.800000000000000  -0.800000000000000  -0.800000000000000  0.815498319838598
-0.750000000000000  -0.750000000000000  0.250000000000000  -0.280203089091978
-0.750000000000000  -0.250000000000000  -0.250000000000000  -0.280203089091978
-0.750000000000000  0.250000000000000  -0.750000000000000  -0.280203089091978
-0.250000000000000  -0.750000000000000  -0.250000000000000  -0.280203089091978
-0.250000000000000  -0.250000000000000  -0.750000000000000  -0.280203089091978
0.250000000000000  -0.750000000000000  -0.750000000000000  -0.280203089091978
-0.750000000000000  -0.750000000000000  -0.250000000000000  -0.280203089091978
-0.750000000000000  -0.250000000000000  -0.750000000000000  -0.280203089091978
-0.250000000000000  -0.750000000000000  -0.750000000000000  -0.280203089091978
-0.750000000000000  -0.750000000000000  -0.750000000000000  -0.280203089091978
-0.666666666666667  -0.666666666666667  0.000000000000000  0.032544642857143
-0.666666666666667  0.000000000000000  -0.666666666666667  0.032544642857143
0.000000000000000  -0.666666666666667  -0.666666666666667  0.032544642857143
-0.666666666666667  -0.666666666666667  -0.666666666666667  0.032544642857143
-0.500000000000000  -0.500000000000000  -0.500000000000000  -0.000752498530276
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=11

210
-0.875000000000000  -0.875000000000000  0.625000000000000  0.420407272132140
-0.875000000000000  -0.625000000000000  0.375000000000000  0.420407272132140
-0.875000000000000  -0.375000000000000  0.125000000000000  0.420407272132140
-0.875000000000000  -0.125000000000000  -0.125000000000000  0.420407272132140
-0.875000000000000  0.125000000000000  -0.375000000000000  0.420407272132140
-0.875000000000000  0.375000000000000  -0.625000000000000  0.420407272132140
-0.875000000000000  0.625000000000000  -0.875000000000000  0.420407272132140
-0.625000000000000  -0.875000000000000  0.375000000000000  0.420407272132140
-0.625000000000000  -0.625000000000000  0.125000000000000  0.420407272132140
-0.625000000000000  -0.375000000000000  -0.125000000000000  0.420407272132140
-0.625000000000000  -0.125000000000000  -0.375000000000000  0.420407272132140
-0.625000000000000  0.125000000000000  -0.625000000000000  0.420407272132140
-0.625000000000000  0.375000000000000  -0.875000000000000  0.420407272132140
-0.375000000000000  -0.875000000000000  0.125000000000000  0.420407272132140
-0.375000000000000  -0.625000000000000  -0.125000000000000  0.420407272132140
-0.375000000000000  -0.375000000000000  -0.375000000000000  0.420407272132140
-0.375000000000000  -0.125000000000000  -0.625000000000000  0.420407272132140
-0.375000000000000  0.125000000000000  -0.875000000000000  0.420407272132140
-0.125000000000000  -0.875000000000000  -0.125000000000000  0.420407272132140
-0.125000000000000  -0.625000000000000  -0.375000000000000  0.420407272132140
-0.125000000000000  -0.375000000000000  -0.625000000000000  0.420407272132140
-0.125000000000000  -0.125000000000000  -0.875000000000000  0.420407272132140
0.125000000000000  -0.875000000000000  -0.375000000000000  0.420407272132140
0.125000000000000  -0.625000000000000  -0.625000000000000  0.420407272132140
0.125000000000000  -0.375000000000000  -0.875000000000000  0.420407272132140
0.375000000000000  -0.875000000000000  -0.625000000000000  0.420407272132140
0.375000000000000  -0.625000000000000  -0.875000000000000  0.420407272132140
0.625000000000000  -0.875000000000000  -0.875000000000000  0.420407272132140
-0.875000000000000  -0.875000000000000  0.375000000000000  0.420407272132140
-0.875000000000000  -0.625000000000000  0.125000000000000  0.420407272132140
-0.875000000000000  -0.375000000000000  -0.125000000000000  0.420407272132140
-0.875000000000000  -0.125000000000000  -0.375000000000000  0.420407272132140
-0.875000000000000  0.125000000000000  -0.625000000000000  0.420407272132140
-0.875000000000000  0.375000000000000  -0.875000000000000  0.420407272132140
-0.625000000000000  -0.875000000000000  0.125000000000000  0.420407272132140
-0.625000000000000  -0.625000000000000  -0.125000000000000  0.420407272132140
-0.625000000000000  -0.375000000000000  -0.375000000000000  0.420407272132140
-0.625000000000000  -0.125000000000000  -0.625000000000000  0.420407272132140
-0.625000000000000  0.125000000000000  -0.875000000000000  0.420407272132140
-0.375000000000000  -0.875000000000000  -0.125000000000000  0.420407272132140
-0.375000000000000  -0.625000000000000  -0.375000000000000  0.420407272132140
-0.375000000000000  -0.375000000000000  -0.625000000000000  0.420407272132140
-0.375000000000000  -0.125000000000000  -0.875000000000000  0.420407272132140
-0.125000000000000  -0.875000000000000  -0.375000000000000  0.420407272132140
-0.125000000000000  -0.625000000000000  -0.625000000000000  0.420407272132140
-0.125000000000000  -0.375000000000000  -0.875000000000000  0.420407272132140
0.125000000000000  -0.875000000000000  -0.625000000000000  0.420407272132140
0.125000000000000  -0.625000000000000  -0.875000000000000  0.420407272132140
0.375000000000000  -0.875000000000000  -0.875000000000000  0.420407272132140
-0.875000000000000  -0.875000000000000  0.125000000000000  0.420407272132140
-0.875000000000000  -0.625000000000000  -0.125000000000000  0.420407272132140
-0.875000000000000  -0.375000000000000  -0.375000000000000  0.420407272132140
-0.875000000000000  -0.125000000000000  -0.625000000000000  0.420407272132140
-0.875000000000000  0.125000000000000  -0.875000000000000  0.420407272132140
-0.625000000000000  -0.875000000000000  -0.125000000000000  0.420407272132140
-0.625000000000000  -0.625000000000000  -0.375000000000000  0.420407272132140
-0.625000000000000  -0.375000000000000  -0.625000000000000  0.420407272132140
-0.625000000000000  -0.125000000000000  -0.875000000000000  0.420407272132140
-0.375000000000000  -0.875000000000000  -0.375000000000000  0.420407272132140
-0.375000000000000  -0.625000000000000  -0.625000000000000  0.420407272132140
-0.375000000000000  -0.375000000000000  -0.875000000000000  0.420407272132140
-0.125000000000000  -0.875000000000000  -0.625000000000000  0.420407272132140
-0.125000000000000  -0.625000000000000  -0.875000000000000  0.420407272132140
0.125000000000000  -0.875000000000000  -0.875000000000000  0.420407272132140
-0.875000000000000  -0.875000000000000  -0.125000000000000  0.420407272132140
-0.875000000000000  -0.625000000000000  -0.375000000000000  0.420407272132140
-0.875000000000000  -0.375000000000000  -0.625000000000000  0.420407272132140
-0.875000000000000  -0.125000000000000  -0.875000000000000  0.420407272132140
-0.625000000000000  -0.875000000000000  -0.375000000000000  0.420407272132140
-0.625000000000000  -0.625000000000000  -0.625000000000000  0.420407272132140
-0.625000000000000  -0.375000000000000  -0.875000000000000  0.420407272132140
-0.375000000000000  -0.875000000000000  -0.625000000000000  0.420407272132140
-0.375000000000000  -0.625000000000000  -0.875000000000000  0.420407272132140
-0.125000000000000  -0.875000000000000  -0.875000000000000  0.420407272132140
-0.875000000000000  -0.875000000000000  -0.375000000000000  0.420407272132140
-0.875000000000000  -0.625000000000000  -0.625000000000000  0.420407272132140
-0.875000000000000  -0.375000000000000  -0.875000000000000  0.420407272132140
-0.625000000000000  -0.875000000000000  -0.625000000000000  0.420407272132140
-0.625000000000000  -0.625000000000000  -0.875000000000000  0.420407272132140
-0.375000000000000  -0.875000000000000  -0.875000000000000  0.420407272132140
-0.875000000000000  -0.875000000000000  -0.625000000000000  0.420407272132140
-0.875000000000000  -0.625000000000000  -0.875000000000000  0.420407272132140
-0.625000000000000  -0.875000000000000  -0.875000000000000  0.420407272132140
-0.875000000000000  -0.875000000000000  -0.875000000000000  0.420407272132140
-0.857142857142857  -0.857142857142857  0.571428571428571  -1.185481802234117
-0.857142857142857  -0.571428571428571  0.285714285714286  -1.185481802234117
-0.857142857142857  -0.285714285714286  0.000000000000000  -1.185481802234117
-0.857142857142857  0.000000000000000  -0.285714285714286  -1.185481802234117
-0.857142857142857  0.285714285714286  -0.571428571428571  -1.185481802234117
-0.857142857142857  0.571428571428571  -0.857142857142857  -1.185481802234117
-0.571428571428571  -0.857142857142857  0.285714285714286  -1.185481802234117
-0.571428571428571  -0.571428571428571  0.000000000000000  -1.185481802234117
-0.571428571428571  -0.285714285714286  -0.285714285714286  -1.185481802234117
-0.571428571428571  0.000000000000000  -0.571428571428571  -1.185481802234117
-0.571428571428571  0.285714285714286  -0.857142857142857  -1.185481802234117
-0.285714285714286  -0.857142857142857  0.000000000000000  -1.185481802234117
-0.285714285714286  -0.571428571428571  -0.285714285714286  -1.185481802234117
-0.285714285714286  -0.285714285714286  -0.571428571428571  -1.185481802234117
-0.285714285714286  0.000000000000000  -0.857142857142857  -1.185481802234117
0.000000000000000  -0.857142857142857  -0.285714285714286  -1.185481802234117
0.000000000000000  -0.571428571428571  -0.571428571428571  -1.185481802234117
0.000000000000000  -0.285714285714286  -0.857142857142857  -1.185481802234117
0.285714285714286  -0.857142857142857  -0.571428571428571  -1.185481802234117
0.285714285714286  -0.571428571428571  -0.857142857142857  -1.185481802234117
0.571428571428571  -0.857142857142857  -0.857142857142857  -1.185481802234117
-0.857142857142857  -0.857142857142857  0.285714285714286  -1.185481802234117
-0.857142857142857  -0.571428571428571  0.000000000000000  -1.185481802234117
-0.857142857142857  -0.285714285714286  -0.285714285714286  -1.185481802234117
-0.857142857142857  0.000000000000000  -0.571428571428571  -1.185481802234117
-0.857142857142857  0.285714285714286  -0.857142857142857  -1.185481802234117
-0.571428571428571  -0.857142857142857  0.000000000000000  -1.185481802234117
-0.571428571428571  -0.571428571428571  -0.285714285714286  -1.185481802234117
-0.571428571428571  -0.285714285714286  -0.571428571428571  -1.185481802234117
-0.571428571428571  0.000000000000000  -0.857142857142857  -1.185481802234117
-0.285714285714286  -0.857142857142857  -0.285714285714286  -1.185481802234117
-0.285714285714286  -0.571428571428571  -0.571428571428571  -1.185481802234117
-0.285714285714286  -0.285714285714286  -0.857142857142857  -1.185481802234117
0.000000000000000  -0.857142857142857  -0.571428571428571  -1.185481802234117
0.000000000000000  -0.571428571428571  -0.857142857142857  -1.185481802234117
0.285714285714286  -0.857142857142857  -0.857142857142857  -1.185481802234117
-0.857142857142857  -0.857142857142857  0.000000000000000  -1.185481802234117
-0.857142857142857  -0.571428571428571  -0.285714285714286  -1.185481802234117
-0.857142857142857  -0.285714285714286  -0.571428571428571  -1.185481802234117
-0.857142857142857  0.000000000000000  -0.857142857142857  -1.185481802234117
-0.571428571428571  -0.857142857142857  -0.285714285714286  -1.185481802234117
-0.571428571428571  -0.571428571428571  -0.571428571428571  -1.185481802234117
-0.571428571428571  -0.285714285714286  -0.857142857142857  -1.185481802234117
-0.285714285714286  -0.857142857142857  -0.571428571428571  -1.185481802234117
-0.285714285714286  -0.571428571428571  -0.857142857142857  -1.185481802234117
0.000000000000000  -0.857142857142857  -0.857142857142857  -1.185481802234117
-0.857142857142857  -0.857142857142857  -0.285714285714286  -1.185481802234117
-0.857142857142857  -0.571428571428571  -0.571428571428571  -1.185481802234117
-0.857142857142857  -0.285714285714286  -0.857142857142857  -1.185481802234117
-0.571428571428571  -0.857142857142857  -0.571428571428571  -1.185481802234117
-0.571428571428571  -0.571428571428571  -0.857142857142857  -1.185481802234117
-0.285714285714286  -0.857142857142857  -0.857142857142857  -1.185481802234117
-0.857142857142857  -0.857142857142857  -0.571428571428571  -1.185481802234117
-0.857142857142857  -0.571428571428571  -0.857142857142857  -1.185481802234117
-0.571428571428571  -0.857142857142857  -0.857142857142857  -1.185481802234117
-0.857142857142857  -0.857142857142857  -0.857142857142857  -1.185481802234117
-0.833333333333333  -0.833333333333333  0.500000000000000  1.198527187098616
-0.833333333333333  -0.500000000000000  0.166666666666667  1.198527187098616
-0.833333333333333  -0.166666666666667  -0.166666666666667  1.198527187098616
-0.833333333333333  0.166666666666667  -0.500000000000000  1.198527187098616
-0.833333333333333  0.500000000000000  -0.833333333333333  1.198527187098616
-0.500000000000000  -0.833333333333333  0.166666666666667  1.198527187098616
-0.500000000000000  -0.500000000000000  -0.166666666666667  1.198527187098616
-0.500000000000000  -0.166666666666667  -0.500000000000000  1.198527187098616
-0.500000000000000  0.166666666666667  -0.833333333333333  1.198527187098616
-0.166666666666667  -0.833333333333333  -0.166666666666667  1.198527187098616
-0.166666666666667  -0.500000000000000  -0.500000000000000  1.198527187098616
-0.166666666666667  -0.166666666666667  -0.833333333333333  1.198527187098616
0.166666666666667  -0.833333333333333  -0.500000000000000  1.198527187098616
0.166666666666667  -0.500000000000000  -0.833333333333333  1.198527187098616
0.500000000000000  -0.833333333333333  -0.833333333333333  1.198527187098616
-0.833333333333333  -0.833333333333333  0.166666666666667  1.198527187098616
-0.833333333333333  -0.500000000000000  -0.166666666666667  1.198527187098616
-0.833333333333333  -0.166666666666667  -0.500000000000000  1.198527187098616
-0.833333333333333  0.166666666666667  -0.833333333333333  1.198527187098616
-0.500000000000000  -0.833333333333333  -0.166666666666667  1.198527187098616
-0.500000000000000  -0.500000000000000  -0.500000000000000  1.198527187098616
-0.500000000000000  -0.166666666666667  -0.833333333333333  1.198527187098616
-0.166666666666667  -0.833333333333333  -0.500000000000000  1.198527187098616
-0.166666666666667  -0.500000000000000  -0.833333333333333  1.198527187098616
0.166666666666667  -0.833333333333333  -0.833333333333333  1.198527187098616
-0.833333333333333  -0.833333333333333  -0.166666666666667  1.198527187098616
-0.833333333333333  -0.500000000000000  -0.500000000000000  1.198527187098616
-0.833333333333333  -0.166666666666667  -0.833333333333333  1.198527187098616
-0.500000000000000  -0.833333333333333  -0.500000000000000  1.198527187098616
-0.500000000000000  -0.500000000000000  -0.833333333333333  1.198527187098616
-0.166666666666667  -0.833333333333333  -0.833333333333333  1.198527187098616
-0.833333333333333  -0.833333333333333  -0.500000000000000  1.198527187098616
-0.833333333333333  -0.500000000000000  -0.833333333333333  1.198527187098616
-0.500000000000000  -0.833333333333333  -0.833333333333333  1.198527187098616
-0.833333333333333  -0.833333333333333  -0.833333333333333  1.198527187098616
-0.800000000000000  -0.800000000000000  0.400000000000000  -0.522755333229870
-0.800000000000000  -0.400000000000000  0.000000000000000  -0.522755333229870
-0.800000000000000  0.000000000000000  -0.400000000000000  -0.522755333229870
-0.800000000000000  0.400000000000000  -0.800000000000000  -0.522755333229870
-0.400000000000000  -0.800000000000000  0.000000000000000  -0.522755333229870
-0.400000000000000  -0.400000000000000  -0.400000000000000  -0.522755333229870
-0.400000000000000  0.000000000000000  -0.800000000000000  -0.522755333229870
0.000000000000000  -0.800000000000000  -0.400000000000000  -0.522755333229870
0.000000000000000  -0.400000000000000  -0.800000000000000  -0.522755333229870
0.400000000000000  -0.800000000000000  -0.800000000000000  -0.522755333229870
-0.800000000000000  -0.800000000000000  0.000000000000000  -0.522755333229870
-0.800000000000000  -0.400000000000000  -0.400000000000000  -0.522755333229870
-0.800000000000000  0.000000000000000  -0.800000000000000  -0.522755333229870
-0.400000000000000  -0.800000000000000  -0.400000000000000  -0.522755333229870
-0.400000000000000  -0.400000000000000  -0.800000000000000  -0.522755333229870
0.000000000000000  -0.800000000000000  -0.800000000000000  -0.522755333229870
-0.800000000000000  -0.800000000000000  -0.400000000000000  -0.522755333229870
-0.800000000000000  -0.400000000000000  -0.800000000000000  -0.522755333229870
-0.400000000000000  -0.800000000000000  -0.800000000000000  -0.522755333229870
-0.800000000000000  -0.800000000000000  -0.800000000000000  -0.522755333229870
-0.750000000000000  -0.750000000000000  0.250000000000000  0.093401029697326
-0.750000000000000  -0.250000000000000  -0.250000000000000  0.093401029697326
-0.750000000000000  0.250000000000000  -0.750000000000000  0.093401029697326
-0.250000000000000  -0.750000000000000  -0.250000000000000  0.093401029697326
-0.250000000000000  -0.250000000000000  -0.750000000000000  0.093401029697326
0.250000000000000  -0.750000000000000  -0.750000000000000  0.093401029697326
-0.750000000000000  -0.750000000000000  -0.250000000000000  0.093401029697326
-0.750000000000000  -0.250000000000000  -0.750000000000000  0.093401029697326
-0.250000000000000  -0.750000000000000  -0.750000000000000  0.093401029697326
-0.750000000000000  -0.750000000000000  -0.750000000000000  0.093401029697326
-0.666666666666667  -0.666666666666667  0.000000000000000  -0.005325487012987
-0.666666666666667  0.000000000000000  -0.666666666666667  -0.005325487012987
0.000000000000000  -0.666666666666667  -0.666666666666667  -0.005325487012987
-0.666666666666667  -0.666666666666667  -0.666666666666667  -0.005325487012987
-0.500000000000000  -0.500000000000000  -0.500000000000000  0.000050166568685
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=13

330
-0.888888888888889  -0.888888888888889  0.666666666666667  0.514536992829256
-0.888888888888889  -0.666666666666667  0.444444444444444  0.514536992829256
-0.888888888888889  -0.444444444444444  0.222222222222222  0.514536992829256
-0.888888888888889  -0.222222222222222  0.000000000000000  0.514536992829256
-0.888888888888889  0.000000000000000  -0.222222222222222  0.514536992829256
-0.888888888888889  0.222222222222222  -0.444444444444444  0.514536992829256
-0.888888888888889  0.444444444444444  -0.666666666666667  0.514536992829256
-0.888888888888889  0.666666666666667  -0.888888888888889  0.514536992829256
-0.666666666666667  -0.888888888888889  0.444444444444444  0.514536992829256
-0.666666666666667  -0.666666666666667  0.222222222222222  0.514536992829256
-0.666666666666667  -0.444444444444444  0.000000000000000  0.514536992829256
-0.666666666666667  -0.222222222222222  -0.222222222222222  0.514536992829256
-0.666666666666667  0.000000000000000  -0.444444444444444  0.514536992829256
-0.666666666666667  0.222222222222222  -0.666666666666667  0.514536992829256
-0.666666666666667  0.444444444444444  -0.888888888888889  0.514536992829256
-0.444444444444444  -0.888888888888889  0.222222222222222  0.514536992829256
-0.444444444444444  -0.666666666666667  0.000000000000000  0.514536992829256
-0.444444444444444  -0.444444444444444  -0.222222222222222  0.514536992829256
-0.444444444444444  -0.222222222222222  -0.444444444444444  0.514536992829256
-0.444444444444444  0.000000000000000  -0.666666666666667  0.514536992829256
-0.444444444444444  0.222222222222222  -0.888888888888889  0.514536992829256
-0.222222222222222  -0.888888888888889  0.000000000000000  0.514536992829256
-0.222222222222222  -0.666666666666667  -0.222222222222222  0.514536992829256
-0.222222222222222  -0.444444444444444  -0.444444444444444  0.514536992829256
-0.222222222222222  -0.222222222222222  -0.666666666666667  0.514536992829256
-0.222222222222222  0.000000000000000  -0.888888888888889  0.514536992829256
0.000000000000000  -0.888888888888889  -0.222222222222222  0.514536992829256
0.000000000000000  -0.666666666666667  -0.444444444444444  0.514536992829256
0.000000000000000  -0.444444444444444  -0.666666666666667  0.514536992829256
0.000000000000000  -0.222222222222222  -0.888888888888889  0.514536992829256
0.222222222222222  -0.888888888888889  -0.444444444444444  0.514536992829256
0.222222222222222  -0.666666666666667  -0.666666666666667  0.514536992829256
0.222222222222222  -0.444444444444444  -0.888888888888889  0.514536992829256
0.444444444444444  -0.888888888888889  -0.666666666666667  0.514536992829256
0.444444444444444  -0.666666666666667  -0.888888888888889  0.514536992829256
0.666666666666667  -0.888888888888889  -0.888888888888889  0.514536992829256
-0.888888888888889  -0.888888888888889  0.444444444444444  0.514536992829256
-0.888888888888889  -0.666666666666667  0.222222222222222  0.514536992829256
-0.888888888888889  -0.444444444444444  0.000000000000000  0.514536992829256
-0.888888888888889  -0.222222222222222  -0.222222222222222  0.514536992829256
-0.888888888888889  0.000000000000000  -0.444444444444444  0.514536992829256
-0.888888888888889  0.222222222222222  -0.666666666666667  0.514536992829256
-0.888888888888889  0.444444444444444  -0.888888888888889  0.514536992829256
-0.666666666666667  -0.888888888888889  0.222222222222222  0.514536992829256
-0.666666666666667  -0.666666666666667  0.000000000000000  0.514536992829256
-0.666666666666667  -0.444444444444444  -0.222222222222222  0.514536992829256
-0.666666666666667  -0.222222222222222  -0.444444444444444  0.514536992829256
-0.666666666666667  0.000000000000000  -0.666666666666667  0.514536992829256
-0.666666666666667  0.222222222222222  -0.888888888888889  0.514536992829256
-0.444444444444444  -0.888888888888889  0.000000000000000  0.514536992829256
-0.444444444444444  -0.666666666666667  -0.222222222222222  0.514536992829256
-0.444444444444444  -0.444444444444444  -0.444444444444444  0.514536992829256
-0.444444444444444  -0.222222222222222  -0.666666666666667  0.514536992829256
-0.444444444444444  0.000000000000000  -0.888888888888889  0.514536992829256
-0.222222222222222  -0.888888888888889  -0.222222222222222  0.514536992829256
-0.222222222222222  -0.666666666666667  -0.444444444444444  0.514536992829256
-0.222222222222222  -0.444444444444444  -0.666666666666667  0.514536992829256
-0.222222222222222  -0.222222222222222  -0.888888888888889  0.514536992829256
0.000000000000000  -0.888888888888889  -0.444444444444444  0.514536992829256
0.000000000000000  -0.666666666666667  -0.666666666666667  0.514536992829256
0.000000000000000  -0.444444444444444  -0.888888888888889  0.514536992829256
0.222222222222222  -0.888888888888889  -0.666666666666667  0.514536992829256
0.222222222222222  -0.666666666666667  -0.888888888888889  0.514536992829256
0.444444444444444  -0.888888888888889  -0.888888888888889  0.514536992829256
-0.888888888888889  -0.888888888888889  0.222222222222222  0.514536992829256
-0.888888888888889  -0.666666666666667  0.000000000000000  0.514536992829256
-0.888888888888889  -0.444444444444444  -0.222222222222222  0.514536992829256
-0.888888888888889  -0.222222222222222  -0.444444444444444  0.514536992829256
-0.888888888888889  0.000000000000000  -0.666666666666667  0.514536992829256
-0.888888888888889  0.222222222222222  -0.888888888888889  0.514536992829256
-0.666666666666667  -0.888888888888889  0.000000000000000  0.514536992829256
-0.666666666666667  -0.666666666666667  -0.222222222222222  0.514536992829256
-0.666666666666667  -0.444444444444444  -0.444444444444444  0.514536992829256
-0.666666666666667  -0.222222222222222  -0.666666666666667  0.514536992829256
-0.666666666666667  0.000000000000000  -0.888888888888889  0.514536992829256
-0.444444444444444  -0.888888888888889  -0.222222222222222  0.514536992829256
-0.444444444444444  -0.666666666666667  -0.444444444444444  0.514536992829256
-0.444444444444444  -0.444444444444444  -0.666666666666667  0.514536992829256
-0.444444444444444  -0.222222222222222  -0.888888888888889  0.514536992829256
-0.222222222222222  -0.888888888888889  -0.444444444444444  0.514536992829256
-0.222222222222222  -0.666666666666667  -0.666666666666667  0.514536992829256
-0.222222222222222  -0.444444444444444  -0.888888888888889  0.514536992829256
0.000000000000000  -0.888888888888889  -0.666666666666667  0.514536992829256
0.000000000000000  -0.666666666666667  -0.888888888888889  0.514536992829256
0.222222222222222  -0.888888888888889  -0.888888888888889  0.514536992829256
-0.888888888888889  -0.888888888888889  0.000000000000000  0.514536992829256
-0.888888888888889  -0.666666666666667  -0.222222222222222  0.514536992829256
-0.888888888888889  -0.444444444444444  -0.444444444444444  0.514536992829256
-0.888888888888889  -0.222222222222222  -0.666666666666667  0.514536992829256
-0.888888888888889  0.000000000000000  -0.888888888888889  0.514536992829256
-0.666666666666667  -0.888888888888889  -0.222222222222222  0.514536992829256
-0.666666666666667  -0.666666666666667  -0.444444444444444  0.514536992829256
-0.666666666666667  -0.444444444444444  -0.666666666666667  0.514536992829256
-0.666666666666667  -0.222222222222222  -0.888888888888889  0.514536992829256
-0.444444444444444  -0.888888888888889  -0.444444444444444  0.514536992829256
-0.444444444444444  -0.666666666666667  -0.666666666666667  0.514536992829256
-0.444444444444444  -0.444444444444444  -0.888888888888889  0.514536992829256
-0.222222222222222  -0.888888888888889  -0.666666666666667  0.514536992829256
-0.222222222222222  -0.666666666666667  -0.888888888888889  0.514536992829256
0.000000000000000  -0.888888888888889  -0.888888888888889  0.514536992829256
-0.888888888888889  -0.888888888888889  -0.222222222222222  0.514536992829256
-0.888888888888889  -0.666666666666667  -0.444444444444444  0.514536992829256
-0.888888888888889  -0.444444444444444  -0.666666666666667  0.514536992829256
-0.888888888888889  -0.222222222222222  -0.888888888888889  0.514536992829256
-0.666666666666667  -0.888888888888889  -0.444444444444444  0.514536992829256
-0.666666666666667  -0.666666666666667  -0.666666666666667  0.514536992829256
-0.666666666666667  -0.444444444444444  -0.888888888888889  0.514536992829256
-0.444444444444444  -0.888888888888889  -0.666666666666667  0.514536992829256
-0.444444444444444  -0.666666666666667  -0.888888888888889  0.514536992829256
-0.222222222222222  -0.888888888888889  -0.888888888888889  0.514536992829256
-0.888888888888889  -0.888888888888889  -0.444444444444444  0.514536992829256
-0.888888888888889  -0.666666666666667  -0.666666666666667  0.514536992829256
-0.888888888888889  -0.444444444444444  -0.888888888888889  0.514536992829256
-0.666666666666667  -0.888888888888889  -0.666666666666667  0.514536992829256
-0.666666666666667  -0.666666666666667  -0.888888888888889  0.514536992829256
-0.444444444444444  -0.888888888888889  -0.888888888888889  0.514536992829256
-0.888888888888889  -0.888888888888889  -0.666666666666667  0.514536992829256
-0.888888888888889  -0.666666666666667  -0.888888888888889  0.514536992829256
-0.666666666666667  -0.888888888888889  -0.888888888888889  0.514536992829256
-0.888888888888889  -0.888888888888889  -0.888888888888889  0.514536992829256
-0.875000000000000  -0.875000000000000  0.625000000000000  -1.582709730379821
-0.875000000000000  -0.625000000000000  0.375000000000000  -1.582709730379821
-0.875000000000000  -0.375000000000000  0.125000000000000  -1.582709730379821
-0.875000000000000  -0.125000000000000  -0.125000000000000  -1.582709730379821
-0.875000000000000  0.125000000000000  -0.375000000000000  -1.582709730379821
-0.875000000000000  0.375000000000000  -0.625000000000000  -1.582709730379821
-0.875000000000000  0.625000000000000  -0.875000000000000  -1.582709730379821
-0.625000000000000  -0.875000000000000  0.375000000000000  -1.582709730379821
-0.625000000000000  -0.625000000000000  0.125000000000000  -1.582709730379821
-0.625000000000000  -0.375000000000000  -0.125000000000000  -1.582709730379821
-0.625000000000000  -0.125000000000000  -0.375000000000000  -1.582709730379821
-0.625000000000000  0.125000000000000  -0.625000000000000  -1.582709730379821
-0.625000000000000  0.375000000000000  -0.875000000000000  -1.582709730379821
-0.375000000000000  -0.875000000000000  0.125000000000000  -1.582709730379821
-0.375000000000000  -0.625000000000000  -0.125000000000000  -1.582709730379821
-0.375000000000000  -0.375000000000000  -0.375000000000000  -1.582709730379821
-0.375000000000000  -0.125000000000000  -0.625000000000000  -1.582709730379821
-0.375000000000000  0.125000000000000  -0.875000000000000  -1.582709730379821
-0.125000000000000  -0.875000000000000  -0.125000000000000  -1.582709730379821
-0.125000000000000  -0.625000000000000  -0.375000000000000  -1.582709730379821
-0.125000000000000  -0.375000000000000  -0.625000000000000  -1.582709730379821
-0.125000000000000  -0.125000000000000  -0.875000000000000  -1.582709730379821
0.125000000000000  -0.875000000000000  -0.375000000000000  -1.582709730379821
0.125000000000000  -0.625000000000000  -0.625000000000000  -1.582709730379821
0.125000000000000  -0.375000000000000  -0.875000000000000  -1.582709730379821
0.375000000000000  -0.875000000000000  -0.625000000000000  -1.582709730379821
0.375000000000000  -0.625000000000000  -0.875000000000000  -1.582709730379821
0.625000000000000  -0.875000000000000  -0.875000000000000  -1.582709730379821
-0.875000000000000  -0.875000000000000  0.375000000000000  -1.582709730379821
-0.875000000000000  -0.625000000000000  0.125000000000000  -1.582709730379821
-0.875000000000000  -0.375000000000000  -0.125000000000000  -1.582709730379821
-0.875000000000000  -0.125000000000000  -0.375000000000000  -1.582709730379821
-0.875000000000000  0.125000000000000  -0.625000000000000  -1.582709730379821
-0.875000000000000  0.375000000000000  -0.875000000000000  -1.582709730379821
-0.625000000000000  -0.875000000000000  0.125000000000000  -1.582709730379821
-0.625000000000000  -0.625000000000000  -0.125000000000000  -1.582709730379821
-0.625000000000000  -0.375000000000000  -0.375000000000000  -1.582709730379821
-0.625000000000000  -0.125000000000000  -0.625000000000000  -1.582709730379821
-0.625000000000000  0.125000000000000  -0.875000000000000  -1.582709730379821
-0.375000000000000  -0.875000000000000  -0.125000000000000  -1.582709730379821
-0.375000000000000  -0.625000000000000  -0.375000000000000  -1.582709730379821
-0.375000000000000  -0.375000000000000  -0.625000000000000  -1.582709730379821
-0.375000000000000  -0.125000000000000  -0.875000000000000  -1.582709730379821
-0.125000000000000  -0.875000000000000  -0.375000000000000  -1.582709730379821
-0.125000000000000  -0.625000000000000  -0.625000000000000  -1.582709730379821
-0.125000000000000  -0.375000000000000  -0.875000000000000  -1.582709730379821
0.125000000000000  -0.875000000000000  -0.625000000000000  -1.582709730379821
0.125000000000000  -0.625000000000000  -0.875000000000000  -1.582709730379821
0.375000000000000  -0.875000000000000  -0.875000000000000  -1.582709730379821
-0.875000000000000  -0.875000000000000  0.125000000000000  -1.582709730379821
-0.875000000000000  -0.625000000000000  -0.125000000000000  -1.582709730379821
-0.875000000000000  -0.375000000000000  -0.375000000000000  -1.582709730379821
-0.875000000000000  -0.125000000000000  -0.625000000000000  -1.582709730379821
-0.875000000000000  0.125000000000000  -0.875000000000000  -1.582709730379821
-0.625000000000000  -0.875000000000000  -0.125000000000000  -1.582709730379821
-0.625000000000000  -0.625000000000000  -0.375000000000000  -1.582709730379821
-0.625000000000000  -0.375000000000000  -0.625000000000000  -1.582709730379821
-0.625000000000000  -0.125000000000000  -0.875000000000000  -1.582709730379821
-0.375000000000000  -0.875000000000000  -0.375000000000000  -1.582709730379821
-0.375000000000000  -0.625000000000000  -0.625000000000000  -1.582709730379821
-0.375000000000000  -0.375000000000000  -0.875000000000000  -1.582709730379821
-0.125000000000000  -0.875000000000000  -0.625000000000000  -1.582709730379821
-0.125000000000000  -0.625000000000000  -0.875000000000000  -1.582709730379821
0.125000000000000  -0.875000000000000  -0.875000000000000  -1.582709730379821
-0.875000000000000  -0.875000000000000  -0.125000000000000  -1.582709730379821
-0.875000000000000  -0.625000000000000  -0.375000000000000  -1.582709730379821
-0.875000000000000  -0.375000000000000  -0.625000000000000  -1.582709730379821
-0.875000000000000  -0.125000000000000  -0.875000000000000  -1.582709730379821
-0.625000000000000  -0.875000000000000  -0.375000000000000  -1.582709730379821
-0.625000000000000  -0.625000000000000  -0.625000000000000  -1.582709730379821
-0.625000000000000  -0.375000000000000  -0.875000000000000  -1.582709730379821
-0.375000000000000  -0.875000000000000  -0.625000000000000  -1.582709730379821
-0.375000000000000  -0.625000000000000  -0.875000000000000  -1.582709730379821
-0.125000000000000  -0.875000000000000  -0.875000000000000  -1.582709730379821
-0.875000000000000  -0.875000000000000  -0.375000000000000  -1.582709730379821
-0.875000000000000  -0.625000000000000  -0.625000000000000  -1.582709730379821
-0.875000000000000  -0.375000000000000  -0.875000000000000  -1.582709730379821
-0.625000000000000  -0.875000000000000  -0.625000000000000  -1.582709730379821
-0.625000000000000  -0.625000000000000  -0.875000000000000  -1.582709730379821
-0.375000000000000  -0.875000000000000  -0.875000000000000  -1.582709730379821
-0.875000000000000  -0.875000000000000  -0.625000000000000  -1.582709730379821
-0.875000000000000  -0.625000000000000  -0.875000000000000  -1.582709730379821
-0.625000000000000  -0.875000000000000  -0.875000000000000  -1.582709730379821
-0.875000000000000  -0.875000000000000  -0.875000000000000  -1.582709730379821
-0.857142857142857  -0.857142857142857  0.571428571428571  1.815269009670992
-0.857142857142857  -0.571428571428571  0.285714285714286  1.815269009670992
-0.857142857142857  -0.285714285714286  0.000000000000000  1.815269009670992
-0.857142857142857  0.000000000000000  -0.285714285714286  1.815269009670992
-0.857142857142857  0.285714285714286  -0.571428571428571  1.815269009670992
-0.857142857142857  0.571428571428571  -0.857142857142857  1.815269009670992
-0.571428571428571  -0.857142857142857  0.285714285714286  1.815269009670992
-0.571428571428571  -0.571428571428571  0.000000000000000  1.815269009670992
-0.571428571428571  -0.285714285714286  -0.285714285714286  1.815269009670992
-0.571428571428571  0.000000000000000  -0.571428571428571  1.815269009670992
-0.571428571428571  0.285714285714286  -0.857142857142857  1.815269009670992
-0.285714285714286  -0.857142857142857  0.000000000000000  1.815269009670992
-0.285714285714286  -0.571428571428571  -0.285714285714286  1.815269009670992
-0.285714285714286  -0.285714285714286  -0.571428571428571  1.815269009670992
-0.285714285714286  0.000000000000000  -0.857142857142857  1.815269009670992
0.000000000000000  -0.857142857142857  -0.285714285714286  1.815269009670992
0.000000000000000  -0.571428571428571  -0.571428571428571  1.815269009670992
0.000000000000000  -0.285714285714286  -0.857142857142857  1.815269009670992
0.285714285714286  -0.857142857142857  -0.571428571428571  1.815269009670992
0.285714285714286  -0.571428571428571  -0.857142857142857  1.815269009670992
0.571428571428571  -0.857142857142857  -0.857142857142857  1.815269009670992
-0.857142857142857  -0.857142857142857  0.285714285714286  1.815269009670992
-0.857142857142857  -0.571428571428571  0.000000000000000  1.815269009670992
-0.857142857142857  -0.285714285714286  -0.285714285714286  1.815269009670992
-0.857142857142857  0.000000000000000  -0.571428571428571  1.815269009670992
-0.857142857142857  0.285714285714286  -0.857142857142857  1.815269009670992
-0.571428571428571  -0.857142857142857  0.000000000000000  1.815269009670992
-0.571428571428571  -0.571428571428571  -0.285714285714286  1.815269009670992
-0.571428571428571  -0.285714285714286  -0.571428571428571  1.815269009670992
-0.571428571428571  0.000000000000000  -0.857142857142857  1.815269009670992
-0.285714285714286  -0.857142857142857  -0.285714285714286  1.815269009670992
-0.285714285714286  -0.571428571428571  -0.571428571428571  1.815269009670992
-0.285714285714286  -0.285714285714286  -0.857142857142857  1.815269009670992
0.000000000000000  -0.857142857142857  -0.571428571428571  1.815269009670992
0.000000000000000  -0.571428571428571  -0.857142857142857  1.815269009670992
0.285714285714286  -0.857142857142857  -0.857142857142857  1.815269009670992
-0.857142857142857  -0.857142857142857  0.000000000000000  1.815269009670992
-0.857142857142857  -0.571428571428571  -0.285714285714286  1.815269009670992
-0.857142857142857  -0.285714285714286  -0.571428571428571  1.815269009670992
-0.857142857142857  0.000000000000000  -0.857142857142857  1.815269009670992
-0.571428571428571  -0.857142857142857  -0.285714285714286  1.815269009670992
-0.571428571428571  -0.571428571428571  -0.571428571428571  1.815269009670992
-0.571428571428571  -0.285714285714286  -0.857142857142857  1.815269009670992
-0.285714285714286  -0.857142857142857  -0.571428571428571  1.815269009670992
-0.285714285714286  -0.571428571428571  -0.857142857142857  1.815269009670992
0.000000000000000  -0.857142857142857  -0.857142857142857  1.815269009670992
-0.857142857142857  -0.857142857142857  -0.285714285714286  1.815269009670992
-0.857142857142857  -0.571428571428571  -0.571428571428571  1.815269009670992
-0.857142857142857  -0.285714285714286  -0.857142857142857  1.815269009670992
-0.571428571428571  -0.857142857142857  -0.571428571428571  1.815269009670992
-0.571428571428571  -0.571428571428571  -0.857142857142857  1.815269009670992
-0.285714285714286  -0.857142857142857  -0.857142857142857  1.815269009670992
-0.857142857142857  -0.857142857142857  -0.571428571428571  1.815269009670992
-0.857142857142857  -0.571428571428571  -0.857142857142857  1.815269009670992
-0.571428571428571  -0.857142857142857  -0.857142857142857  1.815269009670992
-0.857142857142857  -0.857142857142857  -0.857142857142857  1.815269009670992
-0.833333333333333  -0.833333333333333  0.500000000000000  -0.958821749678893
-0.833333333333333  -0.500000000000000  0.166666666666667  -0.958821749678893
-0.833333333333333  -0.166666666666667  -0.166666666666667  -0.958821749678893
-0.833333333333333  0.166666666666667  -0.500000000000000  -0.958821749678893
-0.833333333333333  0.500000000000000  -0.833333333333333  -0.958821749678893
-0.500000000000000  -0.833333333333333  0.166666666666667  -0.958821749678893
-0.500000000000000  -0.500000000000000  -0.166666666666667  -0.958821749678893
-0.500000000000000  -0.166666666666667  -0.500000000000000  -0.958821749678893
-0.500000000000000  0.166666666666667  -0.833333333333333  -0.958821749678893
-0.166666666666667  -0.833333333333333  -0.166666666666667  -0.958821749678893
-0.166666666666667  -0.500000000000000  -0.500000000000000  -0.958821749678893
-0.166666666666667  -0.166666666666667  -0.833333333333333  -0.958821749678893
0.166666666666667  -0.833333333333333  -0.500000000000000  -0.958821749678893
0.166666666666667  -0.500000000000000  -0.833333333333333  -0.958821749678893
0.500000000000000  -0.833333333333333  -0.833333333333333  -0.958821749678893
-0.833333333333333  -0.833333333333333  0.166666666666667  -0.958821749678893
-0.833333333333333  -0.500000000000000  -0.166666666666667  -0.958821749678893
-0.833333333333333  -0.166666666666667  -0.500000000000000  -0.958821749678893
-0.833333333333333  0.166666666666667  -0.833333333333333  -0.958821749678893
-0.500000000000000  -0.833333333333333  -0.166666666666667  -0.958821749678893
-0.500000000000000  -0.500000000000000  -0.500000000000000  -0.958821749678893
-0.500000000000000  -0.166666666666667  -0.833333333333333  -0.958821749678893
-0.166666666666667  -0.833333333333333  -0.500000000000000  -0.958821749678893
-0.166666666666667  -0.500000000000000  -0.833333333333333  -0.958821749678893
0.166666666666667  -0.833333333333333  -0.833333333333333  -0.958821749678893
-0.833333333333333  -0.833333333333333  -0.166666666666667  -0.958821749678893
-0.833333333333333  -0.500000000000000  -0.500000000000000  -0.958821749678893
-0.833333333333333  -0.166666666666667  -0.833333333333333  -0.958821749678893
-0.500000000000000  -0.833333333333333  -0.500000000000000  -0.958821749678893
-0.500000000000000  -0.500000000000000  -0.833333333333333  -0.958821749678893
-0.166666666666667  -0.833333333333333  -0.833333333333333  -0.958821749678893
-0.833333333333333  -0.833333333333333  -0.500000000000000  -0.958821749678893
-0.833333333333333  -0.500000000000000  -0.833333333333333  -0.958821749678893
-0.500000000000000  -0.833333333333333  -0.833333333333333  -0.958821749678893
-0.833333333333333  -0.833333333333333  -0.833333333333333  -0.958821749678893
-0.800000000000000  -0.800000000000000  0.400000000000000  0.233372916620478
-0.800000000000000  -0.400000000000000  0.000000000000000  0.233372916620478
-0.800000000000000  0.000000000000000  -0.400000000000000  0.233372916620478
-0.800000000000000  0.400000000000000  -0.800000000000000  0.233372916620478
-0.400000000000000  -0.800000000000000  0.000000000000000  0.233372916620478
-0.400000000000000  -0.400000000000000  -0.400000000000000  0.233372916620478
-0.400000000000000  0.000000000000000  -0.800000000000000  0.233372916620478
0.000000000000000  -0.800000000000000  -0.400000000000000  0.233372916620478
0.000000000000000  -0.400000000000000  -0.800000000000000  0.233372916620478
0.400000000000000  -0.800000000000000  -0.800000000000000  0.233372916620478
-0.800000000000000  -0.800000000000000  0.000000000000000  0.233372916620478
-0.800000000000000  -0.400000000000000  -0.400000000000000  0.233372916620478
-0.800000000000000  0.000000000000000  -0.800000000000000  0.233372916620478
-0.400000000000000  -0.800000000000000  -0.400000000000000  0.233372916620478
-0.400000000000000  -0.400000000000000  -0.800000000000000  0.233372916620478
0.000000000000000  -0.800000000000000  -0.800000000000000  0.233372916620478
-0.800000000000000  -0.800000000000000  -0.400000000000000  0.233372916620478
-0.800000000000000  -0.400000000000000  -0.800000000000000  0.233372916620478
-0.400000000000000  -0.800000000000000  -0.800000000000000  0.233372916620478
-0.800000000000000  -0.800000000000000  -0.800000000000000  0.233372916620478
-0.750000000000000  -0.750000000000000  0.250000000000000  -0.022991022694726
-0.750000000000000  -0.250000000000000  -0.250000000000000  -0.022991022694726
-0.750000000000000  0.250000000000000  -0.750000000000000  -0.022991022694726
-0.250000000000000  -0.750000000000000  -0.250000000000000  -0.022991022694726
-0.250000000000000  -0.250000000000000  -0.750000000000000  -0.022991022694726
0.250000000000000  -0.750000000000000  -0.750000000000000  -0.022991022694726
-0.750000000000000  -0.750000000000000  -0.250000000000000  -0.022991022694726
-0.750000000000000  -0.250000000000000  -0.750000000000000  -0.022991022694726
-0.250000000000000  -0.750000000000000  -0.750000000000000  -0.022991022694726
-0.750000000000000  -0.750000000000000  -0.750000000000000  -0.022991022694726
-0.666666666666667  -0.666666666666667  0.000000000000000  0.000665685876623
-0.666666666666667  0.000000000000000  -0.666666666666667  0.000665685876623
0.000000000000000  -0.666666666666667  -0.666666666666667  0.000665685876623
-0.666666666666667  -0.666666666666667  -0.666666666666667  0.000665685876623
-0.500000000000000  -0.500000000000000  -0.500000000000000  -0.000002606055516
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=15

495
-0.900000000000000  -0.900000000000000  0.700000000000000  0.657650819729946
-0.900000000000000  -0.700000000000000  0.500000000000000  0.657650819729946
-0.900000000000000  -0.500000000000000  0.300000000000000  0.657650819729946
-0.900000000000000  -0.300000000000000  0.100000000000000  0.657650819729946
-0.900000000000000  -0.100000000000000  -0.100000000000000  0.657650819729946
-0.900000000000000  0.100000000000000  -0.300000000000000  0.657650819729946
-0.900000000000000  0.300000000000000  -0.500000000000000  0.657650819729946
-0.900000000000000  0.500000000000000  -0.700000000000000  0.657650819729946
-0.900000000000000  0.700000000000000  -0.900000000000000  0.657650819729946
-0.700000000000000  -0.900000000000000  0.500000000000000  0.657650819729946
-0.700000000000000  -0.700000000000000  0.300000000000000  0.657650819729946
-0.700000000000000  -0.500000000000000  0.100000000000000  0.657650819729946
-0.700000000000000  -0.300000000000000  -0.100000000000000  0.657650819729946
-0.700000000000000  -0.100000000000000  -0.300000000000000  0.657650819729946
-0.700000000000000  0.100000000000000  -0.500000000000000  0.657650819729946
-0.700000000000000  0.300000000000000  -0.700000000000000  0.657650819729946
-0.700000000000000  0.500000000000000  -0.900000000000000  0.657650819729946
-0.500000000000000  -0.900000000000000  0.300000000000000  0.657650819729946
-0.500000000000000  -0.700000000000000  0.100000000000000  0.657650819729946
-0.500000000000000  -0.500000000000000  -0.100000000000000  0.657650819729946
-0.500000000000000  -0.300000000000000  -0.300000000000000  0.657650819729946
-0.500000000000000  -0.100000000000000  -0.500000000000000  0.657650819729946
-0.500000000000000  0.100000000000000  -0.700000000000000  0.657650819729946
-0.500000000000000  0.300000000000000  -0.900000000000000  0.657650819729946
-0.300000000000000  -0.900000000000000  0.100000000000000  0.657650819729946
-0.300000000000000  -0.700000000000000  -0.100000000000000  0.657650819729946
-0.300000000000000  -0.500000000000000  -0.300000000000000  0.657650819729946
-0.300000000000000  -0.300000000000000  -0.500000000000000  0.657650819729946
-0.300000000000000  -0.100000000000000  -0.700000000000000  0.657650819729946
-0.300000000000000  0.100000000000000  -0.900000000000000  0.657650819729946
-0.100000000000000  -0.900000000000000  -0.100000000000000  0.657650819729946
-0.100000000000000  -0.700000000000000  -0.300000000000000  0.657650819729946
-0.100000000000000  -0.500000000000000  -0.500000000000000  0.657650819729946
-0.100000000000000  -0.300000000000000  -0.700000000000000  0.657650819729946
-0.100000000000000  -0.100000000000000  -0.900000000000000  0.657650819729946
0.100000000000000  -0.900000000000000  -0.300000000000000  0.657650819729946
0.100000000000000  -0.700000000000000  -0.500000000000000  0.657650819729946
0.100000000000000  -0.500000000000000  -0.700000000000000  0.657650819729946
0.100000000000000  -0.300000000000000  -0.900000000000000  0.657650819729946
0.300000000000000  -0.900000000000000  -0.500000000000000  0.657650819729946
0.300000000000000  -0.700000000000000  -0.700000000000000  0.657650819729946
0.300000000000000  -0.500000000000000  -0.900000000000000  0.657650819729946
0.500000000000000  -0.900000000000000  -0.700000000000000  0.657650819729946
0.500000000000000  -0.700000000000000  -0.900000000000000  0.657650819729946
0.700000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.900000000000000  -0.900000000000000  0.500000000000000  0.657650819729946
-0.900000000000000  -0.700000000000000  0.300000000000000  0.657650819729946
-0.900000000000000  -0.500000000000000  0.100000000000000  0.657650819729946
-0.900000000000000  -0.300000000000000  -0.100000000000000  0.657650819729946
-0.900000000000000  -0.100000000000000  -0.300000000000000  0.657650819729946
-0.900000000000000  0.100000000000000  -0.500000000000000  0.657650819729946
-0.900000000000000  0.300000000000000  -0.700000000000000  0.657650819729946
-0.900000000000000  0.500000000000000  -0.900000000000000  0.657650819729946
-0.700000000000000  -0.900000000000000  0.300000000000000  0.657650819729946
-0.700000000000000  -0.700000000000000  0.100000000000000  0.657650819729946
-0.700000000000000  -0.500000000000000  -0.100000000000000  0.657650819729946
-0.700000000000000  -0.300000000000000  -0.300000000000000  0.657650819729946
-0.700000000000000  -0.100000000000000  -0.500000000000000  0.657650819729946
-0.700000000000000  0.100000000000000  -0.700000000000000  0.657650819729946
-0.700000000000000  0.300000000000000  -0.900000000000000  0.657650819729946
-0.500000000000000  -0.900000000000000  0.100000000000000  0.657650819729946
-0.500000000000000  -0.700000000000000  -0.100000000000000  0.657650819729946
-0.500000000000000  -0.500000000000000  -0.300000000000000  0.657650819729946
-0.500000000000000  -0.300000000000000  -0.500000000000000  0.657650819729946
-0.500000000000000  -0.100000000000000  -0.700000000000000  0.657650819729946
-0.500000000000000  0.100000000000000  -0.900000000000000  0.657650819729946
-0.300000000000000  -0.900000000000000  -0.100000000000000  0.657650819729946
-0.300000000000000  -0.700000000000000  -0.300000000000000  0.657650819729946
-0.300000000000000  -0.500000000000000  -0.500000000000000  0.657650819729946
-0.300000000000000  -0.300000000000000  -0.700000000000000  0.657650819729946
-0.300000000000000  -0.100000000000000  -0.900000000000000  0.657650819729946
-0.100000000000000  -0.900000000000000  -0.300000000000000  0.657650819729946
-0.100000000000000  -0.700000000000000  -0.500000000000000  0.657650819729946
-0.100000000000000  -0.500000000000000  -0.700000000000000  0.657650819729946
-0.100000000000000  -0.300000000000000  -0.900000000000000  0.657650819729946
0.100000000000000  -0.900000000000000  -0.500000000000000  0.657650819729946
0.100000000000000  -0.700000000000000  -0.700000000000000  0.657650819729946
0.100000000000000  -0.500000000000000  -0.900000000000000  0.657650819729946
0.300000000000000  -0.900000000000000  -0.700000000000000  0.657650819729946
0.300000000000000  -0.700000000000000  -0.900000000000000  0.657650819729946
0.500000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.900000000000000  -0.900000000000000  0.300000000000000  0.657650819729946
-0.900000000000000  -0.700000000000000  0.100000000000000  0.657650819729946
-0.900000000000000  -0.500000000000000  -0.100000000000000  0.657650819729946
-0.900000000000000  -0.300000000000000  -0.300000000000000  0.657650819729946
-0.900000000000000  -0.100000000000000  -0.500000000000000  0.657650819729946
-0.900000000000000  0.100000000000000  -0.700000000000000  0.657650819729946
-0.900000000000000  0.300000000000000  -0.900000000000000  0.657650819729946
-0.700000000000000  -0.900000000000000  0.100000000000000  0.657650819729946
-0.700000000000000  -0.700000000000000  -0.100000000000000  0.657650819729946
-0.700000000000000  -0.500000000000000  -0.300000000000000  0.657650819729946
-0.700000000000000  -0.300000000000000  -0.500000000000000  0.657650819729946
-0.700000000000000  -0.100000000000000  -0.700000000000000  0.657650819729946
-0.700000000000000  0.100000000000000  -0.900000000000000  0.657650819729946
-0.500000000000000  -0.900000000000000  -0.100000000000000  0.657650819729946
-0.500000000000000  -0.700000000000000  -0.300000000000000  0.657650819729946
-0.500000000000000  -0.500000000000000  -0.500000000000000  0.657650819729946
-0.500000000000000  -0.300000000000000  -0.700000000000000  0.657650819729946
-0.500000000000000  -0.100000000000000  -0.900000000000000  0.657650819729946
-0.300000000000000  -0.900000000000000  -0.300000000000000  0.657650819729946
-0.300000000000000  -0.700000000000000  -0.500000000000000  0.657650819729946
-0.300000000000000  -0.500000000000000  -0.700000000000000  0.657650819729946
-0.300000000000000  -0.300000000000000  -0.900000000000000  0.657650819729946
-0.100000000000000  -0.900000000000000  -0.500000000000000  0.657650819729946
-0.100000000000000  -0.700000000000000  -0.700000000000000  0.657650819729946
-0.100000000000000  -0.500000000000000  -0.900000000000000  0.657650819729946
0.100000000000000  -0.900000000000000  -0.700000000000000  0.657650819729946
0.100000000000000  -0.700000000000000  -0.900000000000000  0.657650819729946
0.300000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.900000000000000  -0.900000000000000  0.100000000000000  0.657650819729946
-0.900000000000000  -0.700000000000000  -0.100000000000000  0.657650819729946
-0.900000000000000  -0.500000000000000  -0.300000000000000  0.657650819729946
-0.900000000000000  -0.300000000000000  -0.500000000000000  0.657650819729946
-0.900000000000000  -0.100000000000000  -0.700000000000000  0.657650819729946
-0.900000000000000  0.100000000000000  -0.900000000000000  0.657650819729946
-0.700000000000000  -0.900000000000000  -0.100000000000000  0.657650819729946
-0.700000000000000  -0.700000000000000  -0.300000000000000  0.657650819729946
-0.700000000000000  -0.500000000000000  -0.500000000000000  0.657650819729946
-0.700000000000000  -0.300000000000000  -0.700000000000000  0.657650819729946
-0.700000000000000  -0.100000000000000  -0.900000000000000  0.657650819729946
-0.500000000000000  -0.900000000000000  -0.300000000000000  0.657650819729946
-0.500000000000000  -0.700000000000000  -0.500000000000000  0.657650819729946
-0.500000000000000  -0.500000000000000  -0.700000000000000  0.657650819729946
-0.500000000000000  -0.300000000000000  -0.900000000000000  0.657650819729946
-0.300000000000000  -0.900000000000000  -0.500000000000000  0.657650819729946
-0.300000000000000  -0.700000000000000  -0.700000000000000  0.657650819729946
-0.300000000000000  -0.500000000000000  -0.900000000000000  0.657650819729946
-0.100000000000000  -0.900000000000000  -0.700000000000000  0.657650819729946
-0.100000000000000  -0.700000000000000  -0.900000000000000  0.657650819729946
0.100000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.900000000000000  -0.900000000000000  -0.100000000000000  0.657650819729946
-0.900000000000000  -0.700000000000000  -0.300000000000000  0.657650819729946
-0.900000000000000  -0.500000000000000  -0.500000000000000  0.657650819729946
-0.900000000000000  -0.300000000000000  -0.700000000000000  0.657650819729946
-0.900000000000000  -0.100000000000000  -0.900000000000000  0.657650819729946
-0.700000000000000  -0.900000000000000  -0.300000000000000  0.657650819729946
-0.700000000000000  -0.700000000000000  -0.500000000000000  0.657650819729946
-0.700000000000000  -0.500000000000000  -0.700000000000000  0.657650819729946
-0.700000000000000  -0.300000000000000  -0.900000000000000  0.657650819729946
-0.500000000000000  -0.900000000000000  -0.500000000000000  0.657650819729946
-0.500000000000000  -0.700000000000000  -0.700000000000000  0.657650819729946
-0.500000000000000  -0.500000000000000  -0.900000000000000  0.657650819729946
-0.300000000000000  -0.900000000000000  -0.700000000000000  0.657650819729946
-0.300000000000000  -0.700000000000000  -0.900000000000000  0.657650819729946
-0.100000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.900000000000000  -0.900000000000000  -0.300000000000000  0.657650819729946
-0.900000000000000  -0.700000000000000  -0.500000000000000  0.657650819729946
-0.900000000000000  -0.500000000000000  -0.700000000000000  0.657650819729946
-0.900000000000000  -0.300000000000000  -0.900000000000000  0.657650819729946
-0.700000000000000  -0.900000000000000  -0.500000000000000  0.657650819729946
-0.700000000000000  -0.700000000000000  -0.700000000000000  0.657650819729946
-0.700000000000000  -0.500000000000000  -0.900000000000000  0.657650819729946
-0.500000000000000  -0.900000000000000  -0.700000000000000  0.657650819729946
-0.500000000000000  -0.700000000000000  -0.900000000000000  0.657650819729946
-0.300000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.900000000000000  -0.900000000000000  -0.500000000000000  0.657650819729946
-0.900000000000000  -0.700000000000000  -0.700000000000000  0.657650819729946
-0.900000000000000  -0.500000000000000  -0.900000000000000  0.657650819729946
-0.700000000000000  -0.900000000000000  -0.700000000000000  0.657650819729946
-0.700000000000000  -0.700000000000000  -0.900000000000000  0.657650819729946
-0.500000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.900000000000000  -0.900000000000000  -0.700000000000000  0.657650819729946
-0.900000000000000  -0.700000000000000  -0.900000000000000  0.657650819729946
-0.700000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.900000000000000  -0.900000000000000  -0.900000000000000  0.657650819729946
-0.888888888888889  -0.888888888888889  0.666666666666667  -2.193552443114196
-0.888888888888889  -0.666666666666667  0.444444444444444  -2.193552443114196
-0.888888888888889  -0.444444444444444  0.222222222222222  -2.193552443114196
-0.888888888888889  -0.222222222222222  0.000000000000000  -2.193552443114196
-0.888888888888889  0.000000000000000  -0.222222222222222  -2.193552443114196
-0.888888888888889  0.222222222222222  -0.444444444444444  -2.193552443114196
-0.888888888888889  0.444444444444444  -0.666666666666667  -2.193552443114196
-0.888888888888889  0.666666666666667  -0.888888888888889  -2.193552443114196
-0.666666666666667  -0.888888888888889  0.444444444444444  -2.193552443114196
-0.666666666666667  -0.666666666666667  0.222222222222222  -2.193552443114196
-0.666666666666667  -0.444444444444444  0.000000000000000  -2.193552443114196
-0.666666666666667  -0.222222222222222  -0.222222222222222  -2.193552443114196
-0.666666666666667  0.000000000000000  -0.444444444444444  -2.193552443114196
-0.666666666666667  0.222222222222222  -0.666666666666667  -2.193552443114196
-0.666666666666667  0.444444444444444  -0.888888888888889  -2.193552443114196
-0.444444444444444  -0.888888888888889  0.222222222222222  -2.193552443114196
-0.444444444444444  -0.666666666666667  0.000000000000000  -2.193552443114196
-0.444444444444444  -0.444444444444444  -0.222222222222222  -2.193552443114196
-0.444444444444444  -0.222222222222222  -0.444444444444444  -2.193552443114196
-0.444444444444444  0.000000000000000  -0.666666666666667  -2.193552443114196
-0.444444444444444  0.222222222222222  -0.888888888888889  -2.193552443114196
-0.222222222222222  -0.888888888888889  0.000000000000000  -2.193552443114196
-0.222222222222222  -0.666666666666667  -0.222222222222222  -2.193552443114196
-0.222222222222222  -0.444444444444444  -0.444444444444444  -2.193552443114196
-0.222222222222222  -0.222222222222222  -0.666666666666667  -2.193552443114196
-0.222222222222222  0.000000000000000  -0.888888888888889  -2.193552443114196
0.000000000000000  -0.888888888888889  -0.222222222222222  -2.193552443114196
0.000000000000000  -0.666666666666667  -0.444444444444444  -2.193552443114196
0.000000000000000  -0.444444444444444  -0.666666666666667  -2.193552443114196
0.000000000000000  -0.222222222222222  -0.888888888888889  -2.193552443114196
0.222222222222222  -0.888888888888889  -0.444444444444444  -2.193552443114196
0.222222222222222  -0.666666666666667  -0.666666666666667  -2.193552443114196
0.222222222222222  -0.444444444444444  -0.888888888888889  -2.193552443114196
0.444444444444444  -0.888888888888889  -0.666666666666667  -2.193552443114196
0.444444444444444  -0.666666666666667  -0.888888888888889  -2.193552443114196
0.666666666666667  -0.888888888888889  -0.888888888888889  -2.193552443114196
-0.888888888888889  -0.888888888888889  0.444444444444444  -2.193552443114196
-0.888888888888889  -0.666666666666667  0.222222222222222  -2.193552443114196
-0.888888888888889  -0.444444444444444  0.000000000000000  -2.193552443114196
-0.888888888888889  -0.222222222222222  -0.222222222222222  -2.193552443114196
-0.888888888888889  0.000000000000000  -0.444444444444444  -2.193552443114196
-0.888888888888889  0.222222222222222  -0.666666666666667  -2.193552443114196
-0.888888888888889  0.444444444444444  -0.888888888888889  -2.193552443114196
-0.666666666666667  -0.888888888888889  0.222222222222222  -2.193552443114196
-0.666666666666667  -0.666666666666667  0.000000000000000  -2.193552443114196
-0.666666666666667  -0.444444444444444  -0.222222222222222  -2.193552443114196
-0.666666666666667  -0.222222222222222  -0.444444444444444  -2.193552443114196
-0.666666666666667  0.000000000000000  -0.666666666666667  -2.193552443114196
-0.666666666666667  0.222222222222222  -0.888888888888889  -2.193552443114196
-0.444444444444444  -0.888888888888889  0.000000000000000  -2.193552443114196
-0.444444444444444  -0.666666666666667  -0.222222222222222  -2.193552443114196
-0.444444444444444  -0.444444444444444  -0.444444444444444  -2.193552443114196
-0.444444444444444  -0.222222222222222  -0.666666666666667  -2.193552443114196
-0.444444444444444  0.000000000000000  -0.888888888888889  -2.193552443114196
-0.222222222222222  -0.888888888888889  -0.222222222222222  -2.193552443114196
-0.222222222222222  -0.666666666666667  -0.444444444444444  -2.193552443114196
-0.222222222222222  -0.444444444444444  -0.666666666666667  -2.193552443114196
-0.222222222222222  -0.222222222222222  -0.888888888888889  -2.193552443114196
0.000000000000000  -0.888888888888889  -0.444444444444444  -2.193552443114196
0.000000000000000  -0.666666666666667  -0.666666666666667  -2.193552443114196
0.000000000000000  -0.444444444444444  -0.888888888888889  -2.193552443114196
0.222222222222222  -0.888888888888889  -0.666666666666667  -2.193552443114196
0.222222222222222  -0.666666666666667  -0.888888888888889  -2.193552443114196
0.444444444444444  -0.888888888888889  -0.888888888888889  -2.193552443114196
-0.888888888888889  -0.888888888888889  0.222222222222222  -2.193552443114196
-0.888888888888889  -0.666666666666667  0.000000000000000  -2.193552443114196
-0.888888888888889  -0.444444444444444  -0.222222222222222  -2.193552443114196
-0.888888888888889  -0.222222222222222  -0.444444444444444  -2.193552443114196
-0.888888888888889  0.000000000000000  -0.666666666666667  -2.193552443114196
-0.888888888888889  0.222222222222222  -0.888888888888889  -2.193552443114196
-0.666666666666667  -0.888888888888889  0.000000000000000  -2.193552443114196
-0.666666666666667  -0.666666666666667  -0.222222222222222  -2.193552443114196
-0.666666666666667  -0.444444444444444  -0.444444444444444  -2.193552443114196
-0.666666666666667  -0.222222222222222  -0.666666666666667  -2.193552443114196
-0.666666666666667  0.000000000000000  -0.888888888888889  -2.193552443114196
-0.444444444444444  -0.888888888888889  -0.222222222222222  -2.193552443114196
-0.444444444444444  -0.666666666666667  -0.444444444444444  -2.193552443114196
-0.444444444444444  -0.444444444444444  -0.666666666666667  -2.193552443114196
-0.444444444444444  -0.222222222222222  -0.888888888888889  -2.193552443114196
-0.222222222222222  -0.888888888888889  -0.444444444444444  -2.193552443114196
-0.222222222222222  -0.666666666666667  -0.666666666666667  -2.193552443114196
-0.222222222222222  -0.444444444444444  -0.888888888888889  -2.193552443114196
0.000000000000000  -0.888888888888889  -0.666666666666667  -2.193552443114196
0.000000000000000  -0.666666666666667  -0.888888888888889  -2.193552443114196
0.222222222222222  -0.888888888888889  -0.888888888888889  -2.193552443114196
-0.888888888888889  -0.888888888888889  0.000000000000000  -2.193552443114196
-0.888888888888889  -0.666666666666667  -0.222222222222222  -2.193552443114196
-0.888888888888889  -0.444444444444444  -0.444444444444444  -2.193552443114196
-0.888888888888889  -0.222222222222222  -0.666666666666667  -2.193552443114196
-0.888888888888889  0.000000000000000  -0.888888888888889  -2.193552443114196
-0.666666666666667  -0.888888888888889  -0.222222222222222  -2.193552443114196
-0.666666666666667  -0.666666666666667  -0.444444444444444  -2.193552443114196
-0.666666666666667  -0.444444444444444  -0.666666666666667  -2.193552443114196
-0.666666666666667  -0.222222222222222  -0.888888888888889  -2.193552443114196
-0.444444444444444  -0.888888888888889  -0.444444444444444  -2.193552443114196
-0.444444444444444  -0.666666666666667  -0.666666666666667  -2.193552443114196
-0.444444444444444  -0.444444444444444  -0.888888888888889  -2.193552443114196
-0.222222222222222  -0.888888888888889  -0.666666666666667  -2.193552443114196
-0.222222222222222  -0.666666666666667  -0.888888888888889  -2.193552443114196
0.000000000000000  -0.888888888888889  -0.888888888888889  -2.193552443114196
-0.888888888888889  -0.888888888888889  -0.222222222222222  -2.193552443114196
-0.888888888888889  -0.666666666666667  -0.444444444444444  -2.193552443114196
-0.888888888888889  -0.444444444444444  -0.666666666666667  -2.193552443114196
-0.888888888888889  -0.222222222222222  -0.888888888888889  -2.193552443114196
-0.666666666666667  -0.888888888888889  -0.444444444444444  -2.193552443114196
-0.666666666666667  -0.666666666666667  -0.666666666666667  -2.193552443114196
-0.666666666666667  -0.444444444444444  -0.888888888888889  -2.193552443114196
-0.444444444444444  -0.888888888888889  -0.666666666666667  -2.193552443114196
-0.444444444444444  -0.666666666666667  -0.888888888888889  -2.193552443114196
-0.222222222222222  -0.888888888888889  -0.888888888888889  -2.193552443114196
-0.888888888888889  -0.888888888888889  -0.444444444444444  -2.193552443114196
-0.888888888888889  -0.666666666666667  -0.666666666666667  -2.193552443114196
-0.888888888888889  -0.444444444444444  -0.888888888888889  -2.193552443114196
-0.666666666666667  -0.888888888888889  -0.666666666666667  -2.193552443114196
-0.666666666666667  -0.666666666666667  -0.888888888888889  -2.193552443114196
-0.444444444444444  -0.888888888888889  -0.888888888888889  -2.193552443114196
-0.888888888888889  -0.888888888888889  -0.666666666666667  -2.193552443114196
-0.888888888888889  -0.666666666666667  -0.888888888888889  -2.193552443114196
-0.666666666666667  -0.888888888888889  -0.888888888888889  -2.193552443114196
-0.888888888888889  -0.888888888888889  -0.888888888888889  -2.193552443114196
-0.875000000000000  -0.875000000000000  0.625000000000000  2.813706187341903
-0.875000000000000  -0.625000000000000  0.375000000000000  2.813706187341903
-0.875000000000000  -0.375000000000000  0.125000000000000  2.813706187341903
-0.875000000000000  -0.125000000000000  -0.125000000000000  2.813706187341903
-0.875000000000000  0.125000000000000  -0.375000000000000  2.813706187341903
-0.875000000000000  0.375000000000000  -0.625000000000000  2.813706187341903
-0.875000000000000  0.625000000000000  -0.875000000000000  2.813706187341903
-0.625000000000000  -0.875000000000000  0.375000000000000  2.813706187341903
-0.625000000000000  -0.625000000000000  0.125000000000000  2.813706187341903
-0.625000000000000  -0.375000000000000  -0.125000000000000  2.813706187341903
-0.625000000000000  -0.125000000000000  -0.375000000000000  2.813706187341903
-0.625000000000000  0.125000000000000  -0.625000000000000  2.813706187341903
-0.625000000000000  0.375000000000000  -0.875000000000000  2.813706187341903
-0.375000000000000  -0.875000000000000  0.125000000000000  2.813706187341903
-0.375000000000000  -0.625000000000000  -0.125000000000000  2.813706187341903
-0.375000000000000  -0.375000000000000  -0.375000000000000  2.813706187341903
-0.375000000000000  -0.125000000000000  -0.625000000000000  2.813706187341903
-0.375000000000000  0.125000000000000  -0.875000000000000  2.813706187341903
-0.125000000000000  -0.875000000000000  -0.125000000000000  2.813706187341903
-0.125000000000000  -0.625000000000000  -0.375000000000000  2.813706187341903
-0.125000000000000  -0.375000000000000  -0.625000000000000  2.813706187341903
-0.125000000000000  -0.125000000000000  -0.875000000000000  2.813706187341903
0.125000000000000  -0.875000000000000  -0.375000000000000  2.813706187341903
0.125000000000000  -0.625000000000000  -0.625000000000000  2.813706187341903
0.125000000000000  -0.375000000000000  -0.875000000000000  2.813706187341903
0.375000000000000  -0.875000000000000  -0.625000000000000  2.813706187341903
0.375000000000000  -0.625000000000000  -0.875000000000000  2.813706187341903
0.625000000000000  -0.875000000000000  -0.875000000000000  2.813706187341903
-0.875000000000000  -0.875000000000000  0.375000000000000  2.813706187341903
-0.875000000000000  -0.625000000000000  0.125000000000000  2.813706187341903
-0.875000000000000  -0.375000000000000  -0.125000000000000  2.813706187341903
-0.875000000000000  -0.125000000000000  -0.375000000000000  2.813706187341903
-0.875000000000000  0.125000000000000  -0.625000000000000  2.813706187341903
-0.875000000000000  0.375000000000000  -0.875000000000000  2.813706187341903
-0.625000000000000  -0.875000000000000  0.125000000000000  2.813706187341903
-0.625000000000000  -0.625000000000000  -0.125000000000000  2.813706187341903
-0.625000000000000  -0.375000000000000  -0.375000000000000  2.813706187341903
-0.625000000000000  -0.125000000000000  -0.625000000000000  2.813706187341903
-0.625000000000000  0.125000000000000  -0.875000000000000  2.813706187341903
-0.375000000000000  -0.875000000000000  -0.125000000000000  2.813706187341903
-0.375000000000000  -0.625000000000000  -0.375000000000000  2.813706187341903
-0.375000000000000  -0.375000000000000  -0.625000000000000  2.813706187341903
-0.375000000000000  -0.125000000000000  -0.875000000000000  2.813706187341903
-0.125000000000000  -0.875000000000000  -0.375000000000000  2.813706187341903
-0.125000000000000  -0.625000000000000  -0.625000000000000  2.813706187341903
-0.125000000000000  -0.375000000000000  -0.875000000000000  2.813706187341903
0.125000000000000  -0.875000000000000  -0.625000000000000  2.813706187341903
0.125000000000000  -0.625000000000000  -0.875000000000000  2.813706187341903
0.375000000000000  -0.875000000000000  -0.875000000000000  2.813706187341903
-0.875000000000000  -0.875000000000000  0.125000000000000  2.813706187341903
-0.875000000000000  -0.625000000000000  -0.125000000000000  2.813706187341903
-0.875000000000000  -0.375000000000000  -0.375000000000000  2.813706187341903
-0.875000000000000  -0.125000000000000  -0.625000000000000  2.813706187341903
-0.875000000000000  0.125000000000000  -0.875000000000000  2.813706187341903
-0.625000000000000  -0.875000000000000  -0.125000000000000  2.813706187341903
-0.625000000000000  -0.625000000000000  -0.375000000000000  2.813706187341903
-0.625000000000000  -0.375000000000000  -0.625000000000000  2.813706187341903
-0.625000000000000  -0.125000000000000  -0.875000000000000  2.813706187341903
-0.375000000000000  -0.875000000000000  -0.375000000000000  2.813706187341903
-0.375000000000000  -0.625000000000000  -0.625000000000000  2.813706187341903
-0.375000000000000  -0.375000000000000  -0.875000000000000  2.813706187341903
-0.125000000000000  -0.875000000000000  -0.625000000000000  2.813706187341903
-0.125000000000000  -0.625000000000000  -0.875000000000000  2.813706187341903
0.125000000000000  -0.875000000000000  -0.875000000000000  2.813706187341903
-0.875000000000000  -0.875000000000000  -0.125000000000000  2.813706187341903
-0.875000000000000  -0.625000000000000  -0.375000000000000  2.813706187341903
-0.875000000000000  -0.375000000000000  -0.625000000000000  2.813706187341903
-0.875000000000000  -0.125000000000000  -0.875000000000000  2.813706187341903
-0.625000000000000  -0.875000000000000  -0.375000000000000  2.813706187341903
-0.625000000000000  -0.625000000000000  -0.625000000000000  2.813706187341903
-0.625000000000000  -0.375000000000000  -0.875000000000000  2.813706187341903
-0.375000000000000  -0.875000000000000  -0.625000000000000  2.813706187341903
-0.375000000000000  -0.625000000000000  -0.875000000000000  2.813706187341903
-0.125000000000000  -0.875000000000000  -0.875000000000000  2.813706187341903
-0.875000000000000  -0.875000000000000  -0.375000000000000  2.813706187341903
-0.875000000000000  -0.625000000000000  -0.625000000000000  2.813706187341903
-0.875000000000000  -0.375000000000000  -0.875000000000000  2.813706187341903
-0.625000000000000  -0.875000000000000  -0.625000000000000  2.813706187341903
-0.625000000000000  -0.625000000000000  -0.875000000000000  2.813706187341903
-0.375000000000000  -0.875000000000000  -0.875000000000000  2.813706187341903
-0.875000000000000  -0.875000000000000  -0.625000000000000  2.813706187341903
-0.875000000000000  -0.625000000000000  -0.875000000000000  2.813706187341903
-0.625000000000000  -0.875000000000000  -0.875000000000000  2.813706187341903
-0.875000000000000  -0.875000000000000  -0.875000000000000  2.813706187341903
-0.857142857142857  -0.857142857142857  0.571428571428571  -1.744081989683894
-0.857142857142857  -0.571428571428571  0.285714285714286  -1.744081989683894
-0.857142857142857  -0.285714285714286  0.000000000000000  -1.744081989683894
-0.857142857142857  0.000000000000000  -0.285714285714286  -1.744081989683894
-0.857142857142857  0.285714285714286  -0.571428571428571  -1.744081989683894
-0.857142857142857  0.571428571428571  -0.857142857142857  -1.744081989683894
-0.571428571428571  -0.857142857142857  0.285714285714286  -1.744081989683894
-0.571428571428571  -0.571428571428571  0.000000000000000  -1.744081989683894
-0.571428571428571  -0.285714285714286  -0.285714285714286  -1.744081989683894
-0.571428571428571  0.000000000000000  -0.571428571428571  -1.744081989683894
-0.571428571428571  0.285714285714286  -0.857142857142857  -1.744081989683894
-0.285714285714286  -0.857142857142857  0.000000000000000  -1.744081989683894
-0.285714285714286  -0.571428571428571  -0.285714285714286  -1.744081989683894
-0.285714285714286  -0.285714285714286  -0.571428571428571  -1.744081989683894
-0.285714285714286  0.000000000000000  -0.857142857142857  -1.744081989683894
0.000000000000000  -0.857142857142857  -0.285714285714286  -1.744081989683894
0.000000000000000  -0.571428571428571  -0.571428571428571  -1.744081989683894
0.000000000000000  -0.285714285714286  -0.857142857142857  -1.744081989683894
0.285714285714286  -0.857142857142857  -0.571428571428571  -1.744081989683894
0.285714285714286  -0.571428571428571  -0.857142857142857  -1.744081989683894
0.571428571428571  -0.857142857142857  -0.857142857142857  -1.744081989683894
-0.857142857142857  -0.857142857142857  0.285714285714286  -1.744081989683894
-0.857142857142857  -0.571428571428571  0.000000000000000  -1.744081989683894
-0.857142857142857  -0.285714285714286  -0.285714285714286  -1.744081989683894
-0.857142857142857  0.000000000000000  -0.571428571428571  -1.744081989683894
-0.857142857142857  0.285714285714286  -0.857142857142857  -1.744081989683894
-0.571428571428571  -0.857142857142857  0.000000000000000  -1.744081989683894
-0.571428571428571  -0.571428571428571  -0.285714285714286  -1.744081989683894
-0.571428571428571  -0.285714285714286  -0.571428571428571  -1.744081989683894
-0.571428571428571  0.000000000000000  -0.857142857142857  -1.744081989683894
-0.285714285714286  -0.857142857142857  -0.285714285714286  -1.744081989683894
-0.285714285714286  -0.571428571428571  -0.571428571428571  -1.744081989683894
-0.285714285714286  -0.285714285714286  -0.857142857142857  -1.744081989683894
0.000000000000000  -0.857142857142857  -0.571428571428571  -1.744081989683894
0.000000000000000  -0.571428571428571  -0.857142857142857  -1.744081989683894
0.285714285714286  -0.857142857142857  -0.857142857142857  -1.744081989683894
-0.857142857142857  -0.857142857142857  0.000000000000000  -1.744081989683894
-0.857142857142857  -0.571428571428571  -0.285714285714286  -1.744081989683894
-0.857142857142857  -0.285714285714286  -0.571428571428571  -1.744081989683894
-0.857142857142857  0.000000000000000  -0.857142857142857  -1.744081989683894
-0.571428571428571  -0.857142857142857  -0.285714285714286  -1.744081989683894
-0.571428571428571  -0.571428571428571  -0.571428571428571  -1.744081989683894
-0.571428571428571  -0.285714285714286  -0.857142857142857  -1.744081989683894
-0.285714285714286  -0.857142857142857  -0.571428571428571  -1.744081989683894
-0.285714285714286  -0.571428571428571  -0.857142857142857  -1.744081989683894
0.000000000000000  -0.857142857142857  -0.857142857142857  -1.744081989683894
-0.857142857142857  -0.857142857142857  -0.285714285714286  -1.744081989683894
-0.857142857142857  -0.571428571428571  -0.571428571428571  -1.744081989683894
-0.857142857142857  -0.285714285714286  -0.857142857142857  -1.744081989683894
-0.571428571428571  -0.857142857142857  -0.571428571428571  -1.744081989683894
-0.571428571428571  -0.571428571428571  -0.857142857142857  -1.744081989683894
-0.285714285714286  -0.857142857142857  -0.857142857142857  -1.744081989683894
-0.857142857142857  -0.857142857142857  -0.571428571428571  -1.744081989683894
-0.857142857142857  -0.571428571428571  -0.857142857142857  -1.744081989683894
-0.571428571428571  -0.857142857142857  -0.857142857142857  -1.744081989683894
-0.857142857142857  -0.857142857142857  -0.857142857142857  -1.744081989683894
-0.833333333333333  -0.833333333333333  0.500000000000000  0.539337234194377
-0.833333333333333  -0.500000000000000  0.166666666666667  0.539337234194377
-0.833333333333333  -0.166666666666667  -0.166666666666667  0.539337234194377
-0.833333333333333  0.166666666666667  -0.500000000000000  0.539337234194377
-0.833333333333333  0.500000000000000  -0.833333333333333  0.539337234194377
-0.500000000000000  -0.833333333333333  0.166666666666667  0.539337234194377
-0.500000000000000  -0.500000000000000  -0.166666666666667  0.539337234194377
-0.500000000000000  -0.166666666666667  -0.500000000000000  0.539337234194377
-0.500000000000000  0.166666666666667  -0.833333333333333  0.539337234194377
-0.166666666666667  -0.833333333333333  -0.166666666666667  0.539337234194377
-0.166666666666667  -0.500000000000000  -0.500000000000000  0.539337234194377
-0.166666666666667  -0.166666666666667  -0.833333333333333  0.539337234194377
0.166666666666667  -0.833333333333333  -0.500000000000000  0.539337234194377
0.166666666666667  -0.500000000000000  -0.833333333333333  0.539337234194377
0.500000000000000  -0.833333333333333  -0.833333333333333  0.539337234194377
-0.833333333333333  -0.833333333333333  0.166666666666667  0.539337234194377
-0.833333333333333  -0.500000000000000  -0.166666666666667  0.539337234194377
-0.833333333333333  -0.166666666666667  -0.500000000000000  0.539337234194377
-0.833333333333333  0.166666666666667  -0.833333333333333  0.539337234194377
-0.500000000000000  -0.833333333333333  -0.166666666666667  0.539337234194377
-0.500000000000000  -0.500000000000000  -0.500000000000000  0.539337234194377
-0.500000000000000  -0.166666666666667  -0.833333333333333  0.539337234194377
-0.166666666666667  -0.833333333333333  -0.500000000000000  0.539337234194377
-0.166666666666667  -0.500000000000000  -0.833333333333333  0.539337234194377
0.166666666666667  -0.833333333333333  -0.833333333333333  0.539337234194377
-0.833333333333333  -0.833333333333333  -0.166666666666667  0.539337234194377
-0.833333333333333  -0.500000000000000  -0.500000000000000  0.539337234194377
-0.833333333333333  -0.166666666666667  -0.833333333333333  0.539337234194377
-0.500000000000000  -0.833333333333333  -0.500000000000000  0.539337234194377
-0.500000000000000  -0.500000000000000  -0.833333333333333  0.539337234194377
-0.166666666666667  -0.833333333333333  -0.833333333333333  0.539337234194377
-0.833333333333333  -0.833333333333333  -0.500000000000000  0.539337234194377
-0.833333333333333  -0.500000000000000  -0.833333333333333  0.539337234194377
-0.500000000000000  -0.833333333333333  -0.833333333333333  0.539337234194377
-0.833333333333333  -0.833333333333333  -0.833333333333333  0.539337234194377
-0.800000000000000  -0.800000000000000  0.400000000000000  -0.077790972206826
-0.800000000000000  -0.400000000000000  0.000000000000000  -0.077790972206826
-0.800000000000000  0.000000000000000  -0.400000000000000  -0.077790972206826
-0.800000000000000  0.400000000000000  -0.800000000000000  -0.077790972206826
-0.400000000000000  -0.800000000000000  0.000000000000000  -0.077790972206826
-0.400000000000000  -0.400000000000000  -0.400000000000000  -0.077790972206826
-0.400000000000000  0.000000000000000  -0.800000000000000  -0.077790972206826
0.000000000000000  -0.800000000000000  -0.400000000000000  -0.077790972206826
0.000000000000000  -0.400000000000000  -0.800000000000000  -0.077790972206826
0.400000000000000  -0.800000000000000  -0.800000000000000  -0.077790972206826
-0.800000000000000  -0.800000000000000  0.000000000000000  -0.077790972206826
-0.800000000000000  -0.400000000000000  -0.400000000000000  -0.077790972206826
-0.800000000000000  0.000000000000000  -0.800000000000000  -0.077790972206826
-0.400000000000000  -0.800000000000000  -0.400000000000000  -0.077790972206826
-0.400000000000000  -0.400000000000000  -0.800000000000000  -0.077790972206826
0.000000000000000  -0.800000000000000  -0.800000000000000  -0.077790972206826
-0.800000000000000  -0.800000000000000  -0.400000000000000  -0.077790972206826
-0.800000000000000  -0.400000000000000  -0.800000000000000  -0.077790972206826
-0.400000000000000  -0.800000000000000  -0.800000000000000  -0.077790972206826
-0.800000000000000  -0.800000000000000  -0.800000000000000  -0.077790972206826
-0.750000000000000  -0.750000000000000  0.250000000000000  0.004379242418043
-0.750000000000000  -0.250000000000000  -0.250000000000000  0.004379242418043
-0.750000000000000  0.250000000000000  -0.750000000000000  0.004379242418043
-0.250000000000000  -0.750000000000000  -0.250000000000000  0.004379242418043
-0.250000000000000  -0.250000000000000  -0.750000000000000  0.004379242418043
0.250000000000000  -0.750000000000000  -0.750000000000000  0.004379242418043
-0.750000000000000  -0.750000000000000  -0.250000000000000  0.004379242418043
-0.750000000000000  -0.250000000000000  -0.750000000000000  0.004379242418043
-0.250000000000000  -0.750000000000000  -0.750000000000000  0.004379242418043
-0.750000000000000  -0.750000000000000  -0.750000000000000  0.004379242418043
-0.666666666666667  -0.666666666666667  0.000000000000000  -0.000065837064721
-0.666666666666667  0.000000000000000  -0.666666666666667  -0.000065837064721
0.000000000000000  -0.666666666666667  -0.666666666666667  -0.000065837064721
-0.666666666666667  -0.666666666666667  -0.666666666666667  -0.000065837064721
-0.500000000000000  -0.500000000000000  -0.500000000000000   0.000000108585647
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=17

715
-0.909090909090909  -0.909090909090909  0.727272727272727  0.870591471154497
-0.909090909090909  -0.727272727272727  0.545454545454545  0.870591471154497
-0.909090909090909  -0.545454545454545  0.363636363636364  0.870591471154497
-0.909090909090909  -0.363636363636364  0.181818181818182  0.870591471154497
-0.909090909090909  -0.181818181818182  0.000000000000000  0.870591471154497
-0.909090909090909  0.000000000000000  -0.181818181818182  0.870591471154497
-0.909090909090909  0.181818181818182  -0.363636363636364  0.870591471154497
-0.909090909090909  0.363636363636364  -0.545454545454545  0.870591471154497
-0.909090909090909  0.545454545454545  -0.727272727272727  0.870591471154497
-0.909090909090909  0.727272727272727  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  0.545454545454545  0.870591471154497
-0.727272727272727  -0.727272727272727  0.363636363636364  0.870591471154497
-0.727272727272727  -0.545454545454545  0.181818181818182  0.870591471154497
-0.727272727272727  -0.363636363636364  0.000000000000000  0.870591471154497
-0.727272727272727  -0.181818181818182  -0.181818181818182  0.870591471154497
-0.727272727272727  0.000000000000000  -0.363636363636364  0.870591471154497
-0.727272727272727  0.181818181818182  -0.545454545454545  0.870591471154497
-0.727272727272727  0.363636363636364  -0.727272727272727  0.870591471154497
-0.727272727272727  0.545454545454545  -0.909090909090909  0.870591471154497
-0.545454545454545  -0.909090909090909  0.363636363636364  0.870591471154497
-0.545454545454545  -0.727272727272727  0.181818181818182  0.870591471154497
-0.545454545454545  -0.545454545454545  0.000000000000000  0.870591471154497
-0.545454545454545  -0.363636363636364  -0.181818181818182  0.870591471154497
-0.545454545454545  -0.181818181818182  -0.363636363636364  0.870591471154497
-0.545454545454545  0.000000000000000  -0.545454545454545  0.870591471154497
-0.545454545454545  0.181818181818182  -0.727272727272727  0.870591471154497
-0.545454545454545  0.363636363636364  -0.909090909090909  0.870591471154497
-0.363636363636364  -0.909090909090909  0.181818181818182  0.870591471154497
-0.363636363636364  -0.727272727272727  0.000000000000000  0.870591471154497
-0.363636363636364  -0.545454545454545  -0.181818181818182  0.870591471154497
-0.363636363636364  -0.363636363636364  -0.363636363636364  0.870591471154497
-0.363636363636364  -0.181818181818182  -0.545454545454545  0.870591471154497
-0.363636363636364  0.000000000000000  -0.727272727272727  0.870591471154497
-0.363636363636364  0.181818181818182  -0.909090909090909  0.870591471154497
-0.181818181818182  -0.909090909090909  0.000000000000000  0.870591471154497
-0.181818181818182  -0.727272727272727  -0.181818181818182  0.870591471154497
-0.181818181818182  -0.545454545454545  -0.363636363636364  0.870591471154497
-0.181818181818182  -0.363636363636364  -0.545454545454545  0.870591471154497
-0.181818181818182  -0.181818181818182  -0.727272727272727  0.870591471154497
-0.181818181818182  0.000000000000000  -0.909090909090909  0.870591471154497
0.000000000000000  -0.909090909090909  -0.181818181818182  0.870591471154497
0.000000000000000  -0.727272727272727  -0.363636363636364  0.870591471154497
0.000000000000000  -0.545454545454545  -0.545454545454545  0.870591471154497
0.000000000000000  -0.363636363636364  -0.727272727272727  0.870591471154497
0.000000000000000  -0.181818181818182  -0.909090909090909  0.870591471154497
0.181818181818182  -0.909090909090909  -0.363636363636364  0.870591471154497
0.181818181818182  -0.727272727272727  -0.545454545454545  0.870591471154497
0.181818181818182  -0.545454545454545  -0.727272727272727  0.870591471154497
0.181818181818182  -0.363636363636364  -0.909090909090909  0.870591471154497
0.363636363636364  -0.909090909090909  -0.545454545454545  0.870591471154497
0.363636363636364  -0.727272727272727  -0.727272727272727  0.870591471154497
0.363636363636364  -0.545454545454545  -0.909090909090909  0.870591471154497
0.545454545454545  -0.909090909090909  -0.727272727272727  0.870591471154497
0.545454545454545  -0.727272727272727  -0.909090909090909  0.870591471154497
0.727272727272727  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  0.545454545454545  0.870591471154497
-0.909090909090909  -0.727272727272727  0.363636363636364  0.870591471154497
-0.909090909090909  -0.545454545454545  0.181818181818182  0.870591471154497
-0.909090909090909  -0.363636363636364  0.000000000000000  0.870591471154497
-0.909090909090909  -0.181818181818182  -0.181818181818182  0.870591471154497
-0.909090909090909  0.000000000000000  -0.363636363636364  0.870591471154497
-0.909090909090909  0.181818181818182  -0.545454545454545  0.870591471154497
-0.909090909090909  0.363636363636364  -0.727272727272727  0.870591471154497
-0.909090909090909  0.545454545454545  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  0.363636363636364  0.870591471154497
-0.727272727272727  -0.727272727272727  0.181818181818182  0.870591471154497
-0.727272727272727  -0.545454545454545  0.000000000000000  0.870591471154497
-0.727272727272727  -0.363636363636364  -0.181818181818182  0.870591471154497
-0.727272727272727  -0.181818181818182  -0.363636363636364  0.870591471154497
-0.727272727272727  0.000000000000000  -0.545454545454545  0.870591471154497
-0.727272727272727  0.181818181818182  -0.727272727272727  0.870591471154497
-0.727272727272727  0.363636363636364  -0.909090909090909  0.870591471154497
-0.545454545454545  -0.909090909090909  0.181818181818182  0.870591471154497
-0.545454545454545  -0.727272727272727  0.000000000000000  0.870591471154497
-0.545454545454545  -0.545454545454545  -0.181818181818182  0.870591471154497
-0.545454545454545  -0.363636363636364  -0.363636363636364  0.870591471154497
-0.545454545454545  -0.181818181818182  -0.545454545454545  0.870591471154497
-0.545454545454545  0.000000000000000  -0.727272727272727  0.870591471154497
-0.545454545454545  0.181818181818182  -0.909090909090909  0.870591471154497
-0.363636363636364  -0.909090909090909  0.000000000000000  0.870591471154497
-0.363636363636364  -0.727272727272727  -0.181818181818182  0.870591471154497
-0.363636363636364  -0.545454545454545  -0.363636363636364  0.870591471154497
-0.363636363636364  -0.363636363636364  -0.545454545454545  0.870591471154497
-0.363636363636364  -0.181818181818182  -0.727272727272727  0.870591471154497
-0.363636363636364  0.000000000000000  -0.909090909090909  0.870591471154497
-0.181818181818182  -0.909090909090909  -0.181818181818182  0.870591471154497
-0.181818181818182  -0.727272727272727  -0.363636363636364  0.870591471154497
-0.181818181818182  -0.545454545454545  -0.545454545454545  0.870591471154497
-0.181818181818182  -0.363636363636364  -0.727272727272727  0.870591471154497
-0.181818181818182  -0.181818181818182  -0.909090909090909  0.870591471154497
0.000000000000000  -0.909090909090909  -0.363636363636364  0.870591471154497
0.000000000000000  -0.727272727272727  -0.545454545454545  0.870591471154497
0.000000000000000  -0.545454545454545  -0.727272727272727  0.870591471154497
0.000000000000000  -0.363636363636364  -0.909090909090909  0.870591471154497
0.181818181818182  -0.909090909090909  -0.545454545454545  0.870591471154497
0.181818181818182  -0.727272727272727  -0.727272727272727  0.870591471154497
0.181818181818182  -0.545454545454545  -0.909090909090909  0.870591471154497
0.363636363636364  -0.909090909090909  -0.727272727272727  0.870591471154497
0.363636363636364  -0.727272727272727  -0.909090909090909  0.870591471154497
0.545454545454545  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  0.363636363636364  0.870591471154497
-0.909090909090909  -0.727272727272727  0.181818181818182  0.870591471154497
-0.909090909090909  -0.545454545454545  0.000000000000000  0.870591471154497
-0.909090909090909  -0.363636363636364  -0.181818181818182  0.870591471154497
-0.909090909090909  -0.181818181818182  -0.363636363636364  0.870591471154497
-0.909090909090909  0.000000000000000  -0.545454545454545  0.870591471154497
-0.909090909090909  0.181818181818182  -0.727272727272727  0.870591471154497
-0.909090909090909  0.363636363636364  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  0.181818181818182  0.870591471154497
-0.727272727272727  -0.727272727272727  0.000000000000000  0.870591471154497
-0.727272727272727  -0.545454545454545  -0.181818181818182  0.870591471154497
-0.727272727272727  -0.363636363636364  -0.363636363636364  0.870591471154497
-0.727272727272727  -0.181818181818182  -0.545454545454545  0.870591471154497
-0.727272727272727  0.000000000000000  -0.727272727272727  0.870591471154497
-0.727272727272727  0.181818181818182  -0.909090909090909  0.870591471154497
-0.545454545454545  -0.909090909090909  0.000000000000000  0.870591471154497
-0.545454545454545  -0.727272727272727  -0.181818181818182  0.870591471154497
-0.545454545454545  -0.545454545454545  -0.363636363636364  0.870591471154497
-0.545454545454545  -0.363636363636364  -0.545454545454545  0.870591471154497
-0.545454545454545  -0.181818181818182  -0.727272727272727  0.870591471154497
-0.545454545454545  0.000000000000000  -0.909090909090909  0.870591471154497
-0.363636363636364  -0.909090909090909  -0.181818181818182  0.870591471154497
-0.363636363636364  -0.727272727272727  -0.363636363636364  0.870591471154497
-0.363636363636364  -0.545454545454545  -0.545454545454545  0.870591471154497
-0.363636363636364  -0.363636363636364  -0.727272727272727  0.870591471154497
-0.363636363636364  -0.181818181818182  -0.909090909090909  0.870591471154497
-0.181818181818182  -0.909090909090909  -0.363636363636364  0.870591471154497
-0.181818181818182  -0.727272727272727  -0.545454545454545  0.870591471154497
-0.181818181818182  -0.545454545454545  -0.727272727272727  0.870591471154497
-0.181818181818182  -0.363636363636364  -0.909090909090909  0.870591471154497
0.000000000000000  -0.909090909090909  -0.545454545454545  0.870591471154497
0.000000000000000  -0.727272727272727  -0.727272727272727  0.870591471154497
0.000000000000000  -0.545454545454545  -0.909090909090909  0.870591471154497
0.181818181818182  -0.909090909090909  -0.727272727272727  0.870591471154497
0.181818181818182  -0.727272727272727  -0.909090909090909  0.870591471154497
0.363636363636364  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  0.181818181818182  0.870591471154497
-0.909090909090909  -0.727272727272727  0.000000000000000  0.870591471154497
-0.909090909090909  -0.545454545454545  -0.181818181818182  0.870591471154497
-0.909090909090909  -0.363636363636364  -0.363636363636364  0.870591471154497
-0.909090909090909  -0.181818181818182  -0.545454545454545  0.870591471154497
-0.909090909090909  0.000000000000000  -0.727272727272727  0.870591471154497
-0.909090909090909  0.181818181818182  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  0.000000000000000  0.870591471154497
-0.727272727272727  -0.727272727272727  -0.181818181818182  0.870591471154497
-0.727272727272727  -0.545454545454545  -0.363636363636364  0.870591471154497
-0.727272727272727  -0.363636363636364  -0.545454545454545  0.870591471154497
-0.727272727272727  -0.181818181818182  -0.727272727272727  0.870591471154497
-0.727272727272727  0.000000000000000  -0.909090909090909  0.870591471154497
-0.545454545454545  -0.909090909090909  -0.181818181818182  0.870591471154497
-0.545454545454545  -0.727272727272727  -0.363636363636364  0.870591471154497
-0.545454545454545  -0.545454545454545  -0.545454545454545  0.870591471154497
-0.545454545454545  -0.363636363636364  -0.727272727272727  0.870591471154497
-0.545454545454545  -0.181818181818182  -0.909090909090909  0.870591471154497
-0.363636363636364  -0.909090909090909  -0.363636363636364  0.870591471154497
-0.363636363636364  -0.727272727272727  -0.545454545454545  0.870591471154497
-0.363636363636364  -0.545454545454545  -0.727272727272727  0.870591471154497
-0.363636363636364  -0.363636363636364  -0.909090909090909  0.870591471154497
-0.181818181818182  -0.909090909090909  -0.545454545454545  0.870591471154497
-0.181818181818182  -0.727272727272727  -0.727272727272727  0.870591471154497
-0.181818181818182  -0.545454545454545  -0.909090909090909  0.870591471154497
0.000000000000000  -0.909090909090909  -0.727272727272727  0.870591471154497
0.000000000000000  -0.727272727272727  -0.909090909090909  0.870591471154497
0.181818181818182  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  0.000000000000000  0.870591471154497
-0.909090909090909  -0.727272727272727  -0.181818181818182  0.870591471154497
-0.909090909090909  -0.545454545454545  -0.363636363636364  0.870591471154497
-0.909090909090909  -0.363636363636364  -0.545454545454545  0.870591471154497
-0.909090909090909  -0.181818181818182  -0.727272727272727  0.870591471154497
-0.909090909090909  0.000000000000000  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  -0.181818181818182  0.870591471154497
-0.727272727272727  -0.727272727272727  -0.363636363636364  0.870591471154497
-0.727272727272727  -0.545454545454545  -0.545454545454545  0.870591471154497
-0.727272727272727  -0.363636363636364  -0.727272727272727  0.870591471154497
-0.727272727272727  -0.181818181818182  -0.909090909090909  0.870591471154497
-0.545454545454545  -0.909090909090909  -0.363636363636364  0.870591471154497
-0.545454545454545  -0.727272727272727  -0.545454545454545  0.870591471154497
-0.545454545454545  -0.545454545454545  -0.727272727272727  0.870591471154497
-0.545454545454545  -0.363636363636364  -0.909090909090909  0.870591471154497
-0.363636363636364  -0.909090909090909  -0.545454545454545  0.870591471154497
-0.363636363636364  -0.727272727272727  -0.727272727272727  0.870591471154497
-0.363636363636364  -0.545454545454545  -0.909090909090909  0.870591471154497
-0.181818181818182  -0.909090909090909  -0.727272727272727  0.870591471154497
-0.181818181818182  -0.727272727272727  -0.909090909090909  0.870591471154497
0.000000000000000  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  -0.181818181818182  0.870591471154497
-0.909090909090909  -0.727272727272727  -0.363636363636364  0.870591471154497
-0.909090909090909  -0.545454545454545  -0.545454545454545  0.870591471154497
-0.909090909090909  -0.363636363636364  -0.727272727272727  0.870591471154497
-0.909090909090909  -0.181818181818182  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  -0.363636363636364  0.870591471154497
-0.727272727272727  -0.727272727272727  -0.545454545454545  0.870591471154497
-0.727272727272727  -0.545454545454545  -0.727272727272727  0.870591471154497
-0.727272727272727  -0.363636363636364  -0.909090909090909  0.870591471154497
-0.545454545454545  -0.909090909090909  -0.545454545454545  0.870591471154497
-0.545454545454545  -0.727272727272727  -0.727272727272727  0.870591471154497
-0.545454545454545  -0.545454545454545  -0.909090909090909  0.870591471154497
-0.363636363636364  -0.909090909090909  -0.727272727272727  0.870591471154497
-0.363636363636364  -0.727272727272727  -0.909090909090909  0.870591471154497
-0.181818181818182  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  -0.363636363636364  0.870591471154497
-0.909090909090909  -0.727272727272727  -0.545454545454545  0.870591471154497
-0.909090909090909  -0.545454545454545  -0.727272727272727  0.870591471154497
-0.909090909090909  -0.363636363636364  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  -0.545454545454545  0.870591471154497
-0.727272727272727  -0.727272727272727  -0.727272727272727  0.870591471154497
-0.727272727272727  -0.545454545454545  -0.909090909090909  0.870591471154497
-0.545454545454545  -0.909090909090909  -0.727272727272727  0.870591471154497
-0.545454545454545  -0.727272727272727  -0.909090909090909  0.870591471154497
-0.363636363636364  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  -0.545454545454545  0.870591471154497
-0.909090909090909  -0.727272727272727  -0.727272727272727  0.870591471154497
-0.909090909090909  -0.545454545454545  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  -0.727272727272727  0.870591471154497
-0.727272727272727  -0.727272727272727  -0.909090909090909  0.870591471154497
-0.545454545454545  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  -0.727272727272727  0.870591471154497
-0.909090909090909  -0.727272727272727  -0.909090909090909  0.870591471154497
-0.727272727272727  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.909090909090909  -0.909090909090909  -0.909090909090909  0.870591471154497
-0.900000000000000  -0.900000000000000  0.700000000000000  -3.131670570142602
-0.900000000000000  -0.700000000000000  0.500000000000000  -3.131670570142602
-0.900000000000000  -0.500000000000000  0.300000000000000  -3.131670570142602
-0.900000000000000  -0.300000000000000  0.100000000000000  -3.131670570142602
-0.900000000000000  -0.100000000000000  -0.100000000000000  -3.131670570142602
-0.900000000000000  0.100000000000000  -0.300000000000000  -3.131670570142602
-0.900000000000000  0.300000000000000  -0.500000000000000  -3.131670570142602
-0.900000000000000  0.500000000000000  -0.700000000000000  -3.131670570142602
-0.900000000000000  0.700000000000000  -0.900000000000000  -3.131670570142602
-0.700000000000000  -0.900000000000000  0.500000000000000  -3.131670570142602
-0.700000000000000  -0.700000000000000  0.300000000000000  -3.131670570142602
-0.700000000000000  -0.500000000000000  0.100000000000000  -3.131670570142602
-0.700000000000000  -0.300000000000000  -0.100000000000000  -3.131670570142602
-0.700000000000000  -0.100000000000000  -0.300000000000000  -3.131670570142602
-0.700000000000000  0.100000000000000  -0.500000000000000  -3.131670570142602
-0.700000000000000  0.300000000000000  -0.700000000000000  -3.131670570142602
-0.700000000000000  0.500000000000000  -0.900000000000000  -3.131670570142602
-0.500000000000000  -0.900000000000000  0.300000000000000  -3.131670570142602
-0.500000000000000  -0.700000000000000  0.100000000000000  -3.131670570142602
-0.500000000000000  -0.500000000000000  -0.100000000000000  -3.131670570142602
-0.500000000000000  -0.300000000000000  -0.300000000000000  -3.131670570142602
-0.500000000000000  -0.100000000000000  -0.500000000000000  -3.131670570142602
-0.500000000000000  0.100000000000000  -0.700000000000000  -3.131670570142602
-0.500000000000000  0.300000000000000  -0.900000000000000  -3.131670570142602
-0.300000000000000  -0.900000000000000  0.100000000000000  -3.131670570142602
-0.300000000000000  -0.700000000000000  -0.100000000000000  -3.131670570142602
-0.300000000000000  -0.500000000000000  -0.300000000000000  -3.131670570142602
-0.300000000000000  -0.300000000000000  -0.500000000000000  -3.131670570142602
-0.300000000000000  -0.100000000000000  -0.700000000000000  -3.131670570142602
-0.300000000000000  0.100000000000000  -0.900000000000000  -3.131670570142602
-0.100000000000000  -0.900000000000000  -0.100000000000000  -3.131670570142602
-0.100000000000000  -0.700000000000000  -0.300000000000000  -3.131670570142602
-0.100000000000000  -0.500000000000000  -0.500000000000000  -3.131670570142602
-0.100000000000000  -0.300000000000000  -0.700000000000000  -3.131670570142602
-0.100000000000000  -0.100000000000000  -0.900000000000000  -3.131670570142602
0.100000000000000  -0.900000000000000  -0.300000000000000  -3.131670570142602
0.100000000000000  -0.700000000000000  -0.500000000000000  -3.131670570142602
0.100000000000000  -0.500000000000000  -0.700000000000000  -3.131670570142602
0.100000000000000  -0.300000000000000  -0.900000000000000  -3.131670570142602
0.300000000000000  -0.900000000000000  -0.500000000000000  -3.131670570142602
0.300000000000000  -0.700000000000000  -0.700000000000000  -3.131670570142602
0.300000000000000  -0.500000000000000  -0.900000000000000  -3.131670570142602
0.500000000000000  -0.900000000000000  -0.700000000000000  -3.131670570142602
0.500000000000000  -0.700000000000000  -0.900000000000000  -3.131670570142602
0.700000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.900000000000000  -0.900000000000000  0.500000000000000  -3.131670570142602
-0.900000000000000  -0.700000000000000  0.300000000000000  -3.131670570142602
-0.900000000000000  -0.500000000000000  0.100000000000000  -3.131670570142602
-0.900000000000000  -0.300000000000000  -0.100000000000000  -3.131670570142602
-0.900000000000000  -0.100000000000000  -0.300000000000000  -3.131670570142602
-0.900000000000000  0.100000000000000  -0.500000000000000  -3.131670570142602
-0.900000000000000  0.300000000000000  -0.700000000000000  -3.131670570142602
-0.900000000000000  0.500000000000000  -0.900000000000000  -3.131670570142602
-0.700000000000000  -0.900000000000000  0.300000000000000  -3.131670570142602
-0.700000000000000  -0.700000000000000  0.100000000000000  -3.131670570142602
-0.700000000000000  -0.500000000000000  -0.100000000000000  -3.131670570142602
-0.700000000000000  -0.300000000000000  -0.300000000000000  -3.131670570142602
-0.700000000000000  -0.100000000000000  -0.500000000000000  -3.131670570142602
-0.700000000000000  0.100000000000000  -0.700000000000000  -3.131670570142602
-0.700000000000000  0.300000000000000  -0.900000000000000  -3.131670570142602
-0.500000000000000  -0.900000000000000  0.100000000000000  -3.131670570142602
-0.500000000000000  -0.700000000000000  -0.100000000000000  -3.131670570142602
-0.500000000000000  -0.500000000000000  -0.300000000000000  -3.131670570142602
-0.500000000000000  -0.300000000000000  -0.500000000000000  -3.131670570142602
-0.500000000000000  -0.100000000000000  -0.700000000000000  -3.131670570142602
-0.500000000000000  0.100000000000000  -0.900000000000000  -3.131670570142602
-0.300000000000000  -0.900000000000000  -0.100000000000000  -3.131670570142602
-0.300000000000000  -0.700000000000000  -0.300000000000000  -3.131670570142602
-0.300000000000000  -0.500000000000000  -0.500000000000000  -3.131670570142602
-0.300000000000000  -0.300000000000000  -0.700000000000000  -3.131670570142602
-0.300000000000000  -0.100000000000000  -0.900000000000000  -3.131670570142602
-0.100000000000000  -0.900000000000000  -0.300000000000000  -3.131670570142602
-0.100000000000000  -0.700000000000000  -0.500000000000000  -3.131670570142602
-0.100000000000000  -0.500000000000000  -0.700000000000000  -3.131670570142602
-0.100000000000000  -0.300000000000000  -0.900000000000000  -3.131670570142602
0.100000000000000  -0.900000000000000  -0.500000000000000  -3.131670570142602
0.100000000000000  -0.700000000000000  -0.700000000000000  -3.131670570142602
0.100000000000000  -0.500000000000000  -0.900000000000000  -3.131670570142602
0.300000000000000  -0.900000000000000  -0.700000000000000  -3.131670570142602
0.300000000000000  -0.700000000000000  -0.900000000000000  -3.131670570142602
0.500000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.900000000000000  -0.900000000000000  0.300000000000000  -3.131670570142602
-0.900000000000000  -0.700000000000000  0.100000000000000  -3.131670570142602
-0.900000000000000  -0.500000000000000  -0.100000000000000  -3.131670570142602
-0.900000000000000  -0.300000000000000  -0.300000000000000  -3.131670570142602
-0.900000000000000  -0.100000000000000  -0.500000000000000  -3.131670570142602
-0.900000000000000  0.100000000000000  -0.700000000000000  -3.131670570142602
-0.900000000000000  0.300000000000000  -0.900000000000000  -3.131670570142602
-0.700000000000000  -0.900000000000000  0.100000000000000  -3.131670570142602
-0.700000000000000  -0.700000000000000  -0.100000000000000  -3.131670570142602
-0.700000000000000  -0.500000000000000  -0.300000000000000  -3.131670570142602
-0.700000000000000  -0.300000000000000  -0.500000000000000  -3.131670570142602
-0.700000000000000  -0.100000000000000  -0.700000000000000  -3.131670570142602
-0.700000000000000  0.100000000000000  -0.900000000000000  -3.131670570142602
-0.500000000000000  -0.900000000000000  -0.100000000000000  -3.131670570142602
-0.500000000000000  -0.700000000000000  -0.300000000000000  -3.131670570142602
-0.500000000000000  -0.500000000000000  -0.500000000000000  -3.131670570142602
-0.500000000000000  -0.300000000000000  -0.700000000000000  -3.131670570142602
-0.500000000000000  -0.100000000000000  -0.900000000000000  -3.131670570142602
-0.300000000000000  -0.900000000000000  -0.300000000000000  -3.131670570142602
-0.300000000000000  -0.700000000000000  -0.500000000000000  -3.131670570142602
-0.300000000000000  -0.500000000000000  -0.700000000000000  -3.131670570142602
-0.300000000000000  -0.300000000000000  -0.900000000000000  -3.131670570142602
-0.100000000000000  -0.900000000000000  -0.500000000000000  -3.131670570142602
-0.100000000000000  -0.700000000000000  -0.700000000000000  -3.131670570142602
-0.100000000000000  -0.500000000000000  -0.900000000000000  -3.131670570142602
0.100000000000000  -0.900000000000000  -0.700000000000000  -3.131670570142602
0.100000000000000  -0.700000000000000  -0.900000000000000  -3.131670570142602
0.300000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.900000000000000  -0.900000000000000  0.100000000000000  -3.131670570142602
-0.900000000000000  -0.700000000000000  -0.100000000000000  -3.131670570142602
-0.900000000000000  -0.500000000000000  -0.300000000000000  -3.131670570142602
-0.900000000000000  -0.300000000000000  -0.500000000000000  -3.131670570142602
-0.900000000000000  -0.100000000000000  -0.700000000000000  -3.131670570142602
-0.900000000000000  0.100000000000000  -0.900000000000000  -3.131670570142602
-0.700000000000000  -0.900000000000000  -0.100000000000000  -3.131670570142602
-0.700000000000000  -0.700000000000000  -0.300000000000000  -3.131670570142602
-0.700000000000000  -0.500000000000000  -0.500000000000000  -3.131670570142602
-0.700000000000000  -0.300000000000000  -0.700000000000000  -3.131670570142602
-0.700000000000000  -0.100000000000000  -0.900000000000000  -3.131670570142602
-0.500000000000000  -0.900000000000000  -0.300000000000000  -3.131670570142602
-0.500000000000000  -0.700000000000000  -0.500000000000000  -3.131670570142602
-0.500000000000000  -0.500000000000000  -0.700000000000000  -3.131670570142602
-0.500000000000000  -0.300000000000000  -0.900000000000000  -3.131670570142602
-0.300000000000000  -0.900000000000000  -0.500000000000000  -3.131670570142602
-0.300000000000000  -0.700000000000000  -0.700000000000000  -3.131670570142602
-0.300000000000000  -0.500000000000000  -0.900000000000000  -3.131670570142602
-0.100000000000000  -0.900000000000000  -0.700000000000000  -3.131670570142602
-0.100000000000000  -0.700000000000000  -0.900000000000000  -3.131670570142602
0.100000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.900000000000000  -0.900000000000000  -0.100000000000000  -3.131670570142602
-0.900000000000000  -0.700000000000000  -0.300000000000000  -3.131670570142602
-0.900000000000000  -0.500000000000000  -0.500000000000000  -3.131670570142602
-0.900000000000000  -0.300000000000000  -0.700000000000000  -3.131670570142602
-0.900000000000000  -0.100000000000000  -0.900000000000000  -3.131670570142602
-0.700000000000000  -0.900000000000000  -0.300000000000000  -3.131670570142602
-0.700000000000000  -0.700000000000000  -0.500000000000000  -3.131670570142602
-0.700000000000000  -0.500000000000000  -0.700000000000000  -3.131670570142602
-0.700000000000000  -0.300000000000000  -0.900000000000000  -3.131670570142602
-0.500000000000000  -0.900000000000000  -0.500000000000000  -3.131670570142602
-0.500000000000000  -0.700000000000000  -0.700000000000000  -3.131670570142602
-0.500000000000000  -0.500000000000000  -0.900000000000000  -3.131670570142602
-0.300000000000000  -0.900000000000000  -0.700000000000000  -3.131670570142602
-0.300000000000000  -0.700000000000000  -0.900000000000000  -3.131670570142602
-0.100000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.900000000000000  -0.900000000000000  -0.300000000000000  -3.131670570142602
-0.900000000000000  -0.700000000000000  -0.500000000000000  -3.131670570142602
-0.900000000000000  -0.500000000000000  -0.700000000000000  -3.131670570142602
-0.900000000000000  -0.300000000000000  -0.900000000000000  -3.131670570142602
-0.700000000000000  -0.900000000000000  -0.500000000000000  -3.131670570142602
-0.700000000000000  -0.700000000000000  -0.700000000000000  -3.131670570142602
-0.700000000000000  -0.500000000000000  -0.900000000000000  -3.131670570142602
-0.500000000000000  -0.900000000000000  -0.700000000000000  -3.131670570142602
-0.500000000000000  -0.700000000000000  -0.900000000000000  -3.131670570142602
-0.300000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.900000000000000  -0.900000000000000  -0.500000000000000  -3.131670570142602
-0.900000000000000  -0.700000000000000  -0.700000000000000  -3.131670570142602
-0.900000000000000  -0.500000000000000  -0.900000000000000  -3.131670570142602
-0.700000000000000  -0.900000000000000  -0.700000000000000  -3.131670570142602
-0.700000000000000  -0.700000000000000  -0.900000000000000  -3.131670570142602
-0.500000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.900000000000000  -0.900000000000000  -0.700000000000000  -3.131670570142602
-0.900000000000000  -0.700000000000000  -0.900000000000000  -3.131670570142602
-0.700000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.900000000000000  -0.900000000000000  -0.900000000000000  -3.131670570142602
-0.888888888888889  -0.888888888888889  0.666666666666667  4.441943697306247
-0.888888888888889  -0.666666666666667  0.444444444444444  4.441943697306247
-0.888888888888889  -0.444444444444444  0.222222222222222  4.441943697306247
-0.888888888888889  -0.222222222222222  0.000000000000000  4.441943697306247
-0.888888888888889  0.000000000000000  -0.222222222222222  4.441943697306247
-0.888888888888889  0.222222222222222  -0.444444444444444  4.441943697306247
-0.888888888888889  0.444444444444444  -0.666666666666667  4.441943697306247
-0.888888888888889  0.666666666666667  -0.888888888888889  4.441943697306247
-0.666666666666667  -0.888888888888889  0.444444444444444  4.441943697306247
-0.666666666666667  -0.666666666666667  0.222222222222222  4.441943697306247
-0.666666666666667  -0.444444444444444  0.000000000000000  4.441943697306247
-0.666666666666667  -0.222222222222222  -0.222222222222222  4.441943697306247
-0.666666666666667  0.000000000000000  -0.444444444444444  4.441943697306247
-0.666666666666667  0.222222222222222  -0.666666666666667  4.441943697306247
-0.666666666666667  0.444444444444444  -0.888888888888889  4.441943697306247
-0.444444444444444  -0.888888888888889  0.222222222222222  4.441943697306247
-0.444444444444444  -0.666666666666667  0.000000000000000  4.441943697306247
-0.444444444444444  -0.444444444444444  -0.222222222222222  4.441943697306247
-0.444444444444444  -0.222222222222222  -0.444444444444444  4.441943697306247
-0.444444444444444  0.000000000000000  -0.666666666666667  4.441943697306247
-0.444444444444444  0.222222222222222  -0.888888888888889  4.441943697306247
-0.222222222222222  -0.888888888888889  0.000000000000000  4.441943697306247
-0.222222222222222  -0.666666666666667  -0.222222222222222  4.441943697306247
-0.222222222222222  -0.444444444444444  -0.444444444444444  4.441943697306247
-0.222222222222222  -0.222222222222222  -0.666666666666667  4.441943697306247
-0.222222222222222  0.000000000000000  -0.888888888888889  4.441943697306247
0.000000000000000  -0.888888888888889  -0.222222222222222  4.441943697306247
0.000000000000000  -0.666666666666667  -0.444444444444444  4.441943697306247
0.000000000000000  -0.444444444444444  -0.666666666666667  4.441943697306247
0.000000000000000  -0.222222222222222  -0.888888888888889  4.441943697306247
0.222222222222222  -0.888888888888889  -0.444444444444444  4.441943697306247
0.222222222222222  -0.666666666666667  -0.666666666666667  4.441943697306247
0.222222222222222  -0.444444444444444  -0.888888888888889  4.441943697306247
0.444444444444444  -0.888888888888889  -0.666666666666667  4.441943697306247
0.444444444444444  -0.666666666666667  -0.888888888888889  4.441943697306247
0.666666666666667  -0.888888888888889  -0.888888888888889  4.441943697306247
-0.888888888888889  -0.888888888888889  0.444444444444444  4.441943697306247
-0.888888888888889  -0.666666666666667  0.222222222222222  4.441943697306247
-0.888888888888889  -0.444444444444444  0.000000000000000  4.441943697306247
-0.888888888888889  -0.222222222222222  -0.222222222222222  4.441943697306247
-0.888888888888889  0.000000000000000  -0.444444444444444  4.441943697306247
-0.888888888888889  0.222222222222222  -0.666666666666667  4.441943697306247
-0.888888888888889  0.444444444444444  -0.888888888888889  4.441943697306247
-0.666666666666667  -0.888888888888889  0.222222222222222  4.441943697306247
-0.666666666666667  -0.666666666666667  0.000000000000000  4.441943697306247
-0.666666666666667  -0.444444444444444  -0.222222222222222  4.441943697306247
-0.666666666666667  -0.222222222222222  -0.444444444444444  4.441943697306247
-0.666666666666667  0.000000000000000  -0.666666666666667  4.441943697306247
-0.666666666666667  0.222222222222222  -0.888888888888889  4.441943697306247
-0.444444444444444  -0.888888888888889  0.000000000000000  4.441943697306247
-0.444444444444444  -0.666666666666667  -0.222222222222222  4.441943697306247
-0.444444444444444  -0.444444444444444  -0.444444444444444  4.441943697306247
-0.444444444444444  -0.222222222222222  -0.666666666666667  4.441943697306247
-0.444444444444444  0.000000000000000  -0.888888888888889  4.441943697306247
-0.222222222222222  -0.888888888888889  -0.222222222222222  4.441943697306247
-0.222222222222222  -0.666666666666667  -0.444444444444444  4.441943697306247
-0.222222222222222  -0.444444444444444  -0.666666666666667  4.441943697306247
-0.222222222222222  -0.222222222222222  -0.888888888888889  4.441943697306247
0.000000000000000  -0.888888888888889  -0.444444444444444  4.441943697306247
0.000000000000000  -0.666666666666667  -0.666666666666667  4.441943697306247
0.000000000000000  -0.444444444444444  -0.888888888888889  4.441943697306247
0.222222222222222  -0.888888888888889  -0.666666666666667  4.441943697306247
0.222222222222222  -0.666666666666667  -0.888888888888889  4.441943697306247
0.444444444444444  -0.888888888888889  -0.888888888888889  4.441943697306247
-0.888888888888889  -0.888888888888889  0.222222222222222  4.441943697306247
-0.888888888888889  -0.666666666666667  0.000000000000000  4.441943697306247
-0.888888888888889  -0.444444444444444  -0.222222222222222  4.441943697306247
-0.888888888888889  -0.222222222222222  -0.444444444444444  4.441943697306247
-0.888888888888889  0.000000000000000  -0.666666666666667  4.441943697306247
-0.888888888888889  0.222222222222222  -0.888888888888889  4.441943697306247
-0.666666666666667  -0.888888888888889  0.000000000000000  4.441943697306247
-0.666666666666667  -0.666666666666667  -0.222222222222222  4.441943697306247
-0.666666666666667  -0.444444444444444  -0.444444444444444  4.441943697306247
-0.666666666666667  -0.222222222222222  -0.666666666666667  4.441943697306247
-0.666666666666667  0.000000000000000  -0.888888888888889  4.441943697306247
-0.444444444444444  -0.888888888888889  -0.222222222222222  4.441943697306247
-0.444444444444444  -0.666666666666667  -0.444444444444444  4.441943697306247
-0.444444444444444  -0.444444444444444  -0.666666666666667  4.441943697306247
-0.444444444444444  -0.222222222222222  -0.888888888888889  4.441943697306247
-0.222222222222222  -0.888888888888889  -0.444444444444444  4.441943697306247
-0.222222222222222  -0.666666666666667  -0.666666666666667  4.441943697306247
-0.222222222222222  -0.444444444444444  -0.888888888888889  4.441943697306247
0.000000000000000  -0.888888888888889  -0.666666666666667  4.441943697306247
0.000000000000000  -0.666666666666667  -0.888888888888889  4.441943697306247
0.222222222222222  -0.888888888888889  -0.888888888888889  4.441943697306247
-0.888888888888889  -0.888888888888889  0.000000000000000  4.441943697306247
-0.888888888888889  -0.666666666666667  -0.222222222222222  4.441943697306247
-0.888888888888889  -0.444444444444444  -0.444444444444444  4.441943697306247
-0.888888888888889  -0.222222222222222  -0.666666666666667  4.441943697306247
-0.888888888888889  0.000000000000000  -0.888888888888889  4.441943697306247
-0.666666666666667  -0.888888888888889  -0.222222222222222  4.441943697306247
-0.666666666666667  -0.666666666666667  -0.444444444444444  4.441943697306247
-0.666666666666667  -0.444444444444444  -0.666666666666667  4.441943697306247
-0.666666666666667  -0.222222222222222  -0.888888888888889  4.441943697306247
-0.444444444444444  -0.888888888888889  -0.444444444444444  4.441943697306247
-0.444444444444444  -0.666666666666667  -0.666666666666667  4.441943697306247
-0.444444444444444  -0.444444444444444  -0.888888888888889  4.441943697306247
-0.222222222222222  -0.888888888888889  -0.666666666666667  4.441943697306247
-0.222222222222222  -0.666666666666667  -0.888888888888889  4.441943697306247
0.000000000000000  -0.888888888888889  -0.888888888888889  4.441943697306247
-0.888888888888889  -0.888888888888889  -0.222222222222222  4.441943697306247
-0.888888888888889  -0.666666666666667  -0.444444444444444  4.441943697306247
-0.888888888888889  -0.444444444444444  -0.666666666666667  4.441943697306247
-0.888888888888889  -0.222222222222222  -0.888888888888889  4.441943697306247
-0.666666666666667  -0.888888888888889  -0.444444444444444  4.441943697306247
-0.666666666666667  -0.666666666666667  -0.666666666666667  4.441943697306247
-0.666666666666667  -0.444444444444444  -0.888888888888889  4.441943697306247
-0.444444444444444  -0.888888888888889  -0.666666666666667  4.441943697306247
-0.444444444444444  -0.666666666666667  -0.888888888888889  4.441943697306247
-0.222222222222222  -0.888888888888889  -0.888888888888889  4.441943697306247
-0.888888888888889  -0.888888888888889  -0.444444444444444  4.441943697306247
-0.888888888888889  -0.666666666666667  -0.666666666666667  4.441943697306247
-0.888888888888889  -0.444444444444444  -0.888888888888889  4.441943697306247
-0.666666666666667  -0.888888888888889  -0.666666666666667  4.441943697306247
-0.666666666666667  -0.666666666666667  -0.888888888888889  4.441943697306247
-0.444444444444444  -0.888888888888889  -0.888888888888889  4.441943697306247
-0.888888888888889  -0.888888888888889  -0.666666666666667  4.441943697306247
-0.888888888888889  -0.666666666666667  -0.888888888888889  4.441943697306247
-0.666666666666667  -0.888888888888889  -0.888888888888889  4.441943697306247
-0.888888888888889  -0.888888888888889  -0.888888888888889  4.441943697306247
-0.875000000000000  -0.875000000000000  0.625000000000000  -3.159249052454067
-0.875000000000000  -0.625000000000000  0.375000000000000  -3.159249052454067
-0.875000000000000  -0.375000000000000  0.125000000000000  -3.159249052454067
-0.875000000000000  -0.125000000000000  -0.125000000000000  -3.159249052454067
-0.875000000000000  0.125000000000000  -0.375000000000000  -3.159249052454067
-0.875000000000000  0.375000000000000  -0.625000000000000  -3.159249052454067
-0.875000000000000  0.625000000000000  -0.875000000000000  -3.159249052454067
-0.625000000000000  -0.875000000000000  0.375000000000000  -3.159249052454067
-0.625000000000000  -0.625000000000000  0.125000000000000  -3.159249052454067
-0.625000000000000  -0.375000000000000  -0.125000000000000  -3.159249052454067
-0.625000000000000  -0.125000000000000  -0.375000000000000  -3.159249052454067
-0.625000000000000  0.125000000000000  -0.625000000000000  -3.159249052454067
-0.625000000000000  0.375000000000000  -0.875000000000000  -3.159249052454067
-0.375000000000000  -0.875000000000000  0.125000000000000  -3.159249052454067
-0.375000000000000  -0.625000000000000  -0.125000000000000  -3.159249052454067
-0.375000000000000  -0.375000000000000  -0.375000000000000  -3.159249052454067
-0.375000000000000  -0.125000000000000  -0.625000000000000  -3.159249052454067
-0.375000000000000  0.125000000000000  -0.875000000000000  -3.159249052454067
-0.125000000000000  -0.875000000000000  -0.125000000000000  -3.159249052454067
-0.125000000000000  -0.625000000000000  -0.375000000000000  -3.159249052454067
-0.125000000000000  -0.375000000000000  -0.625000000000000  -3.159249052454067
-0.125000000000000  -0.125000000000000  -0.875000000000000  -3.159249052454067
0.125000000000000  -0.875000000000000  -0.375000000000000  -3.159249052454067
0.125000000000000  -0.625000000000000  -0.625000000000000  -3.159249052454067
0.125000000000000  -0.375000000000000  -0.875000000000000  -3.159249052454067
0.375000000000000  -0.875000000000000  -0.625000000000000  -3.159249052454067
0.375000000000000  -0.625000000000000  -0.875000000000000  -3.159249052454067
0.625000000000000  -0.875000000000000  -0.875000000000000  -3.159249052454067
-0.875000000000000  -0.875000000000000  0.375000000000000  -3.159249052454067
-0.875000000000000  -0.625000000000000  0.125000000000000  -3.159249052454067
-0.875000000000000  -0.375000000000000  -0.125000000000000  -3.159249052454067
-0.875000000000000  -0.125000000000000  -0.375000000000000  -3.159249052454067
-0.875000000000000  0.125000000000000  -0.625000000000000  -3.159249052454067
-0.875000000000000  0.375000000000000  -0.875000000000000  -3.159249052454067
-0.625000000000000  -0.875000000000000  0.125000000000000  -3.159249052454067
-0.625000000000000  -0.625000000000000  -0.125000000000000  -3.159249052454067
-0.625000000000000  -0.375000000000000  -0.375000000000000  -3.159249052454067
-0.625000000000000  -0.125000000000000  -0.625000000000000  -3.159249052454067
-0.625000000000000  0.125000000000000  -0.875000000000000  -3.159249052454067
-0.375000000000000  -0.875000000000000  -0.125000000000000  -3.159249052454067
-0.375000000000000  -0.625000000000000  -0.375000000000000  -3.159249052454067
-0.375000000000000  -0.375000000000000  -0.625000000000000  -3.159249052454067
-0.375000000000000  -0.125000000000000  -0.875000000000000  -3.159249052454067
-0.125000000000000  -0.875000000000000  -0.375000000000000  -3.159249052454067
-0.125000000000000  -0.625000000000000  -0.625000000000000  -3.159249052454067
-0.125000000000000  -0.375000000000000  -0.875000000000000  -3.159249052454067
0.125000000000000  -0.875000000000000  -0.625000000000000  -3.159249052454067
0.125000000000000  -0.625000000000000  -0.875000000000000  -3.159249052454067
0.375000000000000  -0.875000000000000  -0.875000000000000  -3.159249052454067
-0.875000000000000  -0.875000000000000  0.125000000000000  -3.159249052454067
-0.875000000000000  -0.625000000000000  -0.125000000000000  -3.159249052454067
-0.875000000000000  -0.375000000000000  -0.375000000000000  -3.159249052454067
-0.875000000000000  -0.125000000000000  -0.625000000000000  -3.159249052454067
-0.875000000000000  0.125000000000000  -0.875000000000000  -3.159249052454067
-0.625000000000000  -0.875000000000000  -0.125000000000000  -3.159249052454067
-0.625000000000000  -0.625000000000000  -0.375000000000000  -3.159249052454067
-0.625000000000000  -0.375000000000000  -0.625000000000000  -3.159249052454067
-0.625000000000000  -0.125000000000000  -0.875000000000000  -3.159249052454067
-0.375000000000000  -0.875000000000000  -0.375000000000000  -3.159249052454067
-0.375000000000000  -0.625000000000000  -0.625000000000000  -3.159249052454067
-0.375000000000000  -0.375000000000000  -0.875000000000000  -3.159249052454067
-0.125000000000000  -0.875000000000000  -0.625000000000000  -3.159249052454067
-0.125000000000000  -0.625000000000000  -0.875000000000000  -3.159249052454067
0.125000000000000  -0.875000000000000  -0.875000000000000  -3.159249052454067
-0.875000000000000  -0.875000000000000  -0.125000000000000  -3.159249052454067
-0.875000000000000  -0.625000000000000  -0.375000000000000  -3.159249052454067
-0.875000000000000  -0.375000000000000  -0.625000000000000  -3.159249052454067
-0.875000000000000  -0.125000000000000  -0.875000000000000  -3.159249052454067
-0.625000000000000  -0.875000000000000  -0.375000000000000  -3.159249052454067
-0.625000000000000  -0.625000000000000  -0.625000000000000  -3.159249052454067
-0.625000000000000  -0.375000000000000  -0.875000000000000  -3.159249052454067
-0.375000000000000  -0.875000000000000  -0.625000000000000  -3.159249052454067
-0.375000000000000  -0.625000000000000  -0.875000000000000  -3.159249052454067
-0.125000000000000  -0.875000000000000  -0.875000000000000  -3.159249052454067
-0.875000000000000  -0.875000000000000  -0.375000000000000  -3.159249052454067
-0.875000000000000  -0.625000000000000  -0.625000000000000  -3.159249052454067
-0.875000000000000  -0.375000000000000  -0.875000000000000  -3.159249052454067
-0.625000000000000  -0.875000000000000  -0.625000000000000  -3.159249052454067
-0.625000000000000  -0.625000000000000  -0.875000000000000  -3.159249052454067
-0.375000000000000  -0.875000000000000  -0.875000000000000  -3.159249052454067
-0.875000000000000  -0.875000000000000  -0.625000000000000  -3.159249052454067
-0.875000000000000  -0.625000000000000  -0.875000000000000  -3.159249052454067
-0.625000000000000  -0.875000000000000  -0.875000000000000  -3.159249052454067
-0.875000000000000  -0.875000000000000  -0.875000000000000  -3.159249052454067
-0.857142857142857  -0.857142857142857  0.571428571428571  1.186944687423761
-0.857142857142857  -0.571428571428571  0.285714285714286  1.186944687423761
-0.857142857142857  -0.285714285714286  0.000000000000000  1.186944687423761
-0.857142857142857  0.000000000000000  -0.285714285714286  1.186944687423761
-0.857142857142857  0.285714285714286  -0.571428571428571  1.186944687423761
-0.857142857142857  0.571428571428571  -0.857142857142857  1.186944687423761
-0.571428571428571  -0.857142857142857  0.285714285714286  1.186944687423761
-0.571428571428571  -0.571428571428571  0.000000000000000  1.186944687423761
-0.571428571428571  -0.285714285714286  -0.285714285714286  1.186944687423761
-0.571428571428571  0.000000000000000  -0.571428571428571  1.186944687423761
-0.571428571428571  0.285714285714286  -0.857142857142857  1.186944687423761
-0.285714285714286  -0.857142857142857  0.000000000000000  1.186944687423761
-0.285714285714286  -0.571428571428571  -0.285714285714286  1.186944687423761
-0.285714285714286  -0.285714285714286  -0.571428571428571  1.186944687423761
-0.285714285714286  0.000000000000000  -0.857142857142857  1.186944687423761
0.000000000000000  -0.857142857142857  -0.285714285714286  1.186944687423761
0.000000000000000  -0.571428571428571  -0.571428571428571  1.186944687423761
0.000000000000000  -0.285714285714286  -0.857142857142857  1.186944687423761
0.285714285714286  -0.857142857142857  -0.571428571428571  1.186944687423761
0.285714285714286  -0.571428571428571  -0.857142857142857  1.186944687423761
0.571428571428571  -0.857142857142857  -0.857142857142857  1.186944687423761
-0.857142857142857  -0.857142857142857  0.285714285714286  1.186944687423761
-0.857142857142857  -0.571428571428571  0.000000000000000  1.186944687423761
-0.857142857142857  -0.285714285714286  -0.285714285714286  1.186944687423761
-0.857142857142857  0.000000000000000  -0.571428571428571  1.186944687423761
-0.857142857142857  0.285714285714286  -0.857142857142857  1.186944687423761
-0.571428571428571  -0.857142857142857  0.000000000000000  1.186944687423761
-0.571428571428571  -0.571428571428571  -0.285714285714286  1.186944687423761
-0.571428571428571  -0.285714285714286  -0.571428571428571  1.186944687423761
-0.571428571428571  0.000000000000000  -0.857142857142857  1.186944687423761
-0.285714285714286  -0.857142857142857  -0.285714285714286  1.186944687423761
-0.285714285714286  -0.571428571428571  -0.571428571428571  1.186944687423761
-0.285714285714286  -0.285714285714286  -0.857142857142857  1.186944687423761
0.000000000000000  -0.857142857142857  -0.571428571428571  1.186944687423761
0.000000000000000  -0.571428571428571  -0.857142857142857  1.186944687423761
0.285714285714286  -0.857142857142857  -0.857142857142857  1.186944687423761
-0.857142857142857  -0.857142857142857  0.000000000000000  1.186944687423761
-0.857142857142857  -0.571428571428571  -0.285714285714286  1.186944687423761
-0.857142857142857  -0.285714285714286  -0.571428571428571  1.186944687423761
-0.857142857142857  0.000000000000000  -0.857142857142857  1.186944687423761
-0.571428571428571  -0.857142857142857  -0.285714285714286  1.186944687423761
-0.571428571428571  -0.571428571428571  -0.571428571428571  1.186944687423761
-0.571428571428571  -0.285714285714286  -0.857142857142857  1.186944687423761
-0.285714285714286  -0.857142857142857  -0.571428571428571  1.186944687423761
-0.285714285714286  -0.571428571428571  -0.857142857142857  1.186944687423761
0.000000000000000  -0.857142857142857  -0.857142857142857  1.186944687423761
-0.857142857142857  -0.857142857142857  -0.285714285714286  1.186944687423761
-0.857142857142857  -0.571428571428571  -0.571428571428571  1.186944687423761
-0.857142857142857  -0.285714285714286  -0.857142857142857  1.186944687423761
-0.571428571428571  -0.857142857142857  -0.571428571428571  1.186944687423761
-0.571428571428571  -0.571428571428571  -0.857142857142857  1.186944687423761
-0.285714285714286  -0.857142857142857  -0.857142857142857  1.186944687423761
-0.857142857142857  -0.857142857142857  -0.571428571428571  1.186944687423761
-0.857142857142857  -0.571428571428571  -0.857142857142857  1.186944687423761
-0.571428571428571  -0.857142857142857  -0.857142857142857  1.186944687423761
-0.857142857142857  -0.857142857142857  -0.857142857142857  1.186944687423761
-0.833333333333333  -0.833333333333333  0.500000000000000  -0.228425181541148
-0.833333333333333  -0.500000000000000  0.166666666666667  -0.228425181541148
-0.833333333333333  -0.166666666666667  -0.166666666666667  -0.228425181541148
-0.833333333333333  0.166666666666667  -0.500000000000000  -0.228425181541148
-0.833333333333333  0.500000000000000  -0.833333333333333  -0.228425181541148
-0.500000000000000  -0.833333333333333  0.166666666666667  -0.228425181541148
-0.500000000000000  -0.500000000000000  -0.166666666666667  -0.228425181541148
-0.500000000000000  -0.166666666666667  -0.500000000000000  -0.228425181541148
-0.500000000000000  0.166666666666667  -0.833333333333333  -0.228425181541148
-0.166666666666667  -0.833333333333333  -0.166666666666667  -0.228425181541148
-0.166666666666667  -0.500000000000000  -0.500000000000000  -0.228425181541148
-0.166666666666667  -0.166666666666667  -0.833333333333333  -0.228425181541148
0.166666666666667  -0.833333333333333  -0.500000000000000  -0.228425181541148
0.166666666666667  -0.500000000000000  -0.833333333333333  -0.228425181541148
0.500000000000000  -0.833333333333333  -0.833333333333333  -0.228425181541148
-0.833333333333333  -0.833333333333333  0.166666666666667  -0.228425181541148
-0.833333333333333  -0.500000000000000  -0.166666666666667  -0.228425181541148
-0.833333333333333  -0.166666666666667  -0.500000000000000  -0.228425181541148
-0.833333333333333  0.166666666666667  -0.833333333333333  -0.228425181541148
-0.500000000000000  -0.833333333333333  -0.166666666666667  -0.228425181541148
-0.500000000000000  -0.500000000000000  -0.500000000000000  -0.228425181541148
-0.500000000000000  -0.166666666666667  -0.833333333333333  -0.228425181541148
-0.166666666666667  -0.833333333333333  -0.500000000000000  -0.228425181541148
-0.166666666666667  -0.500000000000000  -0.833333333333333  -0.228425181541148
0.166666666666667  -0.833333333333333  -0.833333333333333  -0.228425181541148
-0.833333333333333  -0.833333333333333  -0.166666666666667  -0.228425181541148
-0.833333333333333  -0.500000000000000  -0.500000000000000  -0.228425181541148
-0.833333333333333  -0.166666666666667  -0.833333333333333  -0.228425181541148
-0.500000000000000  -0.833333333333333  -0.500000000000000  -0.228425181541148
-0.500000000000000  -0.500000000000000  -0.833333333333333  -0.228425181541148
-0.166666666666667  -0.833333333333333  -0.833333333333333  -0.228425181541148
-0.833333333333333  -0.833333333333333  -0.500000000000000  -0.228425181541148
-0.833333333333333  -0.500000000000000  -0.833333333333333  -0.228425181541148
-0.500000000000000  -0.833333333333333  -0.833333333333333  -0.228425181541148
-0.833333333333333  -0.833333333333333  -0.833333333333333  -0.228425181541148
-0.800000000000000  -0.800000000000000  0.400000000000000  0.020258065678861
-0.800000000000000  -0.400000000000000  0.000000000000000  0.020258065678861
-0.800000000000000  0.000000000000000  -0.400000000000000  0.020258065678861
-0.800000000000000  0.400000000000000  -0.800000000000000  0.020258065678861
-0.400000000000000  -0.800000000000000  0.000000000000000  0.020258065678861
-0.400000000000000  -0.400000000000000  -0.400000000000000  0.020258065678861
-0.400000000000000  0.000000000000000  -0.800000000000000  0.020258065678861
0.000000000000000  -0.800000000000000  -0.400000000000000  0.020258065678861
0.000000000000000  -0.400000000000000  -0.800000000000000  0.020258065678861
0.400000000000000  -0.800000000000000  -0.800000000000000  0.020258065678861
-0.800000000000000  -0.800000000000000  0.000000000000000  0.020258065678861
-0.800000000000000  -0.400000000000000  -0.400000000000000  0.020258065678861
-0.800000000000000  0.000000000000000  -0.800000000000000  0.020258065678861
-0.400000000000000  -0.800000000000000  -0.400000000000000  0.020258065678861
-0.400000000000000  -0.400000000000000  -0.800000000000000  0.020258065678861
0.000000000000000  -0.800000000000000  -0.800000000000000  0.020258065678861
-0.800000000000000  -0.800000000000000  -0.400000000000000  0.020258065678861
-0.800000000000000  -0.400000000000000  -0.800000000000000  0.020258065678861
-0.400000000000000  -0.800000000000000  -0.800000000000000  0.020258065678861
-0.800000000000000  -0.800000000000000  -0.800000000000000  0.020258065678861
-0.750000000000000  -0.750000000000000  0.250000000000000  -0.000667313130368
-0.750000000000000  -0.250000000000000  -0.250000000000000  -0.000667313130368
-0.750000000000000  0.250000000000000  -0.750000000000000  -0.000667313130368
-0.250000000000000  -0.750000000000000  -0.250000000000000  -0.000667313130368
-0.250000000000000  -0.250000000000000  -0.750000000000000  -0.000667313130368
0.250000000000000  -0.750000000000000  -0.750000000000000  -0.000667313130368
-0.750000000000000  -0.750000000000000  -0.250000000000000  -0.000667313130368
-0.750000000000000  -0.250000000000000  -0.750000000000000  -0.000667313130368
-0.250000000000000  -0.750000000000000  -0.750000000000000  -0.000667313130368
-0.750000000000000  -0.750000000000000  -0.750000000000000  -0.000667313130368
-0.666666666666667  -0.666666666666667  0.000000000000000  0.000005290478415
-0.666666666666667  0.000000000000000  -0.666666666666667  0.000005290478415
0.000000000000000  -0.666666666666667  -0.666666666666667  0.000005290478415
-0.666666666666667  -0.666666666666667  -0.666666666666667  0.000005290478415
-0.500000000000000  -0.500000000000000  -0.500000000000000  -0.000000003712330
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=19

1001
-0.916666666666667  -0.916666666666667  0.750000000000000  1.186370883531411
-0.916666666666667  -0.750000000000000  0.583333333333333  1.186370883531411
-0.916666666666667  -0.583333333333333  0.416666666666667  1.186370883531411
-0.916666666666667  -0.416666666666667  0.250000000000000  1.186370883531411
-0.916666666666667  -0.250000000000000  0.083333333333333  1.186370883531411
-0.916666666666667  -0.083333333333333  -0.083333333333333  1.186370883531411
-0.916666666666667  0.083333333333333  -0.250000000000000  1.186370883531411
-0.916666666666667  0.250000000000000  -0.416666666666667  1.186370883531411
-0.916666666666667  0.416666666666667  -0.583333333333333  1.186370883531411
-0.916666666666667  0.583333333333333  -0.750000000000000  1.186370883531411
-0.916666666666667  0.750000000000000  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  0.583333333333333  1.186370883531411
-0.750000000000000  -0.750000000000000  0.416666666666667  1.186370883531411
-0.750000000000000  -0.583333333333333  0.250000000000000  1.186370883531411
-0.750000000000000  -0.416666666666667  0.083333333333333  1.186370883531411
-0.750000000000000  -0.250000000000000  -0.083333333333333  1.186370883531411
-0.750000000000000  -0.083333333333333  -0.250000000000000  1.186370883531411
-0.750000000000000  0.083333333333333  -0.416666666666667  1.186370883531411
-0.750000000000000  0.250000000000000  -0.583333333333333  1.186370883531411
-0.750000000000000  0.416666666666667  -0.750000000000000  1.186370883531411
-0.750000000000000  0.583333333333333  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  0.416666666666667  1.186370883531411
-0.583333333333333  -0.750000000000000  0.250000000000000  1.186370883531411
-0.583333333333333  -0.583333333333333  0.083333333333333  1.186370883531411
-0.583333333333333  -0.416666666666667  -0.083333333333333  1.186370883531411
-0.583333333333333  -0.250000000000000  -0.250000000000000  1.186370883531411
-0.583333333333333  -0.083333333333333  -0.416666666666667  1.186370883531411
-0.583333333333333  0.083333333333333  -0.583333333333333  1.186370883531411
-0.583333333333333  0.250000000000000  -0.750000000000000  1.186370883531411
-0.583333333333333  0.416666666666667  -0.916666666666667  1.186370883531411
-0.416666666666667  -0.916666666666667  0.250000000000000  1.186370883531411
-0.416666666666667  -0.750000000000000  0.083333333333333  1.186370883531411
-0.416666666666667  -0.583333333333333  -0.083333333333333  1.186370883531411
-0.416666666666667  -0.416666666666667  -0.250000000000000  1.186370883531411
-0.416666666666667  -0.250000000000000  -0.416666666666667  1.186370883531411
-0.416666666666667  -0.083333333333333  -0.583333333333333  1.186370883531411
-0.416666666666667  0.083333333333333  -0.750000000000000  1.186370883531411
-0.416666666666667  0.250000000000000  -0.916666666666667  1.186370883531411
-0.250000000000000  -0.916666666666667  0.083333333333333  1.186370883531411
-0.250000000000000  -0.750000000000000  -0.083333333333333  1.186370883531411
-0.250000000000000  -0.583333333333333  -0.250000000000000  1.186370883531411
-0.250000000000000  -0.416666666666667  -0.416666666666667  1.186370883531411
-0.250000000000000  -0.250000000000000  -0.583333333333333  1.186370883531411
-0.250000000000000  -0.083333333333333  -0.750000000000000  1.186370883531411
-0.250000000000000  0.083333333333333  -0.916666666666667  1.186370883531411
-0.083333333333333  -0.916666666666667  -0.083333333333333  1.186370883531411
-0.083333333333333  -0.750000000000000  -0.250000000000000  1.186370883531411
-0.083333333333333  -0.583333333333333  -0.416666666666667  1.186370883531411
-0.083333333333333  -0.416666666666667  -0.583333333333333  1.186370883531411
-0.083333333333333  -0.250000000000000  -0.750000000000000  1.186370883531411
-0.083333333333333  -0.083333333333333  -0.916666666666667  1.186370883531411
0.083333333333333  -0.916666666666667  -0.250000000000000  1.186370883531411
0.083333333333333  -0.750000000000000  -0.416666666666667  1.186370883531411
0.083333333333333  -0.583333333333333  -0.583333333333333  1.186370883531411
0.083333333333333  -0.416666666666667  -0.750000000000000  1.186370883531411
0.083333333333333  -0.250000000000000  -0.916666666666667  1.186370883531411
0.250000000000000  -0.916666666666667  -0.416666666666667  1.186370883531411
0.250000000000000  -0.750000000000000  -0.583333333333333  1.186370883531411
0.250000000000000  -0.583333333333333  -0.750000000000000  1.186370883531411
0.250000000000000  -0.416666666666667  -0.916666666666667  1.186370883531411
0.416666666666667  -0.916666666666667  -0.583333333333333  1.186370883531411
0.416666666666667  -0.750000000000000  -0.750000000000000  1.186370883531411
0.416666666666667  -0.583333333333333  -0.916666666666667  1.186370883531411
0.583333333333333  -0.916666666666667  -0.750000000000000  1.186370883531411
0.583333333333333  -0.750000000000000  -0.916666666666667  1.186370883531411
0.750000000000000  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  0.583333333333333  1.186370883531411
-0.916666666666667  -0.750000000000000  0.416666666666667  1.186370883531411
-0.916666666666667  -0.583333333333333  0.250000000000000  1.186370883531411
-0.916666666666667  -0.416666666666667  0.083333333333333  1.186370883531411
-0.916666666666667  -0.250000000000000  -0.083333333333333  1.186370883531411
-0.916666666666667  -0.083333333333333  -0.250000000000000  1.186370883531411
-0.916666666666667  0.083333333333333  -0.416666666666667  1.186370883531411
-0.916666666666667  0.250000000000000  -0.583333333333333  1.186370883531411
-0.916666666666667  0.416666666666667  -0.750000000000000  1.186370883531411
-0.916666666666667  0.583333333333333  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  0.416666666666667  1.186370883531411
-0.750000000000000  -0.750000000000000  0.250000000000000  1.186370883531411
-0.750000000000000  -0.583333333333333  0.083333333333333  1.186370883531411
-0.750000000000000  -0.416666666666667  -0.083333333333333  1.186370883531411
-0.750000000000000  -0.250000000000000  -0.250000000000000  1.186370883531411
-0.750000000000000  -0.083333333333333  -0.416666666666667  1.186370883531411
-0.750000000000000  0.083333333333333  -0.583333333333333  1.186370883531411
-0.750000000000000  0.250000000000000  -0.750000000000000  1.186370883531411
-0.750000000000000  0.416666666666667  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  0.250000000000000  1.186370883531411
-0.583333333333333  -0.750000000000000  0.083333333333333  1.186370883531411
-0.583333333333333  -0.583333333333333  -0.083333333333333  1.186370883531411
-0.583333333333333  -0.416666666666667  -0.250000000000000  1.186370883531411
-0.583333333333333  -0.250000000000000  -0.416666666666667  1.186370883531411
-0.583333333333333  -0.083333333333333  -0.583333333333333  1.186370883531411
-0.583333333333333  0.083333333333333  -0.750000000000000  1.186370883531411
-0.583333333333333  0.250000000000000  -0.916666666666667  1.186370883531411
-0.416666666666667  -0.916666666666667  0.083333333333333  1.186370883531411
-0.416666666666667  -0.750000000000000  -0.083333333333333  1.186370883531411
-0.416666666666667  -0.583333333333333  -0.250000000000000  1.186370883531411
-0.416666666666667  -0.416666666666667  -0.416666666666667  1.186370883531411
-0.416666666666667  -0.250000000000000  -0.583333333333333  1.186370883531411
-0.416666666666667  -0.083333333333333  -0.750000000000000  1.186370883531411
-0.416666666666667  0.083333333333333  -0.916666666666667  1.186370883531411
-0.250000000000000  -0.916666666666667  -0.083333333333333  1.186370883531411
-0.250000000000000  -0.750000000000000  -0.250000000000000  1.186370883531411
-0.250000000000000  -0.583333333333333  -0.416666666666667  1.186370883531411
-0.250000000000000  -0.416666666666667  -0.583333333333333  1.186370883531411
-0.250000000000000  -0.250000000000000  -0.750000000000000  1.186370883531411
-0.250000000000000  -0.083333333333333  -0.916666666666667  1.186370883531411
-0.083333333333333  -0.916666666666667  -0.250000000000000  1.186370883531411
-0.083333333333333  -0.750000000000000  -0.416666666666667  1.186370883531411
-0.083333333333333  -0.583333333333333  -0.583333333333333  1.186370883531411
-0.083333333333333  -0.416666666666667  -0.750000000000000  1.186370883531411
-0.083333333333333  -0.250000000000000  -0.916666666666667  1.186370883531411
0.083333333333333  -0.916666666666667  -0.416666666666667  1.186370883531411
0.083333333333333  -0.750000000000000  -0.583333333333333  1.186370883531411
0.083333333333333  -0.583333333333333  -0.750000000000000  1.186370883531411
0.083333333333333  -0.416666666666667  -0.916666666666667  1.186370883531411
0.250000000000000  -0.916666666666667  -0.583333333333333  1.186370883531411
0.250000000000000  -0.750000000000000  -0.750000000000000  1.186370883531411
0.250000000000000  -0.583333333333333  -0.916666666666667  1.186370883531411
0.416666666666667  -0.916666666666667  -0.750000000000000  1.186370883531411
0.416666666666667  -0.750000000000000  -0.916666666666667  1.186370883531411
0.583333333333333  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  0.416666666666667  1.186370883531411
-0.916666666666667  -0.750000000000000  0.250000000000000  1.186370883531411
-0.916666666666667  -0.583333333333333  0.083333333333333  1.186370883531411
-0.916666666666667  -0.416666666666667  -0.083333333333333  1.186370883531411
-0.916666666666667  -0.250000000000000  -0.250000000000000  1.186370883531411
-0.916666666666667  -0.083333333333333  -0.416666666666667  1.186370883531411
-0.916666666666667  0.083333333333333  -0.583333333333333  1.186370883531411
-0.916666666666667  0.250000000000000  -0.750000000000000  1.186370883531411
-0.916666666666667  0.416666666666667  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  0.250000000000000  1.186370883531411
-0.750000000000000  -0.750000000000000  0.083333333333333  1.186370883531411
-0.750000000000000  -0.583333333333333  -0.083333333333333  1.186370883531411
-0.750000000000000  -0.416666666666667  -0.250000000000000  1.186370883531411
-0.750000000000000  -0.250000000000000  -0.416666666666667  1.186370883531411
-0.750000000000000  -0.083333333333333  -0.583333333333333  1.186370883531411
-0.750000000000000  0.083333333333333  -0.750000000000000  1.186370883531411
-0.750000000000000  0.250000000000000  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  0.083333333333333  1.186370883531411
-0.583333333333333  -0.750000000000000  -0.083333333333333  1.186370883531411
-0.583333333333333  -0.583333333333333  -0.250000000000000  1.186370883531411
-0.583333333333333  -0.416666666666667  -0.416666666666667  1.186370883531411
-0.583333333333333  -0.250000000000000  -0.583333333333333  1.186370883531411
-0.583333333333333  -0.083333333333333  -0.750000000000000  1.186370883531411
-0.583333333333333  0.083333333333333  -0.916666666666667  1.186370883531411
-0.416666666666667  -0.916666666666667  -0.083333333333333  1.186370883531411
-0.416666666666667  -0.750000000000000  -0.250000000000000  1.186370883531411
-0.416666666666667  -0.583333333333333  -0.416666666666667  1.186370883531411
-0.416666666666667  -0.416666666666667  -0.583333333333333  1.186370883531411
-0.416666666666667  -0.250000000000000  -0.750000000000000  1.186370883531411
-0.416666666666667  -0.083333333333333  -0.916666666666667  1.186370883531411
-0.250000000000000  -0.916666666666667  -0.250000000000000  1.186370883531411
-0.250000000000000  -0.750000000000000  -0.416666666666667  1.186370883531411
-0.250000000000000  -0.583333333333333  -0.583333333333333  1.186370883531411
-0.250000000000000  -0.416666666666667  -0.750000000000000  1.186370883531411
-0.250000000000000  -0.250000000000000  -0.916666666666667  1.186370883531411
-0.083333333333333  -0.916666666666667  -0.416666666666667  1.186370883531411
-0.083333333333333  -0.750000000000000  -0.583333333333333  1.186370883531411
-0.083333333333333  -0.583333333333333  -0.750000000000000  1.186370883531411
-0.083333333333333  -0.416666666666667  -0.916666666666667  1.186370883531411
0.083333333333333  -0.916666666666667  -0.583333333333333  1.186370883531411
0.083333333333333  -0.750000000000000  -0.750000000000000  1.186370883531411
0.083333333333333  -0.583333333333333  -0.916666666666667  1.186370883531411
0.250000000000000  -0.916666666666667  -0.750000000000000  1.186370883531411
0.250000000000000  -0.750000000000000  -0.916666666666667  1.186370883531411
0.416666666666667  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  0.250000000000000  1.186370883531411
-0.916666666666667  -0.750000000000000  0.083333333333333  1.186370883531411
-0.916666666666667  -0.583333333333333  -0.083333333333333  1.186370883531411
-0.916666666666667  -0.416666666666667  -0.250000000000000  1.186370883531411
-0.916666666666667  -0.250000000000000  -0.416666666666667  1.186370883531411
-0.916666666666667  -0.083333333333333  -0.583333333333333  1.186370883531411
-0.916666666666667  0.083333333333333  -0.750000000000000  1.186370883531411
-0.916666666666667  0.250000000000000  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  0.083333333333333  1.186370883531411
-0.750000000000000  -0.750000000000000  -0.083333333333333  1.186370883531411
-0.750000000000000  -0.583333333333333  -0.250000000000000  1.186370883531411
-0.750000000000000  -0.416666666666667  -0.416666666666667  1.186370883531411
-0.750000000000000  -0.250000000000000  -0.583333333333333  1.186370883531411
-0.750000000000000  -0.083333333333333  -0.750000000000000  1.186370883531411
-0.750000000000000  0.083333333333333  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  -0.083333333333333  1.186370883531411
-0.583333333333333  -0.750000000000000  -0.250000000000000  1.186370883531411
-0.583333333333333  -0.583333333333333  -0.416666666666667  1.186370883531411
-0.583333333333333  -0.416666666666667  -0.583333333333333  1.186370883531411
-0.583333333333333  -0.250000000000000  -0.750000000000000  1.186370883531411
-0.583333333333333  -0.083333333333333  -0.916666666666667  1.186370883531411
-0.416666666666667  -0.916666666666667  -0.250000000000000  1.186370883531411
-0.416666666666667  -0.750000000000000  -0.416666666666667  1.186370883531411
-0.416666666666667  -0.583333333333333  -0.583333333333333  1.186370883531411
-0.416666666666667  -0.416666666666667  -0.750000000000000  1.186370883531411
-0.416666666666667  -0.250000000000000  -0.916666666666667  1.186370883531411
-0.250000000000000  -0.916666666666667  -0.416666666666667  1.186370883531411
-0.250000000000000  -0.750000000000000  -0.583333333333333  1.186370883531411
-0.250000000000000  -0.583333333333333  -0.750000000000000  1.186370883531411
-0.250000000000000  -0.416666666666667  -0.916666666666667  1.186370883531411
-0.083333333333333  -0.916666666666667  -0.583333333333333  1.186370883531411
-0.083333333333333  -0.750000000000000  -0.750000000000000  1.186370883531411
-0.083333333333333  -0.583333333333333  -0.916666666666667  1.186370883531411
0.083333333333333  -0.916666666666667  -0.750000000000000  1.186370883531411
0.083333333333333  -0.750000000000000  -0.916666666666667  1.186370883531411
0.250000000000000  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  0.083333333333333  1.186370883531411
-0.916666666666667  -0.750000000000000  -0.083333333333333  1.186370883531411
-0.916666666666667  -0.583333333333333  -0.250000000000000  1.186370883531411
-0.916666666666667  -0.416666666666667  -0.416666666666667  1.186370883531411
-0.916666666666667  -0.250000000000000  -0.583333333333333  1.186370883531411
-0.916666666666667  -0.083333333333333  -0.750000000000000  1.186370883531411
-0.916666666666667  0.083333333333333  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  -0.083333333333333  1.186370883531411
-0.750000000000000  -0.750000000000000  -0.250000000000000  1.186370883531411
-0.750000000000000  -0.583333333333333  -0.416666666666667  1.186370883531411
-0.750000000000000  -0.416666666666667  -0.583333333333333  1.186370883531411
-0.750000000000000  -0.250000000000000  -0.750000000000000  1.186370883531411
-0.750000000000000  -0.083333333333333  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  -0.250000000000000  1.186370883531411
-0.583333333333333  -0.750000000000000  -0.416666666666667  1.186370883531411
-0.583333333333333  -0.583333333333333  -0.583333333333333  1.186370883531411
-0.583333333333333  -0.416666666666667  -0.750000000000000  1.186370883531411
-0.583333333333333  -0.250000000000000  -0.916666666666667  1.186370883531411
-0.416666666666667  -0.916666666666667  -0.416666666666667  1.186370883531411
-0.416666666666667  -0.750000000000000  -0.583333333333333  1.186370883531411
-0.416666666666667  -0.583333333333333  -0.750000000000000  1.186370883531411
-0.416666666666667  -0.416666666666667  -0.916666666666667  1.186370883531411
-0.250000000000000  -0.916666666666667  -0.583333333333333  1.186370883531411
-0.250000000000000  -0.750000000000000  -0.750000000000000  1.186370883531411
-0.250000000000000  -0.583333333333333  -0.916666666666667  1.186370883531411
-0.083333333333333  -0.916666666666667  -0.750000000000000  1.186370883531411
-0.083333333333333  -0.750000000000000  -0.916666666666667  1.186370883531411
0.083333333333333  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  -0.083333333333333  1.186370883531411
-0.916666666666667  -0.750000000000000  -0.250000000000000  1.186370883531411
-0.916666666666667  -0.583333333333333  -0.416666666666667  1.186370883531411
-0.916666666666667  -0.416666666666667  -0.583333333333333  1.186370883531411
-0.916666666666667  -0.250000000000000  -0.750000000000000  1.186370883531411
-0.916666666666667  -0.083333333333333  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  -0.250000000000000  1.186370883531411
-0.750000000000000  -0.750000000000000  -0.416666666666667  1.186370883531411
-0.750000000000000  -0.583333333333333  -0.583333333333333  1.186370883531411
-0.750000000000000  -0.416666666666667  -0.750000000000000  1.186370883531411
-0.750000000000000  -0.250000000000000  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  -0.416666666666667  1.186370883531411
-0.583333333333333  -0.750000000000000  -0.583333333333333  1.186370883531411
-0.583333333333333  -0.583333333333333  -0.750000000000000  1.186370883531411
-0.583333333333333  -0.416666666666667  -0.916666666666667  1.186370883531411
-0.416666666666667  -0.916666666666667  -0.583333333333333  1.186370883531411
-0.416666666666667  -0.750000000000000  -0.750000000000000  1.186370883531411
-0.416666666666667  -0.583333333333333  -0.916666666666667  1.186370883531411
-0.250000000000000  -0.916666666666667  -0.750000000000000  1.186370883531411
-0.250000000000000  -0.750000000000000  -0.916666666666667  1.186370883531411
-0.083333333333333  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  -0.250000000000000  1.186370883531411
-0.916666666666667  -0.750000000000000  -0.416666666666667  1.186370883531411
-0.916666666666667  -0.583333333333333  -0.583333333333333  1.186370883531411
-0.916666666666667  -0.416666666666667  -0.750000000000000  1.186370883531411
-0.916666666666667  -0.250000000000000  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  -0.416666666666667  1.186370883531411
-0.750000000000000  -0.750000000000000  -0.583333333333333  1.186370883531411
-0.750000000000000  -0.583333333333333  -0.750000000000000  1.186370883531411
-0.750000000000000  -0.416666666666667  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  -0.583333333333333  1.186370883531411
-0.583333333333333  -0.750000000000000  -0.750000000000000  1.186370883531411
-0.583333333333333  -0.583333333333333  -0.916666666666667  1.186370883531411
-0.416666666666667  -0.916666666666667  -0.750000000000000  1.186370883531411
-0.416666666666667  -0.750000000000000  -0.916666666666667  1.186370883531411
-0.250000000000000  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  -0.416666666666667  1.186370883531411
-0.916666666666667  -0.750000000000000  -0.583333333333333  1.186370883531411
-0.916666666666667  -0.583333333333333  -0.750000000000000  1.186370883531411
-0.916666666666667  -0.416666666666667  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  -0.583333333333333  1.186370883531411
-0.750000000000000  -0.750000000000000  -0.750000000000000  1.186370883531411
-0.750000000000000  -0.583333333333333  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  -0.750000000000000  1.186370883531411
-0.583333333333333  -0.750000000000000  -0.916666666666667  1.186370883531411
-0.416666666666667  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  -0.583333333333333  1.186370883531411
-0.916666666666667  -0.750000000000000  -0.750000000000000  1.186370883531411
-0.916666666666667  -0.583333333333333  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  -0.750000000000000  1.186370883531411
-0.750000000000000  -0.750000000000000  -0.916666666666667  1.186370883531411
-0.583333333333333  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  -0.750000000000000  1.186370883531411
-0.916666666666667  -0.750000000000000  -0.916666666666667  1.186370883531411
-0.750000000000000  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.916666666666667  -0.916666666666667  -0.916666666666667  1.186370883531411
-0.909090909090909  -0.909090909090909  0.727272727272727  -4.580068174334528
-0.909090909090909  -0.727272727272727  0.545454545454545  -4.580068174334528
-0.909090909090909  -0.545454545454545  0.363636363636364  -4.580068174334528
-0.909090909090909  -0.363636363636364  0.181818181818182  -4.580068174334528
-0.909090909090909  -0.181818181818182  0.000000000000000  -4.580068174334528
-0.909090909090909  0.000000000000000  -0.181818181818182  -4.580068174334528
-0.909090909090909  0.181818181818182  -0.363636363636364  -4.580068174334528
-0.909090909090909  0.363636363636364  -0.545454545454545  -4.580068174334528
-0.909090909090909  0.545454545454545  -0.727272727272727  -4.580068174334528
-0.909090909090909  0.727272727272727  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  0.545454545454545  -4.580068174334528
-0.727272727272727  -0.727272727272727  0.363636363636364  -4.580068174334528
-0.727272727272727  -0.545454545454545  0.181818181818182  -4.580068174334528
-0.727272727272727  -0.363636363636364  0.000000000000000  -4.580068174334528
-0.727272727272727  -0.181818181818182  -0.181818181818182  -4.580068174334528
-0.727272727272727  0.000000000000000  -0.363636363636364  -4.580068174334528
-0.727272727272727  0.181818181818182  -0.545454545454545  -4.580068174334528
-0.727272727272727  0.363636363636364  -0.727272727272727  -4.580068174334528
-0.727272727272727  0.545454545454545  -0.909090909090909  -4.580068174334528
-0.545454545454545  -0.909090909090909  0.363636363636364  -4.580068174334528
-0.545454545454545  -0.727272727272727  0.181818181818182  -4.580068174334528
-0.545454545454545  -0.545454545454545  0.000000000000000  -4.580068174334528
-0.545454545454545  -0.363636363636364  -0.181818181818182  -4.580068174334528
-0.545454545454545  -0.181818181818182  -0.363636363636364  -4.580068174334528
-0.545454545454545  0.000000000000000  -0.545454545454545  -4.580068174334528
-0.545454545454545  0.181818181818182  -0.727272727272727  -4.580068174334528
-0.545454545454545  0.363636363636364  -0.909090909090909  -4.580068174334528
-0.363636363636364  -0.909090909090909  0.181818181818182  -4.580068174334528
-0.363636363636364  -0.727272727272727  0.000000000000000  -4.580068174334528
-0.363636363636364  -0.545454545454545  -0.181818181818182  -4.580068174334528
-0.363636363636364  -0.363636363636364  -0.363636363636364  -4.580068174334528
-0.363636363636364  -0.181818181818182  -0.545454545454545  -4.580068174334528
-0.363636363636364  0.000000000000000  -0.727272727272727  -4.580068174334528
-0.363636363636364  0.181818181818182  -0.909090909090909  -4.580068174334528
-0.181818181818182  -0.909090909090909  0.000000000000000  -4.580068174334528
-0.181818181818182  -0.727272727272727  -0.181818181818182  -4.580068174334528
-0.181818181818182  -0.545454545454545  -0.363636363636364  -4.580068174334528
-0.181818181818182  -0.363636363636364  -0.545454545454545  -4.580068174334528
-0.181818181818182  -0.181818181818182  -0.727272727272727  -4.580068174334528
-0.181818181818182  0.000000000000000  -0.909090909090909  -4.580068174334528
0.000000000000000  -0.909090909090909  -0.181818181818182  -4.580068174334528
0.000000000000000  -0.727272727272727  -0.363636363636364  -4.580068174334528
0.000000000000000  -0.545454545454545  -0.545454545454545  -4.580068174334528
0.000000000000000  -0.363636363636364  -0.727272727272727  -4.580068174334528
0.000000000000000  -0.181818181818182  -0.909090909090909  -4.580068174334528
0.181818181818182  -0.909090909090909  -0.363636363636364  -4.580068174334528
0.181818181818182  -0.727272727272727  -0.545454545454545  -4.580068174334528
0.181818181818182  -0.545454545454545  -0.727272727272727  -4.580068174334528
0.181818181818182  -0.363636363636364  -0.909090909090909  -4.580068174334528
0.363636363636364  -0.909090909090909  -0.545454545454545  -4.580068174334528
0.363636363636364  -0.727272727272727  -0.727272727272727  -4.580068174334528
0.363636363636364  -0.545454545454545  -0.909090909090909  -4.580068174334528
0.545454545454545  -0.909090909090909  -0.727272727272727  -4.580068174334528
0.545454545454545  -0.727272727272727  -0.909090909090909  -4.580068174334528
0.727272727272727  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  0.545454545454545  -4.580068174334528
-0.909090909090909  -0.727272727272727  0.363636363636364  -4.580068174334528
-0.909090909090909  -0.545454545454545  0.181818181818182  -4.580068174334528
-0.909090909090909  -0.363636363636364  0.000000000000000  -4.580068174334528
-0.909090909090909  -0.181818181818182  -0.181818181818182  -4.580068174334528
-0.909090909090909  0.000000000000000  -0.363636363636364  -4.580068174334528
-0.909090909090909  0.181818181818182  -0.545454545454545  -4.580068174334528
-0.909090909090909  0.363636363636364  -0.727272727272727  -4.580068174334528
-0.909090909090909  0.545454545454545  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  0.363636363636364  -4.580068174334528
-0.727272727272727  -0.727272727272727  0.181818181818182  -4.580068174334528
-0.727272727272727  -0.545454545454545  0.000000000000000  -4.580068174334528
-0.727272727272727  -0.363636363636364  -0.181818181818182  -4.580068174334528
-0.727272727272727  -0.181818181818182  -0.363636363636364  -4.580068174334528
-0.727272727272727  0.000000000000000  -0.545454545454545  -4.580068174334528
-0.727272727272727  0.181818181818182  -0.727272727272727  -4.580068174334528
-0.727272727272727  0.363636363636364  -0.909090909090909  -4.580068174334528
-0.545454545454545  -0.909090909090909  0.181818181818182  -4.580068174334528
-0.545454545454545  -0.727272727272727  0.000000000000000  -4.580068174334528
-0.545454545454545  -0.545454545454545  -0.181818181818182  -4.580068174334528
-0.545454545454545  -0.363636363636364  -0.363636363636364  -4.580068174334528
-0.545454545454545  -0.181818181818182  -0.545454545454545  -4.580068174334528
-0.545454545454545  0.000000000000000  -0.727272727272727  -4.580068174334528
-0.545454545454545  0.181818181818182  -0.909090909090909  -4.580068174334528
-0.363636363636364  -0.909090909090909  0.000000000000000  -4.580068174334528
-0.363636363636364  -0.727272727272727  -0.181818181818182  -4.580068174334528
-0.363636363636364  -0.545454545454545  -0.363636363636364  -4.580068174334528
-0.363636363636364  -0.363636363636364  -0.545454545454545  -4.580068174334528
-0.363636363636364  -0.181818181818182  -0.727272727272727  -4.580068174334528
-0.363636363636364  0.000000000000000  -0.909090909090909  -4.580068174334528
-0.181818181818182  -0.909090909090909  -0.181818181818182  -4.580068174334528
-0.181818181818182  -0.727272727272727  -0.363636363636364  -4.580068174334528
-0.181818181818182  -0.545454545454545  -0.545454545454545  -4.580068174334528
-0.181818181818182  -0.363636363636364  -0.727272727272727  -4.580068174334528
-0.181818181818182  -0.181818181818182  -0.909090909090909  -4.580068174334528
0.000000000000000  -0.909090909090909  -0.363636363636364  -4.580068174334528
0.000000000000000  -0.727272727272727  -0.545454545454545  -4.580068174334528
0.000000000000000  -0.545454545454545  -0.727272727272727  -4.580068174334528
0.000000000000000  -0.363636363636364  -0.909090909090909  -4.580068174334528
0.181818181818182  -0.909090909090909  -0.545454545454545  -4.580068174334528
0.181818181818182  -0.727272727272727  -0.727272727272727  -4.580068174334528
0.181818181818182  -0.545454545454545  -0.909090909090909  -4.580068174334528
0.363636363636364  -0.909090909090909  -0.727272727272727  -4.580068174334528
0.363636363636364  -0.727272727272727  -0.909090909090909  -4.580068174334528
0.545454545454545  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  0.363636363636364  -4.580068174334528
-0.909090909090909  -0.727272727272727  0.181818181818182  -4.580068174334528
-0.909090909090909  -0.545454545454545  0.000000000000000  -4.580068174334528
-0.909090909090909  -0.363636363636364  -0.181818181818182  -4.580068174334528
-0.909090909090909  -0.181818181818182  -0.363636363636364  -4.580068174334528
-0.909090909090909  0.000000000000000  -0.545454545454545  -4.580068174334528
-0.909090909090909  0.181818181818182  -0.727272727272727  -4.580068174334528
-0.909090909090909  0.363636363636364  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  0.181818181818182  -4.580068174334528
-0.727272727272727  -0.727272727272727  0.000000000000000  -4.580068174334528
-0.727272727272727  -0.545454545454545  -0.181818181818182  -4.580068174334528
-0.727272727272727  -0.363636363636364  -0.363636363636364  -4.580068174334528
-0.727272727272727  -0.181818181818182  -0.545454545454545  -4.580068174334528
-0.727272727272727  0.000000000000000  -0.727272727272727  -4.580068174334528
-0.727272727272727  0.181818181818182  -0.909090909090909  -4.580068174334528
-0.545454545454545  -0.909090909090909  0.000000000000000  -4.580068174334528
-0.545454545454545  -0.727272727272727  -0.181818181818182  -4.580068174334528
-0.545454545454545  -0.545454545454545  -0.363636363636364  -4.580068174334528
-0.545454545454545  -0.363636363636364  -0.545454545454545  -4.580068174334528
-0.545454545454545  -0.181818181818182  -0.727272727272727  -4.580068174334528
-0.545454545454545  0.000000000000000  -0.909090909090909  -4.580068174334528
-0.363636363636364  -0.909090909090909  -0.181818181818182  -4.580068174334528
-0.363636363636364  -0.727272727272727  -0.363636363636364  -4.580068174334528
-0.363636363636364  -0.545454545454545  -0.545454545454545  -4.580068174334528
-0.363636363636364  -0.363636363636364  -0.727272727272727  -4.580068174334528
-0.363636363636364  -0.181818181818182  -0.909090909090909  -4.580068174334528
-0.181818181818182  -0.909090909090909  -0.363636363636364  -4.580068174334528
-0.181818181818182  -0.727272727272727  -0.545454545454545  -4.580068174334528
-0.181818181818182  -0.545454545454545  -0.727272727272727  -4.580068174334528
-0.181818181818182  -0.363636363636364  -0.909090909090909  -4.580068174334528
0.000000000000000  -0.909090909090909  -0.545454545454545  -4.580068174334528
0.000000000000000  -0.727272727272727  -0.727272727272727  -4.580068174334528
0.000000000000000  -0.545454545454545  -0.909090909090909  -4.580068174334528
0.181818181818182  -0.909090909090909  -0.727272727272727  -4.580068174334528
0.181818181818182  -0.727272727272727  -0.909090909090909  -4.580068174334528
0.363636363636364  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  0.181818181818182  -4.580068174334528
-0.909090909090909  -0.727272727272727  0.000000000000000  -4.580068174334528
-0.909090909090909  -0.545454545454545  -0.181818181818182  -4.580068174334528
-0.909090909090909  -0.363636363636364  -0.363636363636364  -4.580068174334528
-0.909090909090909  -0.181818181818182  -0.545454545454545  -4.580068174334528
-0.909090909090909  0.000000000000000  -0.727272727272727  -4.580068174334528
-0.909090909090909  0.181818181818182  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  0.000000000000000  -4.580068174334528
-0.727272727272727  -0.727272727272727  -0.181818181818182  -4.580068174334528
-0.727272727272727  -0.545454545454545  -0.363636363636364  -4.580068174334528
-0.727272727272727  -0.363636363636364  -0.545454545454545  -4.580068174334528
-0.727272727272727  -0.181818181818182  -0.727272727272727  -4.580068174334528
-0.727272727272727  0.000000000000000  -0.909090909090909  -4.580068174334528
-0.545454545454545  -0.909090909090909  -0.181818181818182  -4.580068174334528
-0.545454545454545  -0.727272727272727  -0.363636363636364  -4.580068174334528
-0.545454545454545  -0.545454545454545  -0.545454545454545  -4.580068174334528
-0.545454545454545  -0.363636363636364  -0.727272727272727  -4.580068174334528
-0.545454545454545  -0.181818181818182  -0.909090909090909  -4.580068174334528
-0.363636363636364  -0.909090909090909  -0.363636363636364  -4.580068174334528
-0.363636363636364  -0.727272727272727  -0.545454545454545  -4.580068174334528
-0.363636363636364  -0.545454545454545  -0.727272727272727  -4.580068174334528
-0.363636363636364  -0.363636363636364  -0.909090909090909  -4.580068174334528
-0.181818181818182  -0.909090909090909  -0.545454545454545  -4.580068174334528
-0.181818181818182  -0.727272727272727  -0.727272727272727  -4.580068174334528
-0.181818181818182  -0.545454545454545  -0.909090909090909  -4.580068174334528
0.000000000000000  -0.909090909090909  -0.727272727272727  -4.580068174334528
0.000000000000000  -0.727272727272727  -0.909090909090909  -4.580068174334528
0.181818181818182  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  0.000000000000000  -4.580068174334528
-0.909090909090909  -0.727272727272727  -0.181818181818182  -4.580068174334528
-0.909090909090909  -0.545454545454545  -0.363636363636364  -4.580068174334528
-0.909090909090909  -0.363636363636364  -0.545454545454545  -4.580068174334528
-0.909090909090909  -0.181818181818182  -0.727272727272727  -4.580068174334528
-0.909090909090909  0.000000000000000  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  -0.181818181818182  -4.580068174334528
-0.727272727272727  -0.727272727272727  -0.363636363636364  -4.580068174334528
-0.727272727272727  -0.545454545454545  -0.545454545454545  -4.580068174334528
-0.727272727272727  -0.363636363636364  -0.727272727272727  -4.580068174334528
-0.727272727272727  -0.181818181818182  -0.909090909090909  -4.580068174334528
-0.545454545454545  -0.909090909090909  -0.363636363636364  -4.580068174334528
-0.545454545454545  -0.727272727272727  -0.545454545454545  -4.580068174334528
-0.545454545454545  -0.545454545454545  -0.727272727272727  -4.580068174334528
-0.545454545454545  -0.363636363636364  -0.909090909090909  -4.580068174334528
-0.363636363636364  -0.909090909090909  -0.545454545454545  -4.580068174334528
-0.363636363636364  -0.727272727272727  -0.727272727272727  -4.580068174334528
-0.363636363636364  -0.545454545454545  -0.909090909090909  -4.580068174334528
-0.181818181818182  -0.909090909090909  -0.727272727272727  -4.580068174334528
-0.181818181818182  -0.727272727272727  -0.909090909090909  -4.580068174334528
0.000000000000000  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  -0.181818181818182  -4.580068174334528
-0.909090909090909  -0.727272727272727  -0.363636363636364  -4.580068174334528
-0.909090909090909  -0.545454545454545  -0.545454545454545  -4.580068174334528
-0.909090909090909  -0.363636363636364  -0.727272727272727  -4.580068174334528
-0.909090909090909  -0.181818181818182  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  -0.363636363636364  -4.580068174334528
-0.727272727272727  -0.727272727272727  -0.545454545454545  -4.580068174334528
-0.727272727272727  -0.545454545454545  -0.727272727272727  -4.580068174334528
-0.727272727272727  -0.363636363636364  -0.909090909090909  -4.580068174334528
-0.545454545454545  -0.909090909090909  -0.545454545454545  -4.580068174334528
-0.545454545454545  -0.727272727272727  -0.727272727272727  -4.580068174334528
-0.545454545454545  -0.545454545454545  -0.909090909090909  -4.580068174334528
-0.363636363636364  -0.909090909090909  -0.727272727272727  -4.580068174334528
-0.363636363636364  -0.727272727272727  -0.909090909090909  -4.580068174334528
-0.181818181818182  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  -0.363636363636364  -4.580068174334528
-0.909090909090909  -0.727272727272727  -0.545454545454545  -4.580068174334528
-0.909090909090909  -0.545454545454545  -0.727272727272727  -4.580068174334528
-0.909090909090909  -0.363636363636364  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  -0.545454545454545  -4.580068174334528
-0.727272727272727  -0.727272727272727  -0.727272727272727  -4.580068174334528
-0.727272727272727  -0.545454545454545  -0.909090909090909  -4.580068174334528
-0.545454545454545  -0.909090909090909  -0.727272727272727  -4.580068174334528
-0.545454545454545  -0.727272727272727  -0.909090909090909  -4.580068174334528
-0.363636363636364  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  -0.545454545454545  -4.580068174334528
-0.909090909090909  -0.727272727272727  -0.727272727272727  -4.580068174334528
-0.909090909090909  -0.545454545454545  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  -0.727272727272727  -4.580068174334528
-0.727272727272727  -0.727272727272727  -0.909090909090909  -4.580068174334528
-0.545454545454545  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  -0.727272727272727  -4.580068174334528
-0.909090909090909  -0.727272727272727  -0.909090909090909  -4.580068174334528
-0.727272727272727  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.909090909090909  -0.909090909090909  -0.909090909090909  -4.580068174334528
-0.900000000000000  -0.900000000000000  0.700000000000000  7.117433113960459
-0.900000000000000  -0.700000000000000  0.500000000000000  7.117433113960459
-0.900000000000000  -0.500000000000000  0.300000000000000  7.117433113960459
-0.900000000000000  -0.300000000000000  0.100000000000000  7.117433113960459
-0.900000000000000  -0.100000000000000  -0.100000000000000  7.117433113960459
-0.900000000000000  0.100000000000000  -0.300000000000000  7.117433113960459
-0.900000000000000  0.300000000000000  -0.500000000000000  7.117433113960459
-0.900000000000000  0.500000000000000  -0.700000000000000  7.117433113960459
-0.900000000000000  0.700000000000000  -0.900000000000000  7.117433113960459
-0.700000000000000  -0.900000000000000  0.500000000000000  7.117433113960459
-0.700000000000000  -0.700000000000000  0.300000000000000  7.117433113960459
-0.700000000000000  -0.500000000000000  0.100000000000000  7.117433113960459
-0.700000000000000  -0.300000000000000  -0.100000000000000  7.117433113960459
-0.700000000000000  -0.100000000000000  -0.300000000000000  7.117433113960459
-0.700000000000000  0.100000000000000  -0.500000000000000  7.117433113960459
-0.700000000000000  0.300000000000000  -0.700000000000000  7.117433113960459
-0.700000000000000  0.500000000000000  -0.900000000000000  7.117433113960459
-0.500000000000000  -0.900000000000000  0.300000000000000  7.117433113960459
-0.500000000000000  -0.700000000000000  0.100000000000000  7.117433113960459
-0.500000000000000  -0.500000000000000  -0.100000000000000  7.117433113960459
-0.500000000000000  -0.300000000000000  -0.300000000000000  7.117433113960459
-0.500000000000000  -0.100000000000000  -0.500000000000000  7.117433113960459
-0.500000000000000  0.100000000000000  -0.700000000000000  7.117433113960459
-0.500000000000000  0.300000000000000  -0.900000000000000  7.117433113960459
-0.300000000000000  -0.900000000000000  0.100000000000000  7.117433113960459
-0.300000000000000  -0.700000000000000  -0.100000000000000  7.117433113960459
-0.300000000000000  -0.500000000000000  -0.300000000000000  7.117433113960459
-0.300000000000000  -0.300000000000000  -0.500000000000000  7.117433113960459
-0.300000000000000  -0.100000000000000  -0.700000000000000  7.117433113960459
-0.300000000000000  0.100000000000000  -0.900000000000000  7.117433113960459
-0.100000000000000  -0.900000000000000  -0.100000000000000  7.117433113960459
-0.100000000000000  -0.700000000000000  -0.300000000000000  7.117433113960459
-0.100000000000000  -0.500000000000000  -0.500000000000000  7.117433113960459
-0.100000000000000  -0.300000000000000  -0.700000000000000  7.117433113960459
-0.100000000000000  -0.100000000000000  -0.900000000000000  7.117433113960459
0.100000000000000  -0.900000000000000  -0.300000000000000  7.117433113960459
0.100000000000000  -0.700000000000000  -0.500000000000000  7.117433113960459
0.100000000000000  -0.500000000000000  -0.700000000000000  7.117433113960459
0.100000000000000  -0.300000000000000  -0.900000000000000  7.117433113960459
0.300000000000000  -0.900000000000000  -0.500000000000000  7.117433113960459
0.300000000000000  -0.700000000000000  -0.700000000000000  7.117433113960459
0.300000000000000  -0.500000000000000  -0.900000000000000  7.117433113960459
0.500000000000000  -0.900000000000000  -0.700000000000000  7.117433113960459
0.500000000000000  -0.700000000000000  -0.900000000000000  7.117433113960459
0.700000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.900000000000000  -0.900000000000000  0.500000000000000  7.117433113960459
-0.900000000000000  -0.700000000000000  0.300000000000000  7.117433113960459
-0.900000000000000  -0.500000000000000  0.100000000000000  7.117433113960459
-0.900000000000000  -0.300000000000000  -0.100000000000000  7.117433113960459
-0.900000000000000  -0.100000000000000  -0.300000000000000  7.117433113960459
-0.900000000000000  0.100000000000000  -0.500000000000000  7.117433113960459
-0.900000000000000  0.300000000000000  -0.700000000000000  7.117433113960459
-0.900000000000000  0.500000000000000  -0.900000000000000  7.117433113960459
-0.700000000000000  -0.900000000000000  0.300000000000000  7.117433113960459
-0.700000000000000  -0.700000000000000  0.100000000000000  7.117433113960459
-0.700000000000000  -0.500000000000000  -0.100000000000000  7.117433113960459
-0.700000000000000  -0.300000000000000  -0.300000000000000  7.117433113960459
-0.700000000000000  -0.100000000000000  -0.500000000000000  7.117433113960459
-0.700000000000000  0.100000000000000  -0.700000000000000  7.117433113960459
-0.700000000000000  0.300000000000000  -0.900000000000000  7.117433113960459
-0.500000000000000  -0.900000000000000  0.100000000000000  7.117433113960459
-0.500000000000000  -0.700000000000000  -0.100000000000000  7.117433113960459
-0.500000000000000  -0.500000000000000  -0.300000000000000  7.117433113960459
-0.500000000000000  -0.300000000000000  -0.500000000000000  7.117433113960459
-0.500000000000000  -0.100000000000000  -0.700000000000000  7.117433113960459
-0.500000000000000  0.100000000000000  -0.900000000000000  7.117433113960459
-0.300000000000000  -0.900000000000000  -0.100000000000000  7.117433113960459
-0.300000000000000  -0.700000000000000  -0.300000000000000  7.117433113960459
-0.300000000000000  -0.500000000000000  -0.500000000000000  7.117433113960459
-0.300000000000000  -0.300000000000000  -0.700000000000000  7.117433113960459
-0.300000000000000  -0.100000000000000  -0.900000000000000  7.117433113960459
-0.100000000000000  -0.900000000000000  -0.300000000000000  7.117433113960459
-0.100000000000000  -0.700000000000000  -0.500000000000000  7.117433113960459
-0.100000000000000  -0.500000000000000  -0.700000000000000  7.117433113960459
-0.100000000000000  -0.300000000000000  -0.900000000000000  7.117433113960459
0.100000000000000  -0.900000000000000  -0.500000000000000  7.117433113960459
0.100000000000000  -0.700000000000000  -0.700000000000000  7.117433113960459
0.100000000000000  -0.500000000000000  -0.900000000000000  7.117433113960459
0.300000000000000  -0.900000000000000  -0.700000000000000  7.117433113960459
0.300000000000000  -0.700000000000000  -0.900000000000000  7.117433113960459
0.500000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.900000000000000  -0.900000000000000  0.300000000000000  7.117433113960459
-0.900000000000000  -0.700000000000000  0.100000000000000  7.117433113960459
-0.900000000000000  -0.500000000000000  -0.100000000000000  7.117433113960459
-0.900000000000000  -0.300000000000000  -0.300000000000000  7.117433113960459
-0.900000000000000  -0.100000000000000  -0.500000000000000  7.117433113960459
-0.900000000000000  0.100000000000000  -0.700000000000000  7.117433113960459
-0.900000000000000  0.300000000000000  -0.900000000000000  7.117433113960459
-0.700000000000000  -0.900000000000000  0.100000000000000  7.117433113960459
-0.700000000000000  -0.700000000000000  -0.100000000000000  7.117433113960459
-0.700000000000000  -0.500000000000000  -0.300000000000000  7.117433113960459
-0.700000000000000  -0.300000000000000  -0.500000000000000  7.117433113960459
-0.700000000000000  -0.100000000000000  -0.700000000000000  7.117433113960459
-0.700000000000000  0.100000000000000  -0.900000000000000  7.117433113960459
-0.500000000000000  -0.900000000000000  -0.100000000000000  7.117433113960459
-0.500000000000000  -0.700000000000000  -0.300000000000000  7.117433113960459
-0.500000000000000  -0.500000000000000  -0.500000000000000  7.117433113960459
-0.500000000000000  -0.300000000000000  -0.700000000000000  7.117433113960459
-0.500000000000000  -0.100000000000000  -0.900000000000000  7.117433113960459
-0.300000000000000  -0.900000000000000  -0.300000000000000  7.117433113960459
-0.300000000000000  -0.700000000000000  -0.500000000000000  7.117433113960459
-0.300000000000000  -0.500000000000000  -0.700000000000000  7.117433113960459
-0.300000000000000  -0.300000000000000  -0.900000000000000  7.117433113960459
-0.100000000000000  -0.900000000000000  -0.500000000000000  7.117433113960459
-0.100000000000000  -0.700000000000000  -0.700000000000000  7.117433113960459
-0.100000000000000  -0.500000000000000  -0.900000000000000  7.117433113960459
0.100000000000000  -0.900000000000000  -0.700000000000000  7.117433113960459
0.100000000000000  -0.700000000000000  -0.900000000000000  7.117433113960459
0.300000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.900000000000000  -0.900000000000000  0.100000000000000  7.117433113960459
-0.900000000000000  -0.700000000000000  -0.100000000000000  7.117433113960459
-0.900000000000000  -0.500000000000000  -0.300000000000000  7.117433113960459
-0.900000000000000  -0.300000000000000  -0.500000000000000  7.117433113960459
-0.900000000000000  -0.100000000000000  -0.700000000000000  7.117433113960459
-0.900000000000000  0.100000000000000  -0.900000000000000  7.117433113960459
-0.700000000000000  -0.900000000000000  -0.100000000000000  7.117433113960459
-0.700000000000000  -0.700000000000000  -0.300000000000000  7.117433113960459
-0.700000000000000  -0.500000000000000  -0.500000000000000  7.117433113960459
-0.700000000000000  -0.300000000000000  -0.700000000000000  7.117433113960459
-0.700000000000000  -0.100000000000000  -0.900000000000000  7.117433113960459
-0.500000000000000  -0.900000000000000  -0.300000000000000  7.117433113960459
-0.500000000000000  -0.700000000000000  -0.500000000000000  7.117433113960459
-0.500000000000000  -0.500000000000000  -0.700000000000000  7.117433113960459
-0.500000000000000  -0.300000000000000  -0.900000000000000  7.117433113960459
-0.300000000000000  -0.900000000000000  -0.500000000000000  7.117433113960459
-0.300000000000000  -0.700000000000000  -0.700000000000000  7.117433113960459
-0.300000000000000  -0.500000000000000  -0.900000000000000  7.117433113960459
-0.100000000000000  -0.900000000000000  -0.700000000000000  7.117433113960459
-0.100000000000000  -0.700000000000000  -0.900000000000000  7.117433113960459
0.100000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.900000000000000  -0.900000000000000  -0.100000000000000  7.117433113960459
-0.900000000000000  -0.700000000000000  -0.300000000000000  7.117433113960459
-0.900000000000000  -0.500000000000000  -0.500000000000000  7.117433113960459
-0.900000000000000  -0.300000000000000  -0.700000000000000  7.117433113960459
-0.900000000000000  -0.100000000000000  -0.900000000000000  7.117433113960459
-0.700000000000000  -0.900000000000000  -0.300000000000000  7.117433113960459
-0.700000000000000  -0.700000000000000  -0.500000000000000  7.117433113960459
-0.700000000000000  -0.500000000000000  -0.700000000000000  7.117433113960459
-0.700000000000000  -0.300000000000000  -0.900000000000000  7.117433113960459
-0.500000000000000  -0.900000000000000  -0.500000000000000  7.117433113960459
-0.500000000000000  -0.700000000000000  -0.700000000000000  7.117433113960459
-0.500000000000000  -0.500000000000000  -0.900000000000000  7.117433113960459
-0.300000000000000  -0.900000000000000  -0.700000000000000  7.117433113960459
-0.300000000000000  -0.700000000000000  -0.900000000000000  7.117433113960459
-0.100000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.900000000000000  -0.900000000000000  -0.300000000000000  7.117433113960459
-0.900000000000000  -0.700000000000000  -0.500000000000000  7.117433113960459
-0.900000000000000  -0.500000000000000  -0.700000000000000  7.117433113960459
-0.900000000000000  -0.300000000000000  -0.900000000000000  7.117433113960459
-0.700000000000000  -0.900000000000000  -0.500000000000000  7.117433113960459
-0.700000000000000  -0.700000000000000  -0.700000000000000  7.117433113960459
-0.700000000000000  -0.500000000000000  -0.900000000000000  7.117433113960459
-0.500000000000000  -0.900000000000000  -0.700000000000000  7.117433113960459
-0.500000000000000  -0.700000000000000  -0.900000000000000  7.117433113960459
-0.300000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.900000000000000  -0.900000000000000  -0.500000000000000  7.117433113960459
-0.900000000000000  -0.700000000000000  -0.700000000000000  7.117433113960459
-0.900000000000000  -0.500000000000000  -0.900000000000000  7.117433113960459
-0.700000000000000  -0.900000000000000  -0.700000000000000  7.117433113960459
-0.700000000000000  -0.700000000000000  -0.900000000000000  7.117433113960459
-0.500000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.900000000000000  -0.900000000000000  -0.700000000000000  7.117433113960459
-0.900000000000000  -0.700000000000000  -0.900000000000000  7.117433113960459
-0.700000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.900000000000000  -0.900000000000000  -0.900000000000000  7.117433113960459
-0.888888888888889  -0.888888888888889  0.666666666666667  -5.711070467965174
-0.888888888888889  -0.666666666666667  0.444444444444444  -5.711070467965174
-0.888888888888889  -0.444444444444444  0.222222222222222  -5.711070467965174
-0.888888888888889  -0.222222222222222  0.000000000000000  -5.711070467965174
-0.888888888888889  0.000000000000000  -0.222222222222222  -5.711070467965174
-0.888888888888889  0.222222222222222  -0.444444444444444  -5.711070467965174
-0.888888888888889  0.444444444444444  -0.666666666666667  -5.711070467965174
-0.888888888888889  0.666666666666667  -0.888888888888889  -5.711070467965174
-0.666666666666667  -0.888888888888889  0.444444444444444  -5.711070467965174
-0.666666666666667  -0.666666666666667  0.222222222222222  -5.711070467965174
-0.666666666666667  -0.444444444444444  0.000000000000000  -5.711070467965174
-0.666666666666667  -0.222222222222222  -0.222222222222222  -5.711070467965174
-0.666666666666667  0.000000000000000  -0.444444444444444  -5.711070467965174
-0.666666666666667  0.222222222222222  -0.666666666666667  -5.711070467965174
-0.666666666666667  0.444444444444444  -0.888888888888889  -5.711070467965174
-0.444444444444444  -0.888888888888889  0.222222222222222  -5.711070467965174
-0.444444444444444  -0.666666666666667  0.000000000000000  -5.711070467965174
-0.444444444444444  -0.444444444444444  -0.222222222222222  -5.711070467965174
-0.444444444444444  -0.222222222222222  -0.444444444444444  -5.711070467965174
-0.444444444444444  0.000000000000000  -0.666666666666667  -5.711070467965174
-0.444444444444444  0.222222222222222  -0.888888888888889  -5.711070467965174
-0.222222222222222  -0.888888888888889  0.000000000000000  -5.711070467965174
-0.222222222222222  -0.666666666666667  -0.222222222222222  -5.711070467965174
-0.222222222222222  -0.444444444444444  -0.444444444444444  -5.711070467965174
-0.222222222222222  -0.222222222222222  -0.666666666666667  -5.711070467965174
-0.222222222222222  0.000000000000000  -0.888888888888889  -5.711070467965174
0.000000000000000  -0.888888888888889  -0.222222222222222  -5.711070467965174
0.000000000000000  -0.666666666666667  -0.444444444444444  -5.711070467965174
0.000000000000000  -0.444444444444444  -0.666666666666667  -5.711070467965174
0.000000000000000  -0.222222222222222  -0.888888888888889  -5.711070467965174
0.222222222222222  -0.888888888888889  -0.444444444444444  -5.711070467965174
0.222222222222222  -0.666666666666667  -0.666666666666667  -5.711070467965174
0.222222222222222  -0.444444444444444  -0.888888888888889  -5.711070467965174
0.444444444444444  -0.888888888888889  -0.666666666666667  -5.711070467965174
0.444444444444444  -0.666666666666667  -0.888888888888889  -5.711070467965174
0.666666666666667  -0.888888888888889  -0.888888888888889  -5.711070467965174
-0.888888888888889  -0.888888888888889  0.444444444444444  -5.711070467965174
-0.888888888888889  -0.666666666666667  0.222222222222222  -5.711070467965174
-0.888888888888889  -0.444444444444444  0.000000000000000  -5.711070467965174
-0.888888888888889  -0.222222222222222  -0.222222222222222  -5.711070467965174
-0.888888888888889  0.000000000000000  -0.444444444444444  -5.711070467965174
-0.888888888888889  0.222222222222222  -0.666666666666667  -5.711070467965174
-0.888888888888889  0.444444444444444  -0.888888888888889  -5.711070467965174
-0.666666666666667  -0.888888888888889  0.222222222222222  -5.711070467965174
-0.666666666666667  -0.666666666666667  0.000000000000000  -5.711070467965174
-0.666666666666667  -0.444444444444444  -0.222222222222222  -5.711070467965174
-0.666666666666667  -0.222222222222222  -0.444444444444444  -5.711070467965174
-0.666666666666667  0.000000000000000  -0.666666666666667  -5.711070467965174
-0.666666666666667  0.222222222222222  -0.888888888888889  -5.711070467965174
-0.444444444444444  -0.888888888888889  0.000000000000000  -5.711070467965174
-0.444444444444444  -0.666666666666667  -0.222222222222222  -5.711070467965174
-0.444444444444444  -0.444444444444444  -0.444444444444444  -5.711070467965174
-0.444444444444444  -0.222222222222222  -0.666666666666667  -5.711070467965174
-0.444444444444444  0.000000000000000  -0.888888888888889  -5.711070467965174
-0.222222222222222  -0.888888888888889  -0.222222222222222  -5.711070467965174
-0.222222222222222  -0.666666666666667  -0.444444444444444  -5.711070467965174
-0.222222222222222  -0.444444444444444  -0.666666666666667  -5.711070467965174
-0.222222222222222  -0.222222222222222  -0.888888888888889  -5.711070467965174
0.000000000000000  -0.888888888888889  -0.444444444444444  -5.711070467965174
0.000000000000000  -0.666666666666667  -0.666666666666667  -5.711070467965174
0.000000000000000  -0.444444444444444  -0.888888888888889  -5.711070467965174
0.222222222222222  -0.888888888888889  -0.666666666666667  -5.711070467965174
0.222222222222222  -0.666666666666667  -0.888888888888889  -5.711070467965174
0.444444444444444  -0.888888888888889  -0.888888888888889  -5.711070467965174
-0.888888888888889  -0.888888888888889  0.222222222222222  -5.711070467965174
-0.888888888888889  -0.666666666666667  0.000000000000000  -5.711070467965174
-0.888888888888889  -0.444444444444444  -0.222222222222222  -5.711070467965174
-0.888888888888889  -0.222222222222222  -0.444444444444444  -5.711070467965174
-0.888888888888889  0.000000000000000  -0.666666666666667  -5.711070467965174
-0.888888888888889  0.222222222222222  -0.888888888888889  -5.711070467965174
-0.666666666666667  -0.888888888888889  0.000000000000000  -5.711070467965174
-0.666666666666667  -0.666666666666667  -0.222222222222222  -5.711070467965174
-0.666666666666667  -0.444444444444444  -0.444444444444444  -5.711070467965174
-0.666666666666667  -0.222222222222222  -0.666666666666667  -5.711070467965174
-0.666666666666667  0.000000000000000  -0.888888888888889  -5.711070467965174
-0.444444444444444  -0.888888888888889  -0.222222222222222  -5.711070467965174
-0.444444444444444  -0.666666666666667  -0.444444444444444  -5.711070467965174
-0.444444444444444  -0.444444444444444  -0.666666666666667  -5.711070467965174
-0.444444444444444  -0.222222222222222  -0.888888888888889  -5.711070467965174
-0.222222222222222  -0.888888888888889  -0.444444444444444  -5.711070467965174
-0.222222222222222  -0.666666666666667  -0.666666666666667  -5.711070467965174
-0.222222222222222  -0.444444444444444  -0.888888888888889  -5.711070467965174
0.000000000000000  -0.888888888888889  -0.666666666666667  -5.711070467965174
0.000000000000000  -0.666666666666667  -0.888888888888889  -5.711070467965174
0.222222222222222  -0.888888888888889  -0.888888888888889  -5.711070467965174
-0.888888888888889  -0.888888888888889  0.000000000000000  -5.711070467965174
-0.888888888888889  -0.666666666666667  -0.222222222222222  -5.711070467965174
-0.888888888888889  -0.444444444444444  -0.444444444444444  -5.711070467965174
-0.888888888888889  -0.222222222222222  -0.666666666666667  -5.711070467965174
-0.888888888888889  0.000000000000000  -0.888888888888889  -5.711070467965174
-0.666666666666667  -0.888888888888889  -0.222222222222222  -5.711070467965174
-0.666666666666667  -0.666666666666667  -0.444444444444444  -5.711070467965174
-0.666666666666667  -0.444444444444444  -0.666666666666667  -5.711070467965174
-0.666666666666667  -0.222222222222222  -0.888888888888889  -5.711070467965174
-0.444444444444444  -0.888888888888889  -0.444444444444444  -5.711070467965174
-0.444444444444444  -0.666666666666667  -0.666666666666667  -5.711070467965174
-0.444444444444444  -0.444444444444444  -0.888888888888889  -5.711070467965174
-0.222222222222222  -0.888888888888889  -0.666666666666667  -5.711070467965174
-0.222222222222222  -0.666666666666667  -0.888888888888889  -5.711070467965174
0.000000000000000  -0.888888888888889  -0.888888888888889  -5.711070467965174
-0.888888888888889  -0.888888888888889  -0.222222222222222  -5.711070467965174
-0.888888888888889  -0.666666666666667  -0.444444444444444  -5.711070467965174
-0.888888888888889  -0.444444444444444  -0.666666666666667  -5.711070467965174
-0.888888888888889  -0.222222222222222  -0.888888888888889  -5.711070467965174
-0.666666666666667  -0.888888888888889  -0.444444444444444  -5.711070467965174
-0.666666666666667  -0.666666666666667  -0.666666666666667  -5.711070467965174
-0.666666666666667  -0.444444444444444  -0.888888888888889  -5.711070467965174
-0.444444444444444  -0.888888888888889  -0.666666666666667  -5.711070467965174
-0.444444444444444  -0.666666666666667  -0.888888888888889  -5.711070467965174
-0.222222222222222  -0.888888888888889  -0.888888888888889  -5.711070467965174
-0.888888888888889  -0.888888888888889  -0.444444444444444  -5.711070467965174
-0.888888888888889  -0.666666666666667  -0.666666666666667  -5.711070467965174
-0.888888888888889  -0.444444444444444  -0.888888888888889  -5.711070467965174
-0.666666666666667  -0.888888888888889  -0.666666666666667  -5.711070467965174
-0.666666666666667  -0.666666666666667  -0.888888888888889  -5.711070467965174
-0.444444444444444  -0.888888888888889  -0.888888888888889  -5.711070467965174
-0.888888888888889  -0.888888888888889  -0.666666666666667  -5.711070467965174
-0.888888888888889  -0.666666666666667  -0.888888888888889  -5.711070467965174
-0.666666666666667  -0.888888888888889  -0.888888888888889  -5.711070467965174
-0.888888888888889  -0.888888888888889  -0.888888888888889  -5.711070467965174
-0.875000000000000  -0.875000000000000  0.625000000000000  2.527399241963254
-0.875000000000000  -0.625000000000000  0.375000000000000  2.527399241963254
-0.875000000000000  -0.375000000000000  0.125000000000000  2.527399241963254
-0.875000000000000  -0.125000000000000  -0.125000000000000  2.527399241963254
-0.875000000000000  0.125000000000000  -0.375000000000000  2.527399241963254
-0.875000000000000  0.375000000000000  -0.625000000000000  2.527399241963254
-0.875000000000000  0.625000000000000  -0.875000000000000  2.527399241963254
-0.625000000000000  -0.875000000000000  0.375000000000000  2.527399241963254
-0.625000000000000  -0.625000000000000  0.125000000000000  2.527399241963254
-0.625000000000000  -0.375000000000000  -0.125000000000000  2.527399241963254
-0.625000000000000  -0.125000000000000  -0.375000000000000  2.527399241963254
-0.625000000000000  0.125000000000000  -0.625000000000000  2.527399241963254
-0.625000000000000  0.375000000000000  -0.875000000000000  2.527399241963254
-0.375000000000000  -0.875000000000000  0.125000000000000  2.527399241963254
-0.375000000000000  -0.625000000000000  -0.125000000000000  2.527399241963254
-0.375000000000000  -0.375000000000000  -0.375000000000000  2.527399241963254
-0.375000000000000  -0.125000000000000  -0.625000000000000  2.527399241963254
-0.375000000000000  0.125000000000000  -0.875000000000000  2.527399241963254
-0.125000000000000  -0.875000000000000  -0.125000000000000  2.527399241963254
-0.125000000000000  -0.625000000000000  -0.375000000000000  2.527399241963254
-0.125000000000000  -0.375000000000000  -0.625000000000000  2.527399241963254
-0.125000000000000  -0.125000000000000  -0.875000000000000  2.527399241963254
0.125000000000000  -0.875000000000000  -0.375000000000000  2.527399241963254
0.125000000000000  -0.625000000000000  -0.625000000000000  2.527399241963254
0.125000000000000  -0.375000000000000  -0.875000000000000  2.527399241963254
0.375000000000000  -0.875000000000000  -0.625000000000000  2.527399241963254
0.375000000000000  -0.625000000000000  -0.875000000000000  2.527399241963254
0.625000000000000  -0.875000000000000  -0.875000000000000  2.527399241963254
-0.875000000000000  -0.875000000000000  0.375000000000000  2.527399241963254
-0.875000000000000  -0.625000000000000  0.125000000000000  2.527399241963254
-0.875000000000000  -0.375000000000000  -0.125000000000000  2.527399241963254
-0.875000000000000  -0.125000000000000  -0.375000000000000  2.527399241963254
-0.875000000000000  0.125000000000000  -0.625000000000000  2.527399241963254
-0.875000000000000  0.375000000000000  -0.875000000000000  2.527399241963254
-0.625000000000000  -0.875000000000000  0.125000000000000  2.527399241963254
-0.625000000000000  -0.625000000000000  -0.125000000000000  2.527399241963254
-0.625000000000000  -0.375000000000000  -0.375000000000000  2.527399241963254
-0.625000000000000  -0.125000000000000  -0.625000000000000  2.527399241963254
-0.625000000000000  0.125000000000000  -0.875000000000000  2.527399241963254
-0.375000000000000  -0.875000000000000  -0.125000000000000  2.527399241963254
-0.375000000000000  -0.625000000000000  -0.375000000000000  2.527399241963254
-0.375000000000000  -0.375000000000000  -0.625000000000000  2.527399241963254
-0.375000000000000  -0.125000000000000  -0.875000000000000  2.527399241963254
-0.125000000000000  -0.875000000000000  -0.375000000000000  2.527399241963254
-0.125000000000000  -0.625000000000000  -0.625000000000000  2.527399241963254
-0.125000000000000  -0.375000000000000  -0.875000000000000  2.527399241963254
0.125000000000000  -0.875000000000000  -0.625000000000000  2.527399241963254
0.125000000000000  -0.625000000000000  -0.875000000000000  2.527399241963254
0.375000000000000  -0.875000000000000  -0.875000000000000  2.527399241963254
-0.875000000000000  -0.875000000000000  0.125000000000000  2.527399241963254
-0.875000000000000  -0.625000000000000  -0.125000000000000  2.527399241963254
-0.875000000000000  -0.375000000000000  -0.375000000000000  2.527399241963254
-0.875000000000000  -0.125000000000000  -0.625000000000000  2.527399241963254
-0.875000000000000  0.125000000000000  -0.875000000000000  2.527399241963254
-0.625000000000000  -0.875000000000000  -0.125000000000000  2.527399241963254
-0.625000000000000  -0.625000000000000  -0.375000000000000  2.527399241963254
-0.625000000000000  -0.375000000000000  -0.625000000000000  2.527399241963254
-0.625000000000000  -0.125000000000000  -0.875000000000000  2.527399241963254
-0.375000000000000  -0.875000000000000  -0.375000000000000  2.527399241963254
-0.375000000000000  -0.625000000000000  -0.625000000000000  2.527399241963254
-0.375000000000000  -0.375000000000000  -0.875000000000000  2.527399241963254
-0.125000000000000  -0.875000000000000  -0.625000000000000  2.527399241963254
-0.125000000000000  -0.625000000000000  -0.875000000000000  2.527399241963254
0.125000000000000  -0.875000000000000  -0.875000000000000  2.527399241963254
-0.875000000000000  -0.875000000000000  -0.125000000000000  2.527399241963254
-0.875000000000000  -0.625000000000000  -0.375000000000000  2.527399241963254
-0.875000000000000  -0.375000000000000  -0.625000000000000  2.527399241963254
-0.875000000000000  -0.125000000000000  -0.875000000000000  2.527399241963254
-0.625000000000000  -0.875000000000000  -0.375000000000000  2.527399241963254
-0.625000000000000  -0.625000000000000  -0.625000000000000  2.527399241963254
-0.625000000000000  -0.375000000000000  -0.875000000000000  2.527399241963254
-0.375000000000000  -0.875000000000000  -0.625000000000000  2.527399241963254
-0.375000000000000  -0.625000000000000  -0.875000000000000  2.527399241963254
-0.125000000000000  -0.875000000000000  -0.875000000000000  2.527399241963254
-0.875000000000000  -0.875000000000000  -0.375000000000000  2.527399241963254
-0.875000000000000  -0.625000000000000  -0.625000000000000  2.527399241963254
-0.875000000000000  -0.375000000000000  -0.875000000000000  2.527399241963254
-0.625000000000000  -0.875000000000000  -0.625000000000000  2.527399241963254
-0.625000000000000  -0.625000000000000  -0.875000000000000  2.527399241963254
-0.375000000000000  -0.875000000000000  -0.875000000000000  2.527399241963254
-0.875000000000000  -0.875000000000000  -0.625000000000000  2.527399241963254
-0.875000000000000  -0.625000000000000  -0.875000000000000  2.527399241963254
-0.625000000000000  -0.875000000000000  -0.875000000000000  2.527399241963254
-0.875000000000000  -0.875000000000000  -0.875000000000000  2.527399241963254
-0.857142857142857  -0.857142857142857  0.571428571428571  -0.612213575618572
-0.857142857142857  -0.571428571428571  0.285714285714286  -0.612213575618572
-0.857142857142857  -0.285714285714286  0.000000000000000  -0.612213575618572
-0.857142857142857  0.000000000000000  -0.285714285714286  -0.612213575618572
-0.857142857142857  0.285714285714286  -0.571428571428571  -0.612213575618572
-0.857142857142857  0.571428571428571  -0.857142857142857  -0.612213575618572
-0.571428571428571  -0.857142857142857  0.285714285714286  -0.612213575618572
-0.571428571428571  -0.571428571428571  0.000000000000000  -0.612213575618572
-0.571428571428571  -0.285714285714286  -0.285714285714286  -0.612213575618572
-0.571428571428571  0.000000000000000  -0.571428571428571  -0.612213575618572
-0.571428571428571  0.285714285714286  -0.857142857142857  -0.612213575618572
-0.285714285714286  -0.857142857142857  0.000000000000000  -0.612213575618572
-0.285714285714286  -0.571428571428571  -0.285714285714286  -0.612213575618572
-0.285714285714286  -0.285714285714286  -0.571428571428571  -0.612213575618572
-0.285714285714286  0.000000000000000  -0.857142857142857  -0.612213575618572
0.000000000000000  -0.857142857142857  -0.285714285714286  -0.612213575618572
0.000000000000000  -0.571428571428571  -0.571428571428571  -0.612213575618572
0.000000000000000  -0.285714285714286  -0.857142857142857  -0.612213575618572
0.285714285714286  -0.857142857142857  -0.571428571428571  -0.612213575618572
0.285714285714286  -0.571428571428571  -0.857142857142857  -0.612213575618572
0.571428571428571  -0.857142857142857  -0.857142857142857  -0.612213575618572
-0.857142857142857  -0.857142857142857  0.285714285714286  -0.612213575618572
-0.857142857142857  -0.571428571428571  0.000000000000000  -0.612213575618572
-0.857142857142857  -0.285714285714286  -0.285714285714286  -0.612213575618572
-0.857142857142857  0.000000000000000  -0.571428571428571  -0.612213575618572
-0.857142857142857  0.285714285714286  -0.857142857142857  -0.612213575618572
-0.571428571428571  -0.857142857142857  0.000000000000000  -0.612213575618572
-0.571428571428571  -0.571428571428571  -0.285714285714286  -0.612213575618572
-0.571428571428571  -0.285714285714286  -0.571428571428571  -0.612213575618572
-0.571428571428571  0.000000000000000  -0.857142857142857  -0.612213575618572
-0.285714285714286  -0.857142857142857  -0.285714285714286  -0.612213575618572
-0.285714285714286  -0.571428571428571  -0.571428571428571  -0.612213575618572
-0.285714285714286  -0.285714285714286  -0.857142857142857  -0.612213575618572
0.000000000000000  -0.857142857142857  -0.571428571428571  -0.612213575618572
0.000000000000000  -0.571428571428571  -0.857142857142857  -0.612213575618572
0.285714285714286  -0.857142857142857  -0.857142857142857  -0.612213575618572
-0.857142857142857  -0.857142857142857  0.000000000000000  -0.612213575618572
-0.857142857142857  -0.571428571428571  -0.285714285714286  -0.612213575618572
-0.857142857142857  -0.285714285714286  -0.571428571428571  -0.612213575618572
-0.857142857142857  0.000000000000000  -0.857142857142857  -0.612213575618572
-0.571428571428571  -0.857142857142857  -0.285714285714286  -0.612213575618572
-0.571428571428571  -0.571428571428571  -0.571428571428571  -0.612213575618572
-0.571428571428571  -0.285714285714286  -0.857142857142857  -0.612213575618572
-0.285714285714286  -0.857142857142857  -0.571428571428571  -0.612213575618572
-0.285714285714286  -0.571428571428571  -0.857142857142857  -0.612213575618572
0.000000000000000  -0.857142857142857  -0.857142857142857  -0.612213575618572
-0.857142857142857  -0.857142857142857  -0.285714285714286  -0.612213575618572
-0.857142857142857  -0.571428571428571  -0.571428571428571  -0.612213575618572
-0.857142857142857  -0.285714285714286  -0.857142857142857  -0.612213575618572
-0.571428571428571  -0.857142857142857  -0.571428571428571  -0.612213575618572
-0.571428571428571  -0.571428571428571  -0.857142857142857  -0.612213575618572
-0.285714285714286  -0.857142857142857  -0.857142857142857  -0.612213575618572
-0.857142857142857  -0.857142857142857  -0.571428571428571  -0.612213575618572
-0.857142857142857  -0.571428571428571  -0.857142857142857  -0.612213575618572
-0.571428571428571  -0.857142857142857  -0.857142857142857  -0.612213575618572
-0.857142857142857  -0.857142857142857  -0.857142857142857  -0.612213575618572
-0.833333333333333  -0.833333333333333  0.500000000000000  0.076141727180383
-0.833333333333333  -0.500000000000000  0.166666666666667  0.076141727180383
-0.833333333333333  -0.166666666666667  -0.166666666666667  0.076141727180383
-0.833333333333333  0.166666666666667  -0.500000000000000  0.076141727180383
-0.833333333333333  0.500000000000000  -0.833333333333333  0.076141727180383
-0.500000000000000  -0.833333333333333  0.166666666666667  0.076141727180383
-0.500000000000000  -0.500000000000000  -0.166666666666667  0.076141727180383
-0.500000000000000  -0.166666666666667  -0.500000000000000  0.076141727180383
-0.500000000000000  0.166666666666667  -0.833333333333333  0.076141727180383
-0.166666666666667  -0.833333333333333  -0.166666666666667  0.076141727180383
-0.166666666666667  -0.500000000000000  -0.500000000000000  0.076141727180383
-0.166666666666667  -0.166666666666667  -0.833333333333333  0.076141727180383
0.166666666666667  -0.833333333333333  -0.500000000000000  0.076141727180383
0.166666666666667  -0.500000000000000  -0.833333333333333  0.076141727180383
0.500000000000000  -0.833333333333333  -0.833333333333333  0.076141727180383
-0.833333333333333  -0.833333333333333  0.166666666666667  0.076141727180383
-0.833333333333333  -0.500000000000000  -0.166666666666667  0.076141727180383
-0.833333333333333  -0.166666666666667  -0.500000000000000  0.076141727180383
-0.833333333333333  0.166666666666667  -0.833333333333333  0.076141727180383
-0.500000000000000  -0.833333333333333  -0.166666666666667  0.076141727180383
-0.500000000000000  -0.500000000000000  -0.500000000000000  0.076141727180383
-0.500000000000000  -0.166666666666667  -0.833333333333333  0.076141727180383
-0.166666666666667  -0.833333333333333  -0.500000000000000  0.076141727180383
-0.166666666666667  -0.500000000000000  -0.833333333333333  0.076141727180383
0.166666666666667  -0.833333333333333  -0.833333333333333  0.076141727180383
-0.833333333333333  -0.833333333333333  -0.166666666666667  0.076141727180383
-0.833333333333333  -0.500000000000000  -0.500000000000000  0.076141727180383
-0.833333333333333  -0.166666666666667  -0.833333333333333  0.076141727180383
-0.500000000000000  -0.833333333333333  -0.500000000000000  0.076141727180383
-0.500000000000000  -0.500000000000000  -0.833333333333333  0.076141727180383
-0.166666666666667  -0.833333333333333  -0.833333333333333  0.076141727180383
-0.833333333333333  -0.833333333333333  -0.500000000000000  0.076141727180383
-0.833333333333333  -0.500000000000000  -0.833333333333333  0.076141727180383
-0.500000000000000  -0.833333333333333  -0.833333333333333  0.076141727180383
-0.833333333333333  -0.833333333333333  -0.833333333333333  0.076141727180383
-0.800000000000000  -0.800000000000000  0.400000000000000  -0.004255896151021
-0.800000000000000  -0.400000000000000  0.000000000000000  -0.004255896151021
-0.800000000000000  0.000000000000000  -0.400000000000000  -0.004255896151021
-0.800000000000000  0.400000000000000  -0.800000000000000  -0.004255896151021
-0.400000000000000  -0.800000000000000  0.000000000000000  -0.004255896151021
-0.400000000000000  -0.400000000000000  -0.400000000000000  -0.004255896151021
-0.400000000000000  0.000000000000000  -0.800000000000000  -0.004255896151021
0.000000000000000  -0.800000000000000  -0.400000000000000  -0.004255896151021
0.000000000000000  -0.400000000000000  -0.800000000000000  -0.004255896151021
0.400000000000000  -0.800000000000000  -0.800000000000000  -0.004255896151021
-0.800000000000000  -0.800000000000000  0.000000000000000  -0.004255896151021
-0.800000000000000  -0.400000000000000  -0.400000000000000  -0.004255896151021
-0.800000000000000  0.000000000000000  -0.800000000000000  -0.004255896151021
-0.400000000000000  -0.800000000000000  -0.400000000000000  -0.004255896151021
-0.400000000000000  -0.400000000000000  -0.800000000000000  -0.004255896151021
0.000000000000000  -0.800000000000000  -0.800000000000000  -0.004255896151021
-0.800000000000000  -0.800000000000000  -0.400000000000000  -0.004255896151021
-0.800000000000000  -0.400000000000000  -0.800000000000000  -0.004255896151021
-0.400000000000000  -0.800000000000000  -0.800000000000000  -0.004255896151021
-0.800000000000000  -0.800000000000000  -0.800000000000000  -0.004255896151021
-0.750000000000000  -0.750000000000000  0.250000000000000  0.000083414141296
-0.750000000000000  -0.250000000000000  -0.250000000000000  0.000083414141296
-0.750000000000000  0.250000000000000  -0.750000000000000  0.000083414141296
-0.250000000000000  -0.750000000000000  -0.250000000000000  0.000083414141296
-0.250000000000000  -0.250000000000000  -0.750000000000000  0.000083414141296
0.250000000000000  -0.750000000000000  -0.750000000000000  0.000083414141296
-0.750000000000000  -0.750000000000000  -0.250000000000000  0.000083414141296
-0.750000000000000  -0.250000000000000  -0.750000000000000  0.000083414141296
-0.250000000000000  -0.750000000000000  -0.750000000000000  0.000083414141296
-0.750000000000000  -0.750000000000000  -0.750000000000000  0.000083414141296
-0.666666666666667  -0.666666666666667  0.000000000000000  -0.000000352698561
-0.666666666666667  0.000000000000000  -0.666666666666667  -0.000000352698561
0.000000000000000  -0.666666666666667  -0.666666666666667  -0.000000352698561
-0.666666666666667  -0.666666666666667  -0.666666666666667  -0.000000352698561
-0.500000000000000  -0.500000000000000  -0.500000000000000  0.000000000106067
  Gauss  quadrature  constants  for  the  reference  tetrahedron  order  p=21
"""




newton_cotes_table = """\
2
1  1
-1  1
  Closed  Newton-Cotes  quadrature  constants  order  n=1

3
1  1/3
0  4/3
-1  1/3
  Closed  Newton-Cotes  quadrature  constants  order  n=2

4
1  1/4
1/3  3/4
-1/3  3/4
-1  1/4
  Closed  Newton-Cotes  quadrature  constants  order  n=3

5
1  7/45
1/2  32/45
0  12/45
-1/2  32/45
-1  7/45
  Closed  Newton-Cotes  quadrature  constants  order  n=4

6
1  19/144
3/5  75/144
1/5  50/144
-1/5  50/144
-3/5  75/144
-1  19/144
  Closed  Newton-Cotes  quadrature  constants  order  n=5

7
1  41/420
2/3  216/420
1/3  27/420
0  272/420
-1/3  27/420
-2/3  216/420
-1  41/420
  Closed  Newton-Cotes  quadrature  constants  order  n=6

8
1  751/8640
5/7  3577/8640
3/7  1323/8640
1/7  2989/8640
-1/7  2989/8640
-3/7  1323/8640
-5/7  3577/8640
-1  751/8640
  Closed  Newton-Cotes  quadrature  constants  order  n=7
"""


chebyshev_table = """\
2
0.5773502691896257645091488  1
-0.5773502691896257645091488  1
  Chebyshev  quadrature  constants  order  n+1=3

3
0.7071067811865475244008444  2/3
0.0000000000000000000000000  2/3
-0.7071067811865475244008444  2/3
  Chebyshev  quadrature  constants  order  n+1=4

4
0.7946544722917661229555309  1/2
0.1875924740850798998601393  1/2
-0.1875924740850798998601393  1/2
-0.7946544722917661229555309  1/2
  Chebyshev  quadrature  constants  order  n+1=5

5
0.8324974870009818758925836  2/5
0.3745414095535810655860444  2/5
0.0000000000000000000000000  2/5
-0.3745414095535810655860444  2/5
-0.8324974870009818758925836  2/5
  Chebyshev  quadrature  constants  order  n+1=6

6
0.8662468181078205913835981  1/3
0.4225186537611115291185464  1/3
0.2666354015167047203315346  1/3
-0.2666354015167047203315346  1/3
-0.4225186537611115291185464  1/3
-0.8662468181078205913835981  1/3
  Chebyshev  quadrature  constants  order  n+1=7

7
0.8838617007580490357042241  2/7
0.5296567752851568113850475  2/7
0.3239118105199076375196731  2/7
0.0000000000000000000000000  2/7
-0.3239118105199076375196731  2/7
-0.5296567752851568113850475  2/7
-0.8838617007580490357042241  2/7
  Chebyshev  quadrature  constants  order  n+1=8

9
0.9115893077284344736649486  2/9
0.6010186553802380714281279  2/9
0.5287617830578799932601816  2/9
0.1679061842148039430680319  2/9
0.0000000000000000000000000  2/9
-0.1679061842148039430680319  2/9
-0.5287617830578799932601816  2/9
-0.6010186553802380714281279  2/9
-0.9115893077284344736649486  2/9
  Chebyshev  quadrature  constants  order  n+1=10
"""


lobatto_table = """\
3
1.0000000000000000000000000  0.3333333333333333333333333
0.0000000000000000000000000  1.3333333333333333333333333
-1.0000000000000000000000000  0.3333333333333333333333333
  Lobatto  (Radau)  quadrature  constants  order  2n-3=3

4
1.0000000000000000000000000  0.1666666666666666666666667
0.4472135954999579392818347  0.8333333333333333333333333
-0.4472135954999579392818347  0.8333333333333333333333333
-1.0000000000000000000000000  0.1666666666666666666666667
  Lobatto  (Radau)  quadrature  constants  order  2n-3=5

5
1.0000000000000000000000000  0.1000000000000000000000000
0.6546536707079771437982925  0.5444444444444444444444444
0.0000000000000000000000000  0.7111111111111111111111111
-0.6546536707079771437982925  0.5444444444444444444444444
-1.0000000000000000000000000  0.1000000000000000000000000
  Lobatto  (Radau)  quadrature  constants  order  2n-3=7

6
1.0000000000000000000000000  0.0666666666666666666666667
0.7650553239294646928510030  0.3784749562978469803166128
0.2852315164806450963141510  0.5548583770354863530167205
-0.2852315164806450963141510  0.5548583770354863530167205
-0.7650553239294646928510030  0.3784749562978469803166128
-1.0000000000000000000000000  0.0666666666666666666666667
  Lobatto  (Radau)  quadrature  constants  order  2n-3=9

7
1.0000000000000000000000000  0.0476190476190476190476190
0.8302238962785669298720322  0.2768260473615659480107004
0.4688487934707142138037719  0.4317453812098626234178710
0.0000000000000000000000000  0.4876190476190476190476190
-0.4688487934707142138037719  0.4317453812098626234178710
-0.8302238962785669298720322  0.2768260473615659480107004
-1.0000000000000000000000000  0.0476190476190476190476190
  Lobatto  (Radau)  quadrature  constants  order  2n-3=11

8
1.0000000000000000000000000  0.0357142857142857142857143
0.8717401485096066153374457  0.2107042271435060393829921
0.5917001814331423021445107  0.3411226924835043647642407
0.2092992179024788687686573  0.4124587946587038815670530
-0.2092992179024788687686573  0.4124587946587038815670530
-0.5917001814331423021445107  0.3411226924835043647642407
-0.8717401485096066153374457  0.2107042271435060393829921
-1.0000000000000000000000000  0.0357142857142857142857143
  Lobatto  (Radau)  quadrature  constants  order  2n-3=13

9
1.0000000000000000000000000  0.0277777777777777777777778
0.8997579954114601573123452  0.1654953615608055250463397
0.6771862795107377534458854  0.2745387125001617352807056
0.3631174638261781587107521  0.3464285109730463451151315
0.0000000000000000000000000  0.3715192743764172335600907
-0.3631174638261781587107521  0.3464285109730463451151315
-0.6771862795107377534458854  0.2745387125001617352807056
-0.8997579954114601573123452  0.1654953615608055250463397
-1.0000000000000000000000000  0.0277777777777777777777778
  Lobatto  (Radau)  quadrature  constants  order  2n-3=15

10
1.0000000000000000000000000  0.0222222222222222222222222
0.9195339081664588138289327  0.1333059908510701111262272
0.7387738651055050750031062  0.2248893420631264521194578
0.4779249498104444956611751  0.2920426836796837578755823
0.1652789576663870246262198  0.3275397611838974566565105
-0.1652789576663870246262198  0.3275397611838974566565105
-0.4779249498104444956611751  0.2920426836796837578755823
-0.7387738651055050750031062  0.2248893420631264521194578
-0.9195339081664588138289327  0.1333059908510701111262272
-1.0000000000000000000000000  0.0222222222222222222222222
  Lobatto  (Radau)  quadrature  constants  order  2n-3=17

11
1.0000000000000000000000000  0.0181818181818181818181818
0.9340014304080591343322741  0.1096122732669948644614034
0.7844834736631444186224178  0.1871698817803052041081415
0.5652353269962050064709640  0.2480481042640283140400849
0.2957581355869393914319115  0.2868791247790080886792224
0.0000000000000000000000000  0.3002175954556906937859319
-0.2957581355869393914319115  0.2868791247790080886792224
-0.5652353269962050064709640  0.2480481042640283140400849
-0.7844834736631444186224178  0.1871698817803052041081415
-0.9340014304080591343322741  0.1096122732669948644614034
-1.0000000000000000000000000  0.0181818181818181818181818
  Lobatto  (Radau)  quadrature  constants  order  2n-3=19

12
1.0000000000000000000000000  0.0151515151515151515151515
0.9448992722228822234075801  0.0916845174131961306683426
0.8192793216440066783486416  0.1579747055643701151646710
0.6328761530318606776624049  0.2125084177610211453583021
0.3995309409653489322643498  0.2512756031992012802932444
0.1365529328549275548640619  0.2714052409106961770002883
-0.1365529328549275548640619  0.2714052409106961770002883
-0.3995309409653489322643498  0.2512756031992012802932444
-0.6328761530318606776624049  0.2125084177610211453583021
-0.8192793216440066783486416  0.1579747055643701151646710
-0.9448992722228822234075801  0.0916845174131961306683426
-1.0000000000000000000000000  0.0151515151515151515151515
  Lobatto  (Radau)  quadrature  constants  order  2n-3=21
"""

