#% -*- mode: tm; mode: fold -*-

#%{{{Macros 

#i linuxdoc.tm
#d it#1 <it>$1</it>

#d slang \bf{S-lang}
#d exmp#1 \tt{$1}
#d var#1 \tt{$1}

#d ivar#1 \tt{$1}
#d ifun#1 \tt{$1}
#d cvar#1 \tt{$1}
#d cfun#1 \tt{$1}
#d svar#1 \tt{$1}
#d sfun#1 \tt{$1}
#d icon#1 \tt{$1}

#d chapter#1 <chapt>$1<p>
#d preface <preface>
#d tag#1 <tag>$1</tag>

#d function#1 \sect{<bf>$1</bf>\label{$1}}<descrip>
#d variable#1 \sect{<bf>$1</bf>\label{$1}}<descrip>
#d function_sect#1 \sect{$1}
#d begin_constant_sect#1 \sect{$1}<itemize>
#d constant#1 <item><tt>$1</tt>
#d end_constant_sect </itemize>

#d synopsis#1 <tag> Synopsis </tag> $1
#d keywords#1 <tag> Keywords </tag> $1
#d usage#1 <tag> Usage </tag> <tt>$1</tt>
#d description <tag> Description </tag>
#d example <tag> Example </tag>
#d notes <tag> Notes </tag>
#d seealso#1 <tag> See Also </tag> <tt>\linuxdoc_list_to_ref{$1}</tt>
#d done </descrip><p>
#d -1 <tt>-1</tt>
#d 0 <tt>0</tt>
#d 1 <tt>1</tt>
#d 2 <tt>2</tt>
#d 3 <tt>3</tt>
#d 4 <tt>4</tt>
#d 5 <tt>5</tt>
#d 6 <tt>6</tt>
#d 7 <tt>7</tt>
#d 8 <tt>8</tt>
#d 9 <tt>9</tt>
#d NULL <tt>NULL</tt>
#d documentstyle book

#%}}}

#d module#1 \tt{$1}
#d file#1 \tt{$1}

#d slang-documentation \
 \url{http://www.s-lang.org/doc/html/slang.html}{S-Lang documentation}
#d evt2img \bf{evt2img}

\linuxdoc
\begin{\documentstyle}

\title S-Lang Histogram Module Reference
\author John E. Davis, \tt{davis@space.mit.edu}
\date \__today__

\toc

\chapter{Introduction to the Histogram Module}
 The \module{histogram} module contains several functions for the
 efficient creation and manipulation of one and two dimensional
 histograms.

 One of the most useful and distinguishing features of the histogram
 module is the notion of reverse-indices.  Simply put,
 reverse-indices indicate what elements went into a specified
 histogram.  The word "reverse" signifies that it is a mapping from
 the histogram back to the data that produced it.

\chapter{Using the Histogram Module}

 To use the \module{histogram} module in a \slang script, it is first
 necessary to make the functions in the package known to the
 interpreter via
#v+
    () = evalfile ("histogram");
#v-
 or, if the application embedding the interpreter supports the
 \sfun{require} function,
#v+
    require ("histogram");
#v-
 may be used.  If there is a namespace conflict between symbols in the
 script and those defined in the module, it may be necessary to load
 the histogram package into a namespace, e.g.,
#v+
   () = evalfile ("histogram", "hist");
#v-
 will place the histogram symbols into a namespace called \exmp{hist}.

 Once the histogram module has been loaded, functions defined
 by the it may be used in the usual way, e.g.,
#v+
    require ("histogram");
        .
        .
    h = hist1d (points, [min(points):max(points):0.1]);
#v-
 where \ifun{hist1d} is the 1-d histogram function.


#%}}}

\chapter{Examples}

 This section presents examples using the \ifun{hist1d} and \ifun{hist2d}
 functions.  The examples are particularly relevant to X-ray astronomy.
 The \evt2img demo program distributed with the \module{histogram}
 module may be regarded as a more complex combination of the examples
 below.

\sect{Example 1: Light Curves and Color-Color Diagrams}
 Each \em{event} measured by the \url{http://cxc.harvard.edu}{Chandra
 X-Ray Observatory} is characterized by many attributes including
 exposure-number and energy.  Many objects observed by the observatory
 undergo flares causing a change in the observed count-rate.  Suppose
 that the values of the variables \exmp{expno} and \exmp{energy} are
 1-d arrays such that \exmp{expno[i]} and \exmp{energy[i]} represent
 the exposure number and energy of the \it{i}th event, respectively.
 Then an array representing the number of events per exposure number
 may be constructed using the \ifun{hist1d} function:
#v+
    expno_grid = [min(expno):max(expno)];
    count_rate = hist1d (expno, expno_grid);
#v-
 Here, \exmp{expno_grid} represents the grid used for binning the
 exposure numbers into the histogram.  Plotting \exmp{count_rate}
 as a function of \exmp{expno_grid} will show how the count-rate
 changes with exposure number.  Such a plot is known as a light-curve.
 
 For a weak source, there may be few events in each exposure causing
 the measured count-rate to look noisy.  The signal to noise ratio can
 be increased by using larger bin sizes when constructing the histogram.
 For example, 
#v+
   bin_size = 50;   % Use 50 exposures per bin
   expno_grid = [min(expno):max(expno):bin_size];
   count_rate = hist1d (expno, expno_grid)/bin_size;
#v-
 will compute a count-rate using 50 exposures per bin.
 
 By studying the measured count-rate, one can ascertain whether or not
 the source had a flare.  Another important question is whether during
 the flare the spectrum of the source changed.  For example, the
 X-ray spectrum of some sources will change from a soft state (low
 energies) to a hard state (high energies) during a flare.  The mean
 energy per exposure may be used to get a handle upon any spectral
 changes.  The computation of this quantity involves computing the
 mean energy of each event within an exposure.  Although one can use
 brute-force methods to compute this, the simplest and most efficient
 is to use a histogram, but keeping track of what events went into
 each bin of the histogram.  As before
#v+
   bin_size = 50;
   expno_grid = [min(expno):max(expno):bin_size];
   count_rate = hist1d (expno, expno_grid, &rev)/bin_size;
#v-
 computes the count-rate.  Note the use of \exmp{&rev} as an
 additional argument to \ifun{hist1d}.  After \ifun{hist1d} returns,
 the value of \exmp{rev} will be an array-of-arrays of indices that
 indicate how the events were distributed into the histogram.  That
 is, \exmp{rev[0]} represents the list of event indices that
 contributed to the first histogram bin.  Hence, the expression
#v+
    mean (energy[rev[0]]) / bin_size
#v-
 gives the mean energy of the events in the first histogram bin.  The
 mean energy as a function of exposure number may be computed by
#v+
    num_bins = length (expno_grid);
    mean_energy = Double_Type [num_bins];
    for (i = 0; i < num_bins; i++)
      mean_energy[i] = mean (energy[rev[i]])/bin_size;
#v-
 A plot of \exmp{mean_energy} versus \exmp{expno_grid} may give an
 indication of how the spectrum changed during the observation.
 
 Finally, consider the construction of a so-called \em{color-color}
 diagram.  This is simply a plot of the ratios of count-rates in
 various energy bands.  Suppose that one is interested in three energy
 bands: 1-2 keV, 2-6 keV, and 6-12 keV.  The event list may be
 filtered in these three bands as follows:
#v+
    i_low = where ((energy >= 1) and (energy < 2));
    i_med = where ((energy >= 2) and (energy < 6));
    i_hi  = where ((energy >= 6) and (energy < 12));
#v-
 These filters may be used to construct count-rates in the three
 bands:
#v+
    rate_low = hist1d (energy[i_low], expno_grid)/bin_size;
    rate_med = hist1d (energy[i_med], expno_grid)/bin_size;
    rate_hi = hist1d (energy[i_hi], expno_grid)/bin_size;
#v-
 The color-color plot is made by plotting \exmp{rate_hi/rate_med}
 versus \exmp{rate_med/rate_low}.

\sect{Example 2: A true-color image}
 This example shows how to use the \ifun{hist2d} function to create a
 color-coded image.  In addition to the energy and exposure number, an
 event is also associated with an (X,Y) coordinate representing the
 position on the sky where the photon causing the event originated.
 The three energy bands of the previous example will be used.  Events
 with an energy in the lowest band will be represented by the color
 red, the events in the middle band by green, and those in the highest
 energy band by blue.
 
 A full resolution image generated by the Chandra Observatory's ACIS
 detector consists of 8192x8192 pixels.  For economy, a lower
 resolution 1024x1024 image will be produced.  The \ifun{hist2d}
 function will be used to produce the individual planes of the final
 image:
#v+
     xgrid = [1:8192:8];
     ygrid = [1:8192:8];
     r_image = hist2d (X[i_low], Y[i_low], xgrid, ygrid);
     g_image = hist2d (X[i_med], Y[i_med], xgrid, ygrid);
     b_image = hist2d (X[i_hi], Y[i_hi], xgrid, ygrid);
#v-
 Here \exmp{i_low}, \exmp{i_med}, and \exmp{i_hi} are defined as in
 the previous example.

\chapter{Histogram Module Function Reference}

#i histfuns.tm

\end{\documentstyle}
