///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file PickRegion.h 
 * \brief Contains the definition of the Core::PickRegion class and its derived classes. 
 */

#ifndef __OVITO_PICK_REGION_H
#define __OVITO_PICK_REGION_H

namespace Core {

/// The special value that is returned by Window3D::closestHit() when no object was in the picking region.
#define HIT_TEST_NONE		FLOATTYPE_MAX

/**
 * \brief Base class for picking regions used for hit testing in a Window3D.
 * 
 * A picking region defines a 2d region in a viewport window that
 * is used for hit-testing with the 3d scene geometry.
 * 
 * The abstract PickRegion class is the base class for several picking region
 * shapes (PickRegion::RegionType). The shape of a PickRegion instance can be queried using the PickRegion::type()
 * method.
 * 
 * \author Alexander Stukowski
 * \sa Window3D::setPickingRegion()
 */
class PickRegion
{
public:
	/// Possible shapes of pick regions.
	enum RegionType {
		POINT,		//< Small circle region, only a few pixels wide.
		RECTANGLE,	//< Large rectangle region.
	};

	/// \brief Returns the type of this region.
	/// \note The instance can be cast to the appropriate sub-class type.
	RegionType type() const { return _type; }

protected:
	/// \brief Standard constructor that assign the region type.
	/// \param type The shape of the region. Must be provided by the sub-class constructor.	
	PickRegion(RegionType type) : _type(type) {}

	/// Stores the shape of the region.
	RegionType _type;
};

/**
 * \brief This pick region type describes a small circle around a click point.
 * 
 * \author Alexander Stukowski
 */
class PointPickRegion : public PickRegion 
{
public:
	/// \brief Initializes the point region with a center point and a radius.
	/// \param center The center of the picking circle in window coordinates.
	/// \param radius The radius of the picking circle in pixels.
	PointPickRegion(const Point2I& center, int radius = 5) : PickRegion(POINT), _center(center), _radius(radius) {}

	/// \brief Returns the radius in pixels of the point region.
	int radius() const { return _radius; }

	/// \brief Returns the center of the point region.
	const Point2I& center() const { return _center; }

protected:
	/// The center point for hit testing.
	Point2I _center;
	
	/// The size of the circular region in pixels.
	int _radius;
};

/**
 * \brief This pick region type describes a rectangle.
 * 
 * \author Alexander Stukowski
 */
class RectanglePickRegion : public PickRegion 
{
public:
	/// \brief Initializes the rectangle region with two corner points.
	/// \param p1 The first corner of the rectangle in window coordinates.
	/// \param p2 The second corner of the rectangle in window coordinates.
	/// \param crossing Sepcifies whether a geometric primitive must be completely inside the rectangle to generate a hit.
	RectanglePickRegion(const Point2I& p1, const Point2I& p2, bool crossing = true) : PickRegion(RECTANGLE), _crossing(crossing),
		_rect(min(p1.X, p2.X), min(p1.Y, p2.Y), abs(p1.X - p2.X), abs(p1.Y - p2.Y)) {}

	/// \brief Returns the rectangle of the pick region
	const QRect& rect() const { return _rect; }

	/// \brief Returns whether objects have to be completely inside the rectangle or not.
	bool crossing() const { return _crossing; }
	
protected:
	
	/// The rectangle.
	QRect _rect;
	
	/// Indicates if objects must be crossed by the rectangle or must lay
	/// completely inside the rectangle.
	bool _crossing;	
};

};

#endif // __OVITO_PICK_REGION_H
