///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ShapeObject.h
 * \brief Contains the definition of the Core::ShapeObject class.
 */

#ifndef __OVITO_SHAPE_OBJECT_H
#define __OVITO_SHAPE_OBJECT_H

#include <core/Core.h>
#include <core/scene/bezier/BezierShape.h>
#include "../SceneObject.h"

namespace Core {

/**
 * \brief A scene object that consists of a static BezierShape.
 *
 * \author Alexander Stukowski
 *
 * \sa BezierShape
 * \sa SimpleShapeObject
 */
class CORE_DLLEXPORT ShapeObject : public SceneObject
{
public:

	/// \brief Default constructor that creates an empty ShapeObject.
	/// \param isLoading Indicates whether the object is being loaded from a file.
	///                  This parameter is only used by the object serialization system.
	ShapeObject(bool isLoading = false) : SceneObject(isLoading), _shapeValidity(TimeForever) {}

	/// \brief Asks the object for its validity interval at the given time.
	/// \param time The animation time at which the validity interval should be computed.
	/// \return The maximum time interval that contains \a time and during which the object is valid.
	///
	/// This implementation just returns the validity interval set for the shape using setObjectValidity().
	/// \sa setObjectValidity()
	virtual TimeInterval objectValidity(TimeTicks time) {
		return _shapeValidity.contains(time) ? _shapeValidity : TimeNever;
	}

	/// \brief Sets the validity interval for the internal shape of this object.
	/// \return interval The new time interval during which the internal shape is considered valid.
	/// \sa objectValidity()
	void setObjectValidity(TimeInterval interval) { _shapeValidity = interval; }

	/// \brief Makes the object render itself into a viewport.
	/// \param time The animation time at which to render the object
	/// \param contextNode The node context used to render the object.
	/// \param vp The viewport to render in.
	///
	/// This implementation renders the internal BezierShape in the viewport.
	virtual void renderObject(TimeTicks time, ObjectNode* contextNode, Viewport* vp);

	/// \brief Asks the object whether it should show in a rendered image.
	/// \return Returns \c false since shapes are not renderable.
	virtual bool isRenderable() { return false; }

	/// \brief Computes the bounding box of the object.
	/// \param time The animation time for which the bounding box should be computed.
	/// \param contextNode The scene node to which this scene object belongs to.
	/// \return The bounding box of the object in local object coordinates.
	///
	/// This implementation returns the bounding box of the internal BezierShape.
	virtual Box3 boundingBox(TimeTicks time, ObjectNode* contextNode) {
		return shape().boundingBox();
	}

	/// \brief Allows direct read-write access to the internal shape.
	/// \return A reference to the internal BezierShape. This can be used to modify the shape.
	BezierShape& shape() { return _shape; }

	/// \brief Allows direct read access to the shape.
	/// \return A const reference to the internal BezierShape. This can be used to work with the shape.
	const BezierShape& shape() const { return _shape; }

	/// \brief Sets the shape of the ShapeObject.
	/// \param shape The new shape. The method creates an internal copy of the shape.
	/// \note When setting the shape the validity interval should also be updated using setObjectValidity().
	/// \sa shape()
	void setShape(const BezierShape& shape) { _shape = shape; }

protected:

	/// Saves the class' contents to the given stream.
	virtual void saveToStream(ObjectSaveStream& stream);

	/// Loads the class' contents from the given stream.
	virtual void loadFromStream(ObjectLoadStream& stream);

	/// Creates a copy of this object.
	virtual RefTarget::SmartPtr clone(bool deepCopy, CloneHelper& cloneHelper);

protected:

	/// The validity interval of the shape.
	TimeInterval _shapeValidity;

	/// The internal shape object.
	BezierShape _shape;

private:

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ShapeObject)
};

};

#endif // __OVITO_SHAPE_OBJECT_H
