///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file AbstractCameraObject.h 
 * \brief Contains the definition of the Core::AbstractCameraObject class. 
 */

#ifndef __OVITO_ABSTRACT_CAMERA_OBJECT_H
#define __OVITO_ABSTRACT_CAMERA_OBJECT_H

#include <core/Core.h>
#include "SceneObject.h"

namespace Core {
	
struct CameraViewDescription;	// defined in Viewport.h
	
/**
 * \brief Abstract base class for camera objects.
 * 
 * Camera scene objects can be used as view definition sources for rendering.
 * 
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT AbstractCameraObject : public SceneObject
{
public:
	
	/// \brief Default constructor.
	/// \param isLoading Indicates whether the object is being loaded from a file. 
	///                  This parameter is only used by the object serialization system.	
	AbstractCameraObject(bool isLoading = false) : SceneObject(isLoading) {}

	/// \brief Returns a structure that describes the camera view like transformation,
	///        orientation, viewing angle and projection type.
	/// \param[in] time The animation at which the camera view should be returned.
	/// \param[in,out] descriptor The structure that is to be filled with the view information.
	///     The following fields of the CameraViewDescription structure must have been filled in before
	///     the method is called:
	///   - CameraViewDescription::aspectRatio (The aspect ratio (height/width) of the viewport)
	///   - CameraViewDescription::viewAffineTransformation (The world to view space transformation)
	///   - CameraViewDescription::znear (The distance to the bounding box of the scene in view space)
	///   - CameraViewDescription::zfar (The distance to the back side of the bounding box of the scene in view space)
	///     The validity interval field is reduced by this method to the time interval with a constant view.   
	virtual void getCameraDescription(TimeTicks time, CameraViewDescription& descriptor) = 0;

private:

	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(AbstractCameraObject)
};


};

#endif // __OVITO_ABSTRACT_CAMERA_OBJECT_H
