///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/** 
 * \file ImporterExporter.h 
 * \brief Contains the definition of the Core::ImporterExporter class. 
 */

#ifndef __OVITO_IMPORTER_EXPORTER_H
#define __OVITO_IMPORTER_EXPORTER_H

#include <core/Core.h>
#include <core/reference/RefTarget.h>
#include <core/data/DataSet.h>

namespace Core {

/**
 * \brief Virtual base class for import/export plugins.
 * 
 * \author Alexander Stukowski
 * \sa ImportExportManager
 */
class CORE_DLLEXPORT ImporterExporter : public RefTarget
{
protected:
	
	/// \brief The default constructor.
	/// \param isLoading Indicates whether the object is being loaded from a file. 
	///                  This parameter is only used by the object serialization system.
	ImporterExporter(bool isLoading = false) : RefTarget(isLoading) {}

public:

	/// \brief Imports a file into the scene.
	/// \param filePath The absolute path of the file to import.
	/// \param scene The scene into which the file contents should be imported.
	/// \param suppressDialogs Controls the suppression of any dialog boxes. This is used when
	///                        the function is invoked from a script.
	/// \return \c true if the file has been successfuly imported.
	//	        \c false if the import has been aborted by the user.
	/// \throw Exception when the import has failed.
	virtual bool importFile(const QString& filePath, DataSet* scene, bool suppressDialogs = false) {
		throw Exception("Import is not implemented for this file type.");
	}

	/// \brief Exports the scene to a file.
	/// \param filePath The absolute path of the file to export to.
	/// \param scene The scene which should be exported.
	/// \param suppressDialogs Controls the suppression of any dialog boxes. This is used when
	///                        the function is invoked from a script.
	/// \return \c true if the file has been successfuly exported.
	///         \c false if the export has been aborted by the user.
	/// \throw Exception when the export has failed.
	virtual bool exportToFile(const QString& filePath, DataSet* scene, bool suppressDialogs = false) {
		throw Exception("Export is not implemented for this file type.");
	}

	/// \brief Checks if the given file has format that can be read by this importer.
	/// \param filePath The absolute path of the file to check.
	/// \return \c true if the file can be parsed.
	//	        \c false if the file some unknown format.
	/// \throw Exception when the check has failed.
	virtual bool checkFileFormat(const QString& filePath) { return false; }

private:
	Q_OBJECT
	DECLARE_ABSTRACT_PLUGIN_CLASS(ImporterExporter)
};

};

#endif // __OVITO_IMPORTER_EXPORTER_H
