///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/utilities/ProgressIndicator.h>

#include "XYZWriter.h"
#include "XYZWriterSettingsDialog.h"
#include <atomviz/atoms/AtomsObject.h>

namespace AtomViz {

IMPLEMENT_SERIALIZABLE_PLUGIN_CLASS(XYZWriter, MultiFileWriter)

/******************************************************************************
* Opens the settings dialog for this writer.
******************************************************************************/
bool XYZWriter::showSettingsDialog(AtomsObject* atomsObj, QWidget* parent)
{
	XYZWriterSettingsDialog dialog(this, atomsObj, parent);
	if(dialog.exec() != QDialog::Accepted)
		return false;
	return true;
}

/******************************************************************************
* Writes the output file.
******************************************************************************/
bool XYZWriter::writeAtomsFile(const QString& filepath, DataSet* dataset, const QVector<TimeTicks>& exportFrames, bool suppressDialogs)
{
	MsgLogger() << "Opening XYZ file" << filepath << "for writing." << endl;

	QFile stream(filepath);
	if(!stream.open(QIODevice::WriteOnly|QIODevice::Text))
		throw Exception(tr("Failed to open the file %1 for writing: %2").arg(filepath, stream.errorString()));

	ProgressIndicator progress(QString(), exportFrames.size() * 100, suppressDialogs);

	Q_FOREACH(TimeTicks time, exportFrames) {
		int frame = time / dataset->animationSettings()->ticksPerFrame();
		progress.setLabelText(tr("Writing XYZ file (frame %1)").arg(frame));
		if(progress.isCanceled()) return false;

		// Extract the atoms to be exported from the scene.
		PipelineFlowState flowState = retrieveAtoms(dataset, time);
		AtomsObject* atoms = dynamic_object_cast<AtomsObject>(flowState.result());
		if(atoms == NULL)
			throw Exception(tr("The scene does not contain any atoms that could be exported (at animation frame %1).").arg(frame));
		if(progress.isCanceled()) return false;

		DataRecordWriterHelper helper(&channelMapping(), atoms);

		int numAtoms = atoms->atomsCount();
		stream.write(QByteArray::number(numAtoms));
		stream.putChar('\n');

		// Write comment line that contains information about the simulation cell geometry.
		TimeInterval interval;
		AffineTransformation simCell = atoms->simulationCell()->cellMatrix();
		array<bool, 3> pbc = atoms->simulationCell()->periodicity();
		stream.write(QString("Frame %1  ").arg(frame).toLocal8Bit());
		stream.write(QString("cell_orig %1 %2 %3 ").arg(simCell.getTranslation().X).arg(simCell.getTranslation().Y).arg(simCell.getTranslation().Z).toLocal8Bit());
		stream.write(QString("cell_vec1 %1 %2 %3 ").arg(simCell.column(0).X).arg(simCell.column(0).Y).arg(simCell.column(0).Z).toLocal8Bit());
		stream.write(QString("cell_vec2 %1 %2 %3 ").arg(simCell.column(1).X).arg(simCell.column(1).Y).arg(simCell.column(1).Z).toLocal8Bit());
		stream.write(QString("cell_vec3 %1 %2 %3 ").arg(simCell.column(2).X).arg(simCell.column(2).Y).arg(simCell.column(2).Z).toLocal8Bit());
		stream.write(QString("pbc %1 %2 %3 ").arg(pbc[0]).arg(pbc[1]).arg(pbc[2]).toLocal8Bit());
		stream.putChar('\n');

		int progressStartValue = progress.value();
		for(int i=0; i<numAtoms; i++) {

			// Update progress indicator.
			if((i % 1000) == 0) {
				progress.setValue(i * 100 / numAtoms + progressStartValue);
				progress.isCanceled();
			}

			helper.writeAtom(i, stream);
			stream.putChar('\n');
		}
	}
	return true;
}


};

