/*
 *  Copyright (c) 2010 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Parser.h"

#include <fstream>

#include <GTLCore/CompilationMessages.h>
#include <GTLCore/Debug.h>
#include <GTLCore/TimeController.h>
#include <GTLCore/TimeMachine.h>

#include <OpenShiva/Kernel.h>
#include <GTLFragment/Metadata.h>
#include <OpenShiva/SourcesCollection.h>

using namespace shivanimator;

Parser::Parser() : m_timeMachine(new GTLCore::TimeMachine), m_kernel(0), m_start(0), m_end(10), m_frameRate(20)
{
}

Parser::~Parser()
{
  delete m_timeMachine;
  delete m_kernel;
}

GTLCore::TimeMachine* Parser::timeMachine()
{
  return m_timeMachine;
}

OpenShiva::Kernel* Parser::kernel()
{
  return m_kernel;
}

#define CHECK_ELEMENTS_SIZE_EQUAL(_n) \
  if(elements.size() != _n) \
  { \
    std::cerr << line << ": invalid number of elements for '" << elements[0] << "'" << std::endl; \
    return false; \
  }


bool Parser::parse(const GTLCore::String& filename)
{
  std::ifstream in;
  in.open(filename.c_str() );
  if(not in.is_open())
  {
    std::cerr << "Failed to open file: " << filename << std::endl;
    return false;
  }
  std::string str;
  std::getline(in,str);
  int line = 0;
  while ( in ) {
    ++line;
    GTLCore::String line = str;
    if(line[0] != '#')
    {
      std::vector<GTLCore::String> elements = line.split("= ");
      GTL_DEBUG("There are " << elements.size() << " in" << str );
      
      GTLCore::String name = elements[0];
      if(name == "Version")
      { // Ignored for now
      } else if(name == "Kernel") {
        CHECK_ELEMENTS_SIZE_EQUAL(2);
        OpenShiva::SourcesCollection collection;
        OpenShiva::Source source = collection.source(elements[1]);
        if(source.sourceType() == OpenShiva::Source::InvalidSource)
        {
          std::cerr << line << ": unknown kernel '" << elements[1] << "'" << std::endl;
          return false;
        }
        m_kernel = new OpenShiva::Kernel(4);
        m_kernel->setSource(source);
        m_kernel->compile();
        if(not m_kernel->isCompiled())
        {
          std::cerr << "Compilation failure in kernel '" << elements[1] << "'" << std::endl;
          std::cerr << m_kernel->compilationMessages().toString() << std::endl;
          return false;
        }
      } else if(name == "Start") {
        CHECK_ELEMENTS_SIZE_EQUAL(2);
        m_start = elements[1].toFloat();
      } else if(name == "End") {
        CHECK_ELEMENTS_SIZE_EQUAL(2);
        m_end = elements[1].toFloat();
      } else if(name == "Framerate") {
        CHECK_ELEMENTS_SIZE_EQUAL(2);
        m_frameRate = elements[1].toInt();
      } else {
        if(not m_kernel) {
          std::cerr << line << ": no kernel defined" << std::endl;
          return false;
        }
        if(elements.size() < 2)
        {
          std::cerr << line << ": invalid number of elements for '" << elements[0] << "'" << std::endl;
          return false;
        }
        // Get the entry
        const GTLCore::Metadata::ParameterEntry* entry = m_kernel->metadata()->parameter(elements[0]);
        if(not entry)
        {
          std::cerr << line << ": no such parameter for kernel '" << m_kernel->name() << "'" << std::endl;
          return false;
        }
        // Check the TimeController
        const GTLCore::TimeController* controller = 0;
        if(elements[1] == "Linear")
        {
          CHECK_ELEMENTS_SIZE_EQUAL(4);
          controller = new GTLCore::LinearTimeController(elements[2].toFloat(), elements[3].toInt());
        } else if(elements[1] == "Sinusoidal")
        {
          CHECK_ELEMENTS_SIZE_EQUAL(4);
          controller = new GTLCore::SinusoidalTimeController(elements[2].toFloat(), elements[3].toFloat());
        }
        if(not controller)
        {
          std::cerr << line << ": unknow time controller '" << elements[1] << "'" << std::endl;
          return false;
        }
        m_timeMachine->startControlling(entry, controller);
      }
    }
    std::getline(in,str);
  }
  if(not m_kernel) {
    std::cerr << line << ": no kernel defined" << std::endl;
    return false;
  }
  return true;
}

double Parser::start() const
{
  return m_start;
}

double Parser::end() const
{
  return m_end;
}

int Parser::frameRate() const
{
  return m_frameRate;
}
