# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2012-2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""notes-app autopilot tests."""
from __future__ import print_function, absolute_import

import base64
try:
    from http import server
except ImportError:
    import BaseHTTPServer as server
import errno
import glob
try:
    from hashlib import md5
except ImportError:
    from md5 import md5
import os
import os.path
import socket
import threading

from autopilot.input import Mouse, Touch, Pointer
from autopilot.matchers import Eventually
from autopilot.platform import model
from autopilot.testcase import AutopilotTestCase
from testtools.matchers import Equals

from ubuntuuitoolkit.emulators import UbuntuUIToolkitEmulatorBase

from notes_app.emulators.notesapp import NotesApp
from . import imagedata


class NotesAppTestCase(AutopilotTestCase):

    """A common test case class that provides several useful methods for notes
    app tests."""

    TYPING_DELAY = 0.02

    if model() == 'Desktop':
        scenarios = [('with mouse', dict(input_device_class=Mouse))]
    else:
        scenarios = [('with touch', dict(input_device_class=Touch))]

    local_location = "../../NotesApp.qml"

    def setUp(self):
        self.pointing_device = Pointer(self.input_device_class.create())
        super(NotesAppTestCase, self).setUp()
        self.launch_app()
        self.main_window.visible.wait_for(True)
        self.main_window.close_toolbar()

    """Workaround to find the qmlscene binary via shell globbing.
       This is needed since we can't rely on qt5-default being installed on
       devices to make qmlscene available in the path"""
    def qmlscene(self):
        return glob.glob("/usr/lib/*/qt5/bin/qmlscene")[0]

    def launch_app(self):
        if os.path.exists(self.local_location):
            self.launch_app_local()
        elif os.path.exists('/usr/share/notes-app/NotesApp.qml'):
            self.launch_app_installed()
        else:
            self.launch_click_installed()

    def launch_app_local(self):
        self.app = self.launch_test_application(
            self.qmlscene(),
            "-I",
            "../../src",
            "../../NotesApp.qml",
            emulator_base=UbuntuUIToolkitEmulatorBase)

    def launch_app_installed(self):
        self.app = self.launch_test_application(
            self.qmlscene(),
            "/usr/share/notes-app/NotesApp.qml",
            "--desktop_file_hint=/usr/share/applications/notes-app.desktop",
            app_type='qt',
            emulator_base=UbuntuUIToolkitEmulatorBase)

    def launch_click_installed(self):
        self.app = self.launch_click_package(
            "com.ubuntu.notes",
            emulator_base=UbuntuUIToolkitEmulatorBase)

    def launch_and_quit_app(self):
        self.launch_app()
        self.main_window.visible.wait_for(True)

        # When calling launch_app an instance of the spawned process
        # control object will be stored in self.app.process, and a cleanup
        # handler will be registered that essentially kills the process.
        # Therefore, by triggering the cleanup handler here we're killing the
        # process and removing the handler, which allows a clean launch of
        # the process during regular test setup.
        self.doCleanups()

    def ensure_db(self):
        path = os.path.expanduser("~/.local/share/com.ubuntu.notes/Databases")
        db = os.path.join(path, md5(b"notes").hexdigest() + ".sqlite")
        if not os.path.exists(db):
            self.launch_and_quit_app()
        self.assertTrue(os.path.exists(db))
        return db

    def assert_osk_eventually_shown(self):
        if model() != 'Desktop':
            keyboardRectangle = self.main_window.get_keyboard_rectangle()
            self.assertThat(keyboardRectangle.state,
                            Eventually(Equals("shown")))

    def assert_osk_eventually_hidden(self):
        if model() != 'Desktop':
            keyboardRectangle = self.main_window.get_keyboard_rectangle()
            self.assertThat(keyboardRectangle.state,
                            Eventually(Equals("hidden")))

    @property
    def main_window(self):
        return self.app.select_single(NotesApp)

    def assert_note_eventually_expanded(self, note):
        # To properly ensure a note is expanded we have to wait for the
        # height to equal actualExpandedHeight because that is the point
        # when the expanding animation is finished and the note reaches
        # its final expanded geometry.
        # Just testing for isExpanded does not work as the property
        # changes immediately and then the animation is triggered.
        # This applies as well for collapsing the note.

        note.height.wait_for(note.actualExpandedHeight)
        self.assertThat(note.isExpanded, Eventually(Equals(True)))

    def assert_note_eventually_collapsed(self, note):
        note.height.wait_for(note.collapsedHeight)
        self.assertThat(note.isExpanded, Eventually(Equals(False)))



class HTTPRequestHandler(server.BaseHTTPRequestHandler):

    """
    A custom HTTP request handler that serves GET resources.
    """

    def send_image(self, data):
        self.send_header("Content-Type", "image/png")
        self.end_headers()
        self.wfile.write(data)

    def do_GET(self):
        if self.path == "/image.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_DATA))
        elif self.path == "/large_wide.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_LARGE_WIDE))
        elif self.path == "/small_wide.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_SMALL_WIDE))
        elif self.path == "/large_high.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_LARGE_HIGH))
        elif self.path == "/small_high.png":
            self.send_response(200)
            self.send_image(base64.b64decode(imagedata.IMAGE_SMALL_HIGH))
        else:
            self.send_error(404)


class HTTPServerInAThread(threading.Thread):

    """
    A simple custom HTTP server run in a separate thread.
    """

    def __init__(self):
        super(HTTPServerInAThread, self).__init__()
        port = 12345
        self.server = None
        while (self.server is None) and (port < 12350):
            try:
                self.server = server.HTTPServer(("", port),
                                                HTTPRequestHandler)
            except socket.error as error:
                if (error.errno == errno.EADDRINUSE):
                    print("Port %d is already in use" % port)
                    port += 1
                else:
                    print(os.strerror(error.errno))
                    raise
        if self.server is None:
            raise SystemError("Unable to start local HTTP server")
        self.server.allow_reuse_address = True

    @property
    def port(self):
        return self.server.server_port

    def run(self):
        print("now serving on port %d" % self.port)
        self.server.serve_forever()

    def shutdown(self):
        self.server.shutdown()
        self.server.server_close()


class NotesTestCaseBaseWithHTTPServer(NotesAppTestCase):

    """
    A specialization of the common test case class that runs
    a simple custom HTTP server in a separate thread.
    """

    def start_server(self):
        self.server = HTTPServerInAThread()
        self.server.start()
        self.addCleanup(self.server.shutdown)
        self.base_url = "http://localhost:%d" % self.server.port
