/***************************** LICENSE START ***********************************

 Copyright 2012 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#ifndef VERTPROF_H
#define VERTPROF_H

/********************************************************************************
  Application for Vertical Profile.
  It takes a GRIB file as input (in addition to the values given by the user),
  and produces an NetCDF file as output. There is no metadata on output request,
  only the path to NetCDF file.

  First the GRIB while is read, and info about all parameters and level are stored.
  Then several flags is set, based on the data and the input request, to determine
  how the application should run.

  The file is rewound, and sorted by param, date, time, step and expver. All fields
  with the same values for the sorting parameters are one plot. If any of these
  values change, all data are written out as netcdf variables with names put
  together from the sorting values.
********************************************************************************/

#include "Xsect.h"

class Vprofile : public Xsect {

public:

   // Constructors
   Vprofile();
   Vprofile(const char* kw);

   bool writeLevelInfoNetcdf ( MvNetCDF&, ApplicationInfo&, ParamMap&, MvField*, double* );
   bool writeGeoCoordsNetcdf ( MvNetCDF&, ApplicationInfo& );
   bool writeGlobalAttributesNetcdf ( MvNetCDF&, ParamMap&, ApplicationInfo& );
   bool generateExtraData( MvNetCDF&, ApplicationInfo&, ParamMap&, MvFieldSetIterator& );
   bool generateDerivedData( MvNetCDF&, ApplicationInfo&, ParamMap&, MvFieldSetIterator& );
   bool generatePressureFromLNSP( MvNetCDF&, ApplicationInfo&, ParamMap&, MvFieldSetIterator&);
   bool generateQ2( MvNetCDF&, ApplicationInfo&, ParamMap& );
   bool generateOrography( MvNetCDF&, ApplicationInfo&, ParamMap& );
   bool generateSurftype( MvNetCDF&, ApplicationInfo&, ParamMap& );
   bool generateData( ApplicationInfo&, ParamMap&, MvNetCDF&, MvField&, const string& );

   // Get geogrpahical coordinates and application parameters
   bool getCoordinates( MvRequest&, ApplicationInfo& );
   bool getAppParameters( MvRequest&, ApplicationInfo& );

   // Create output request
   MvRequest createOutputRequest( ApplicationInfo&, ParamInfo* );

   int computeGeographicalPoints( ApplicationInfo&, MvField* = 0 );

   bool fillValues( ApplicationInfo&, MvField&, double* );

   bool fieldConsistencyCheck ( MvField&, ApplicationInfo& );

   // Check if parameters between two requests are consistent
   bool consistencyCheck( MvRequest&, MvRequest& );

   string titleVariable( ApplicationInfo&, ParamInfo*);

private:

   int inputMode_;   // VP_POINT/VP_GRIDPOINT/VP_AREA/VP_AREA2
   int outputMode_;  // VP_NORMAL/VP_RTTOV

  // RTTOV specific parameters
   double waterType_, cloudTP_, cloudFraction_;
};

//---------------------------------------------------------------------

class VprofileM3 : public Vprofile {

public:

   VprofileM3() : Vprofile("PM_VPROF") {}

   void serve( MvRequest&, MvRequest& );
};
#endif
