/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_return_poolindex_without_gsi.h
    \brief  This interface of LCMAPS takes user DN+FQANS and a counter as input and
            returns the uid, gids and poolindex, i.e. leaseid.
    \author Martijn Steenbakkers for EGEE.

    This header contains the following LCMAPS library functions:
    -# lcmaps_return_poolindex_without_gsi
        Returns the poolindex based on fixed arguments:
                security context, buffer and buffer length

    In general the behaviour of LCMAPS can be influenced by setting the following
    environment variables:
        Env variable                        Description
        (default value)
        ---------------                     -----------
        LCMAPS_LOG_FILE                     The logfile to which lcmaps log
        ("/var/log/lcmaps.log")             messages will be written

        LCMAPS_DB_FILE                      the location (name) of the lcmaps policy file
        (LCMAPS_ETC_DIR"/lcmaps.db")

        LCMAPS_POLICY_NAME                  the list of the lcmaps policies (in the
        ("acquisition_policy")              policy file) that should be evaluated until
                                            a policy succeeds. The policies are separated
                                            by ':', e.g. "das_voms:default:".

        LCMAPS_LOG_STRING                   A string that will be prepended to all
        ("poolindex :")                     LCMAPS logging messages. If this variable is not
                                            set a default will be taken which includes a 
                                            timestamp

        LCMAPS_DEBUG_LEVEL                  The debug level influences the logging
        ("0")                               verbosity. values 0-5.

        LCMAPS_MODULES_DIR                  The directory in which modules can be found.
        ("")                                Note: may be overwritten by what is in
                                            the policy file (the convential method)

        LCMAPS_ETC_DIR                      The directory of the lcmaps policy files
        ("/opt/glite/etc/lcmaps")           and other config files.

        GRIDMAPDIR                          The directory in which the poolaccount leases
        ("")                                are registered.
                                            Note: may be overwritten by what is in
                                            the policy file (the convential method)

    \ingroup LcmapsInterface
*/

#ifndef LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_H
#define LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_H

#ifndef LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_C
#   define LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_C_STORAGE_CLASS extern "C"
#else
#   define LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_C_STORAGE_CLASS
#endif

/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_account.h"

/******************************************************************************
                   Module definition
 *****************************************************************************/


/******************************************************************************
Function:   lcmaps_return_poolindex_without_gsi
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The call will result in a failure if no poolindex is found (e.g. in the case of
    statically assigned accounts).

Parameters:
    user_dn:          The DN of the user (input)
    fqan_list:        The list of FQANs (type char**) (input)
    nfqan:            The number of FQANs in the list (input)
    plcmaps_account:  A structure that contains account information: (output)
                      uid, gids (primary and secondary) and the poolindex
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure (if no poolindex is found)
******************************************************************************/
/*!
    \fn lcmaps_return_poolindex_without_gsi(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        lcmaps_account_info_t * plcmaps_account
    )

    \brief LCMAPS returns the poolindex and account information
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The call will result in a failure if no poolindex is found (e.g. in the case of
    statically assigned accounts).

    \param user_dn
            The DN of the user (input)
    \param fqan_list
            The list of FQANs (type char**) (input)
    \param nfqan
            The number of FQANs in the list (input)
    \param plcmaps_account
            A structure that contains account information: (output)
            uid, gids (primary and secondary) and the poolindex
            Please use lcmaps_account_info_clean() to clean this
            structure after use.

    \retval 0 success.
    \retval 1 failure (if no poolindex is found).
*/
LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_C_STORAGE_CLASS int
lcmaps_return_poolindex_without_gsi(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        lcmaps_account_info_t * plcmaps_account
);

/******************************************************************************
Function:   lcmaps_return_poolindex_with_mapcounter
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The call will result in a failure if no poolindex is found (e.g. in the case of
    statically assigned accounts).
    An additional counter may be added to enable multiple acccounts per set of user
    credentials.

Parameters:
    user_dn:          The DN of the user (input)
    fqan_list:        The list of FQANs (type char**) (input)
    nfqan:            The number of FQANs in the list (input)
    mapcounter:       The counter which will be added to the poolindex, effectively enabling
                      multiple account mappings
    plcmaps_account:  A structure that contains account information: (output)
                      uid, gids (primary and secondary) and the poolindex
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure (if no poolindex is found)
******************************************************************************/
/*!
    \fn lcmaps_return_poolindex_with_mapcounter(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        int mapcounter,
        lcmaps_account_info_t * plcmaps_account
    )

    \brief LCMAPS returns the poolindex and account information
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex, alternatively named leaseid, to the calling application.
    The call will result in a failure if no poolindex is found (e.g. in the case of
    statically assigned accounts).
    An additional counter may be added to enable multiple acccounts per set of user
    credentials.

    \param user_dn
            The DN of the user (input)
    \param fqan_list
            The list of FQANs (type char**) (input)
    \param nfqan
            The number of FQANs in the list (input)
    \param mapcounter
            The counter which will be added to the poolindex, effectively enabling
            multiple account mappings
    \param plcmaps_account
            A structure that contains account information: (output)
            uid, gids (primary and secondary) and the poolindex
            Please use lcmaps_account_info_clean() to clean this
            structure after use.

    \retval 0 success.
    \retval 1 failure (if no poolindex is found).
*/
LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_C_STORAGE_CLASS int
lcmaps_return_poolindex_with_mapcounter(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        int mapcounter,
        lcmaps_account_info_t * plcmaps_account
);
/******************************************************************************
Function:   lcmaps_return_account_without_gsi
Description:
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex (if available), alternatively named leaseid, to the calling application.
    The call will not result in a failure if no poolindex is found, so it supports
    poolaccounts and statically assigned accounts simultaneously.
    An additional counter may be added to enable multiple acccounts per set of user
    credentials.

Parameters:
    user_dn:          The DN of the user (input)
    fqan_list:        The list of FQANs (type char**) (input)
    nfqan:            The number of FQANs in the list (input)
    mapcounter:       The counter which will be added to the poolindex, effectively enabling
                      multiple account mappings
    plcmaps_account:  A structure that contains account information: (output)
                      uid, gids (primary and secondary) and the poolindex
                      Please use lcmaps_account_info_clean() to clean this
                      structure after use.
Returns:
    0: success
    1: failure
******************************************************************************/
/*!
    \fn lcmaps_return_account_without_gsi(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        int mapcounter,
        lcmaps_account_info_t * plcmaps_account
    )

    \brief LCMAPS returns the poolindex and account information
    If LCMAPS is invoked via this interface it will return the user account info
    and poolindex (if available), alternatively named leaseid, to the calling application.
    The call will not result in a failure if no poolindex is found, so it supports
    poolaccounts and statically assigned accounts simultaneously.
    An additional counter may be added to enable multiple acccounts per set of user
    credentials.

    \param user_dn
            The DN of the user (input)
    \param fqan_list
            The list of FQANs (type char**) (input)
    \param nfqan
            The number of FQANs in the list (input)
    \param mapcounter
            The counter which will be added to the poolindex, effectively enabling
            multiple account mappings
    \param plcmaps_account
            A structure that contains account information: (output)
            uid, gids (primary and secondary) and the poolindex
            Please use lcmaps_account_info_clean() to clean this
            structure after use.

    \retval 0 success.
    \retval 1 failure.
*/
LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_C_STORAGE_CLASS int
lcmaps_return_account_without_gsi(
        char * user_dn,
        char ** fqan_list,
        int nfqan,
        int mapcounter,
        lcmaps_account_info_t * plcmaps_account
);
#endif /* LCMAPS_RETURN_POOLINDEX_WITHOUT_GSI_H */
