"""Tests for bootstrapping juju on openstack

Bootstrap touches a lot of the other parts of the provider, including machine
launching, security groups and so on. Testing this in an end-to-end fashion as
is done currently duplicates many checks from other more focussed tests.
"""

import logging

from juju.lib import mocker, testing, serializer

from juju.providers.openstack.machine import NovaProviderMachine
from juju.providers.openstack.tests import OpenStackTestMixin


class OpenStackBootstrapTest(OpenStackTestMixin, testing.TestCase):

    def expect_verify(self):
        self.expect_swift_put("testing/bootstrap-verify",
            "storage is writable")

    def expect_provider_state_fresh(self):
        self.expect_swift_get("testing/provider-state")
        self.expect_verify()

    def expect_create_group(self):
        self.expect_nova_post("os-security-groups",
            {'security_group': {
                'name': 'juju-testing',
                'description': 'juju group for testing',
            }},
            response={'security_group': {
                'id': 1,
            }})
        self.expect_nova_post("os-security-group-rules",
            {'security_group_rule': {
                'parent_group_id': 1,
                'ip_protocol': "tcp",
                'from_port': 22,
                'to_port': 22,
            }},
            response={'security_group_rule': {
                'id': 144, 'parent_group_id': 1,
            }})
        self.expect_nova_post("os-security-group-rules",
            {'security_group_rule': {
                'parent_group_id': 1,
                'group_id': 1,
                'ip_protocol': "tcp",
                'from_port': 1,
                'to_port': 65535,
            }},
            response={'security_group_rule': {
                'id': 145, 'parent_group_id': 1,
            }})
        self.expect_nova_post("os-security-group-rules",
            {'security_group_rule': {
                'parent_group_id': 1,
                'group_id': 1,
                'ip_protocol': "udp",
                'from_port': 1,
                'to_port': 65535,
            }},
            response={'security_group_rule': {
                'id': 146, 'parent_group_id': 1,
            }})

    def expect_create_machine_group(self, machine_id):
        machine = str(machine_id)
        self.expect_nova_post("os-security-groups",
            {'security_group': {
                'name': 'juju-testing-' + machine,
                'description': 'juju group for testing machine ' + machine,
            }},
            response={'security_group': {
                'id': 2,
            }})

    def _match_server(self, data):
        userdata = data['server'].pop('user_data').decode("base64")
        self.assertEqual("#cloud-config", userdata.split("\n", 1)[0])
        # TODO: assertions on cloud-init content
        # cloud_init = yaml.load(userdata)
        self.assertEqual({'server': {
                'flavorRef': 1,
                'imageRef': 42,
                'name': 'juju testing instance 0',
                'security_groups': [
                    {'name': 'juju-testing'},
                    {'name': 'juju-testing-0'},
                ],
            }},
            data)
        return True

    def expect_launch(self):
        self.expect_nova_get("flavors/detail",
            response={'flavors': self.default_flavors})

        self.expect_nova_post("servers", mocker.MATCH(self._match_server),
            code=202, response={'server': {
                'id': '1000',
                'status': "PENDING",
                'addresses': {'private': [{'version': 4, 'addr': "4.4.4.4"}]},
            }})
        self.expect_swift_put("testing/juju_master_id", "1000")
        self.expect_nova_get("os-floating-ips",
            response={'floating_ips': [
                {'id': 80, 'instance_id': None, 'ip': "8.8.8.8"}
            ]})
        self.expect_nova_post("servers/1000/action",
            {"addFloatingIp": {"address": "8.8.8.8"}}, code=202)
        self.expect_swift_put("testing/provider-state",
            serializer.dump({'zookeeper-instances': ['1000']}))

    def _check_machine(self, machine_list):
        [machine] = machine_list
        self.assertTrue(isinstance(machine, NovaProviderMachine))
        self.assertEqual(machine.instance_id, '1000')

    def bootstrap(self):
        provider = self.get_provider()
        return provider.bootstrap(
            self.constraint_set.load({'instance-type': None}))

    def test_bootstrap_clean(self):
        """Bootstrap from a clean slate makes groups and zookeeper instance"""
        self.expect_provider_state_fresh()
        self.expect_nova_get("os-security-groups",
            response={'security_groups': []})
        self.expect_create_group()
        self.expect_create_machine_group(0)
        self.expect_launch()
        self.mocker.replay()

        log = self.capture_logging("juju.common", level=logging.DEBUG)
        deferred = self.bootstrap()
        deferred.addCallback(self._check_machine)

        def check_log(_):
            log_text = log.getvalue()
            self.assertIn("Launching juju bootstrap instance", log_text)
            self.assertNotIn("previously bootstrapped", log_text)
        return deferred.addCallback(check_log)

    def test_bootstrap_existing_group(self):
        """Bootstrap reuses an exisiting provider security group"""
        self.expect_provider_state_fresh()
        self.expect_nova_get("os-security-groups",
            response={'security_groups': [
                    {'name': "juju-testing", 'id': 1},
                ]})
        self.expect_create_machine_group(0)
        self.expect_launch()
        self.mocker.replay()
        return self.bootstrap().addCallback(self._check_machine)

    def test_bootstrap_existing_machine_group(self):
        """Bootstrap deletes and remakes an existing machine security group"""
        self.expect_provider_state_fresh()
        self.expect_nova_get("os-security-groups",
            response={'security_groups': [
                    {'name': "juju-testing-0", 'id': 3},
                ]})
        self.expect_create_group()
        self.expect_nova_delete("os-security-groups/3")
        self.expect_create_machine_group(0)
        self.expect_launch()
        self.mocker.replay()
        return self.bootstrap().addCallback(self._check_machine)

    def test_existing_machine(self):
        """A preexisting zookeeper instance is returned if present"""
        self.expect_swift_get("testing/provider-state",
            response=serializer.dump({'zookeeper-instances': ['1000']}))
        self.expect_nova_get("servers/1000",
            response={'server': {
                'id': '1000',
                'name': 'juju testing instance 0',
                'state': "RUNNING"
            }})
        self.mocker.replay()

        log = self.capture_logging("juju.common")
        self.capture_logging("juju.openstack")  # Drop to avoid stderr kipple
        deferred = self.bootstrap()
        deferred.addCallback(self._check_machine)

        def check_log(_):
            self.assertEqual("juju environment previously bootstrapped.\n",
                log.getvalue())
        return deferred.addCallback(check_log)

    def test_existing_machine_missing(self):
        """Bootstrap overwrites existing zookeeper if instance is present"""
        self.expect_swift_get("testing/provider-state",
            response=serializer.dump({'zookeeper-instances': [3000]}))
        self.expect_nova_get("servers/3000", code=404,
            response={'itemNotFound':
                {'message': "The resource could not be found.", 'code': 404}
            })
        self.expect_verify()
        self.expect_nova_get("os-security-groups",
            response={'security_groups': [
                    {'name': "juju-testing", 'id': 1},
                    {'name': "juju-testing-0", 'id': 3},
                ]})
        self.expect_nova_delete("os-security-groups/3")
        self.expect_create_machine_group(0)
        self.expect_launch()
        self.mocker.replay()

        log = self.capture_logging("juju.common", level=logging.DEBUG)
        self.capture_logging("juju.openstack")  # Drop to avoid stderr kipple
        deferred = self.bootstrap()
        deferred.addCallback(self._check_machine)

        def check_log(_):
            log_text = log.getvalue()
            self.assertIn("Launching juju bootstrap instance", log_text)
            self.assertNotIn("previously bootstrapped", log_text)
        return deferred.addCallback(check_log)
