//*****************************************************************************
//                                 CmdBase.cpp                                *
//                                -------------                               *
//  Started     : 31/08/2006                                                  *
//  Last Update : 24/12/2010                                                  *
//  Copyright   : (C) 2006 by M.S.Waters                                      *
//  Email       : M.Waters@bom.gov.au                                         *
//*****************************************************************************

//*****************************************************************************
//                                                                            *
//    This program is free software; you can redistribute it and/or modify    *
//    it under the terms of the GNU General Public License as published by    *
//    the Free Software Foundation; either version 2 of the License, or       *
//    (at your option) any later version.                                     *
//                                                                            *
//*****************************************************************************

#include "base/CmdBase.hpp"

//*****************************************************************************
// Constructor.

CmdBase::CmdBase( void ) : wxString( wxT("") )
{
  bSetDefaults( );
}

//*****************************************************************************
// Destructor.

CmdBase::~CmdBase( )
{
}

//*****************************************************************************
// Set the object attributes to their default values.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdBase::bSetDefaults( void )
{
  wxString::Empty( );

  m_osErrMsg = wxT("Invalid");

  m_eSimEng  = eSIMR_NONE;
  m_eCmdType = eCMD_NONE;

  return( TRUE );
}

//*****************************************************************************
// Check that the object attributes are valid.
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdBase::bValidate( void )
{
  m_osErrMsg.Empty( );

  if( IsEmpty( ) )
    SetErrMsg( wxT("The command is empty") );
  if( m_eSimEng == eSIMR_NONE )
    SetErrMsg( wxT("The command has no simulator engine") );
  if( m_eCmdType == eCMD_NONE )
    SetErrMsg( wxT("The command has no type") );

  return( bIsValid( ) );
}

//*****************************************************************************
// Set the command based on a complete command string.
//
// Argument List :
//   rosCmd - A reference to a wxString object
//
// Return Values :
//   TRUE  - Success
//   FALSE - Failure

bool  CmdBase::bSetString( wxString & ros )
{
  bSetDefaults( );
  assign( ros );
  bParse( );

  return( TRUE );
}

//*****************************************************************************
// Copy the contents of another CmdBase object.
//
// Argument List :
//   roCmd - A reference to a CmdBase object
//
// Return Values :
//   A reference to this object

CmdBase & CmdBase::operator = ( const CmdBase & roCmd )
{
  (wxString &) *this = (wxString &) roCmd;

  m_eSimEng  = roCmd.m_eSimEng;
  m_eCmdType = roCmd.m_eCmdType;

  m_osErrMsg = roCmd.m_osErrMsg;

  return( *this );
}

//*****************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  CmdBase::Print( const wxString & rosPrefix )
{
  cout << rosPrefix .mb_str( ) << "wxString::"
       << this     ->mb_str( ) << '\n';

  cout << rosPrefix .mb_str( ) << "m_osErrMsg : \""
       << m_osErrMsg.mb_str( ) << "\"\n";

  cout << rosPrefix .mb_str( ) << "m_eSimEng  : ";
  switch( m_eSimEng )
  {
    case eSIMR_GNUCAP  : cout << "eSIMR_GNUCAP";  break;
    case eSIMR_NGSPICE : cout << "eSIMR_NGSPICE"; break;
    case eSIMR_NONE    : cout << "eSIMR_NONE";    break;
    default            : cout << "Invalid";       break;
  }
  cout << '\n';

  cout << rosPrefix .mb_str( ) << "m_eCmdType : ";
  switch( m_eCmdType )
  {
    case eCMD_OP   : cout << "eANA_OP";   break;  // Operating point analysis
    case eCMD_DC   : cout << "eANA_DC";   break;  // DC analysis
    case eCMD_AC   : cout << "eANA_AC";   break;  // AC analysis
    case eCMD_TR   : cout << "eANA_TR";   break;  // Transient analysis
    case eCMD_FO   : cout << "eANA_FO";   break;  // Fourier analysis
    case eCMD_DI   : cout << "eANA_DI";   break;  // Distortion analysis
    case eCMD_NO   : cout << "eANA_NO";   break;  // Noise analysis
    case eCMD_PZ   : cout << "eANA_PZ";   break;  // Pole-zero analysis
    case eCMD_SE   : cout << "eANA_SE";   break;  // Sensitivity analysis
    case eCMD_TF   : cout << "eANA_TF";   break;  // Transfer function analysis

    case eCMD_OPT  : cout << "eCMD_OPT";  break;  // Options command
    case eCMD_IC   : cout << "eCMD_IC";   break;  // Initial conditions command
    case eCMD_PR   : cout << "eCMD_PR";   break;  // Print command
    case eCMD_GEN  : cout << "eCMD_GEN";  break;  // Generator command

    case eCMD_NONE : cout << "eANA_NONE"; break;  // Command type not set
    default        : cout << "Invalid";   break;  // Command type is invalid
  }
  cout << '\n';
}

//*****************************************************************************
