% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fast.prcomp.R
\name{fast.prcomp}
\alias{fast.prcomp}
\alias{fast.svd}
\title{Efficient computation of principal components and singular value
decompositions.}
\usage{
fast.prcomp(x, retx = TRUE, center = TRUE, scale. = FALSE, tol = NULL)
}
\arguments{
\item{x}{data matrix}

\item{retx}{a logical value indicating whether the rotated variables
    should be returned.}

\item{center}{a logical value indicating whether the variables
    should be shifted to be zero centered. Alternately, a vector of
    length equal the number of columns of \code{x} can be supplied.
    The value is passed to \code{scale}.}

\item{scale.}{a logical value indicating whether the variables should
    be scaled to have unit variance before the analysis takes
    place.  The default is \code{FALSE} for consistency with S, but
    in general scaling is advisable.  Alternatively, a vector of length
    equal the number of columns of \code{x} can be supplied.  The
    value is passed to \code{\link{scale}}.}

\item{tol}{a value indicating the magnitude below which components
    should be omitted. (Components are omitted if their
    standard deviations are less than or equal to \code{tol} times the
    standard deviation of the first component.)  With the default null
    setting, no components are omitted (unless \code{rank.} is specified
    less than \code{min(dim(x))}.).  Other settings for tol could be
    \code{tol = 0} or \code{tol = sqrt(.Machine$double.eps)}, which
    would omit essentially constant components.}
}
\value{
See the documetation for \code{\link[stats:prcomp]{stats::prcomp()}} or
\code{\link[=svd]{svd()}} .
}
\description{
The standard \code{\link[stats:prcomp]{stats::prcomp()}} and \code{\link[=svd]{svd()}} function are
very inefficient for wide matrixes. \code{fast.prcomp} and \code{fast.svd}
are modified versions which are efficient even for matrixes that are very
wide.
}
\details{
The current implementation of the function \code{\link[=svd]{svd()}} in S-Plus and R
is much slower when operating on a matrix with a large number of columns
than on the transpose of this matrix, which has a large number of rows. As a
consequence, \code{\link[stats:prcomp]{stats::prcomp()}}, which uses \code{\link[=svd]{svd()}}, is
also very slow when applied to matrixes with a large number of rows.

The simple solution is to use \code{\link[=La.svd]{La.svd()}} instead of
\code{\link[=svd]{svd()}}.  A suitable patch to \code{\link[stats:prcomp]{stats::prcomp()}} has been
submitted.  In the mean time, the function \code{fast.prcomp} has been
provided as a short-term work-around.

\describe{
\item{list("fast.prcomp")}{is a modified versiom of
\code{\link[stats:prcomp]{stats::prcomp()}} that calls \code{\link[=La.svd]{La.svd()}} instead of
\code{\link[=svd]{svd()}} }
\item{list("fast.svd")}{is simply a wrapper around
\code{\link[=La.svd]{La.svd()}}.  }
}
}
\examples{


  # create test matrix
  set.seed(4943546)
  nr <- 50
  nc <- 2000
  x  <- matrix( rnorm( nr*nc), nrow=nr, ncol=nc )
  tx <- t(x)

  # SVD directly on matrix is SLOW:
  system.time( val.x <- svd(x)$u )

  # SVD on t(matrix) is FAST:
  system.time( val.tx <- svd(tx)$v )

  # and the results are equivalent:
  max( abs(val.x) - abs(val.tx) )

  # Time gap dissapears using fast.svd:
  system.time( val.x <- fast.svd(x)$u )
  system.time( val.tx <- fast.svd(tx)$v )
  max( abs(val.x) - abs(val.tx) )


  library(stats)

  # prcomp directly on matrix is SLOW:
  system.time( pr.x <- prcomp(x) )

  # prcomp.fast is much faster
  system.time( fast.pr.x <- fast.prcomp(x) )

  # and the results are equivalent
  max( pr.x$sdev - fast.pr.x$sdev )
  max( abs(pr.x$rotation[,1:49]) - abs(fast.pr.x$rotation[,1:49]) )
  max( abs(pr.x$x) - abs(fast.pr.x$x)  )

  # (except for the last and least significant component):
  max( abs(pr.x$rotation[,50]) - abs(fast.pr.x$rotation[,50]) )

}
\seealso{
\code{\link[stats:prcomp]{stats::prcomp()}}, \code{\link[base:svd]{base::svd()}}, \code{\link[base:svd]{base::La.svd()}}
}
\author{
Modifications by Gregory R. Warnes \email{greg@warnes.net}
}
\keyword{algebra}
\keyword{array}
\keyword{multivariate}
