/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Some data structures (lrclist and rlilist) require
 * locking scheme that supports multiple readers or 1 writer.  Following
 * functions support this.
 */

#include <globus_io.h>
#include "globus_rls_client.h"
#include "misc.h"
#include "lock.h"
#include <syslog.h>

extern int	loglevel;

int
rls_lock_init(LOCK *lock)

{
  int	r;

  if ((r = globus_mutex_init(&lock->mtx, GLOBUS_NULL)) != 0) {
    logit(LOG_WARNING, "rls_lock_init(%X,%d): globus_mutex_init: %d", lock, r);
    return GLOBUS_RLS_GLOBUSERR;
  }
  if ((r = globus_cond_init(&lock->cond, GLOBUS_NULL)) != 0) {
    logit(LOG_WARNING, "rls_lock_init(%X,%d): globus_cond_init: %d", lock, r);
    return GLOBUS_RLS_GLOBUSERR;
  }
  lock->readers = 0;
  lock->writer = 0;
  return GLOBUS_RLS_SUCCESS;
}

void
rls_lock_destroy(LOCK *lock)

{
  globus_cond_destroy(&lock->cond);
  globus_mutex_destroy(&lock->mtx);
}

void
rls_lock_get(LOCK *lock, LOCKTYPE type)

{
  if (loglevel > 2)
    logit(LOG_DEBUG, "rls_lock_get: %X %s", lock,
	  type == readlock ? "readlock" : "writelock");
  globus_mutex_lock(&lock->mtx);
  if (type == readlock) {
    while (lock->writer)
      globus_cond_wait(&lock->cond, &lock->mtx);
    lock->readers++;
    if (loglevel > 2)
      logit(LOG_DEBUG, "rls_lock_get(%X): %d readers", lock, lock->readers);
  } else {
    while (lock->readers || lock->writer)
      globus_cond_wait(&lock->cond, &lock->mtx);
    lock->writer++;
    if (loglevel > 2)
      logit(LOG_DEBUG, "rls_lock_get(%X): writelock acquired", lock);
  }
  globus_mutex_unlock(&lock->mtx);
}

void
rls_lock_release(LOCK *lock, LOCKTYPE type)

{
  globus_mutex_lock(&lock->mtx);
  if (type == readlock) {
    if (--lock->readers == 0)
      globus_cond_broadcast(&lock->cond);
    if (loglevel > 2)
      logit(LOG_DEBUG, "rls_lock_release(%X): %d readers", lock, lock->readers);
  } else {
    lock->writer = 0;
    globus_cond_broadcast(&lock->cond);
    if (loglevel > 2)
      logit(LOG_DEBUG, "rls_lock_release(%X): writelock released", lock);
  }
  globus_mutex_unlock(&lock->mtx);
}
