// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using Microsoft.EntityFrameworkCore.Query.SqlExpressions;

// ReSharper disable once CheckNamespace
namespace Microsoft.EntityFrameworkCore.Sqlite.Query.Internal;

/// <summary>
///     This is an internal API that supports the Entity Framework Core infrastructure and not subject to
///     the same compatibility standards as public APIs. It may be changed or removed without notice in
///     any release. You should only use it directly in your code with extreme caution and knowing that
///     doing so can result in application failures when updating to a new Entity Framework Core release.
/// </summary>
public class SqliteObjectToStringTranslator : IMethodCallTranslator
{
    private static readonly HashSet<Type> TypeMapping =
    [
        typeof(bool),
        typeof(byte),
        typeof(byte[]),
        typeof(char),
        typeof(DateOnly),
        typeof(DateTime),
        typeof(DateTimeOffset),
        typeof(decimal),
        typeof(double),
        typeof(float),
        typeof(Guid),
        typeof(int),
        typeof(long),
        typeof(sbyte),
        typeof(short),
        typeof(TimeOnly),
        typeof(TimeSpan),
        typeof(uint),
        typeof(ushort)
    ];

    private readonly ISqlExpressionFactory _sqlExpressionFactory;

    /// <summary>
    ///     This is an internal API that supports the Entity Framework Core infrastructure and not subject to
    ///     the same compatibility standards as public APIs. It may be changed or removed without notice in
    ///     any release. You should only use it directly in your code with extreme caution and knowing that
    ///     doing so can result in application failures when updating to a new Entity Framework Core release.
    /// </summary>
    public SqliteObjectToStringTranslator(ISqlExpressionFactory sqlExpressionFactory)
        => _sqlExpressionFactory = sqlExpressionFactory;

    /// <summary>
    ///     This is an internal API that supports the Entity Framework Core infrastructure and not subject to
    ///     the same compatibility standards as public APIs. It may be changed or removed without notice in
    ///     any release. You should only use it directly in your code with extreme caution and knowing that
    ///     doing so can result in application failures when updating to a new Entity Framework Core release.
    /// </summary>
    public virtual SqlExpression? Translate(
        SqlExpression? instance,
        MethodInfo method,
        IReadOnlyList<SqlExpression> arguments,
        IDiagnosticsLogger<DbLoggerCategory.Query> logger)
    {
        if (instance == null || method.Name != nameof(ToString) || arguments.Count != 0)
        {
            return null;
        }

        if (instance.TypeMapping?.ClrType == typeof(string))
        {
            return instance;
        }

        if (instance.Type == typeof(bool))
        {
            if (instance is not ColumnExpression { IsNullable: false })
            {
                return _sqlExpressionFactory.Case(
                    instance,
                    [
                        new CaseWhenClause(
                            _sqlExpressionFactory.Constant(false),
                            _sqlExpressionFactory.Constant(false.ToString())),
                        new CaseWhenClause(
                            _sqlExpressionFactory.Constant(true),
                            _sqlExpressionFactory.Constant(true.ToString()))
                    ],
                    _sqlExpressionFactory.Constant(string.Empty));
            }

            return _sqlExpressionFactory.Case(
                [
                    new CaseWhenClause(
                        instance,
                        _sqlExpressionFactory.Constant(true.ToString()))
                ],
                _sqlExpressionFactory.Constant(false.ToString()));
        }

        // Enums are handled by EnumMethodTranslator

        return TypeMapping.Contains(instance.Type)
            ? _sqlExpressionFactory.Coalesce(
                _sqlExpressionFactory.Convert(instance, typeof(string)),
                _sqlExpressionFactory.Constant(string.Empty))
            : null;
    }
}
