/*
  DB Line In Utility
  ==================

  DBin reads data from a sound device, and forwards the data to DBMix.

  It is through the use of DBin that line in and microphone data can be
  added to the DBMix system.

  Version: 1
  Author:  Bob Dean
  Copyright (c) 1999


   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public Licensse as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

*/

#include <glib.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <limits.h>
#include <fcntl.h>
#include <signal.h>
#include <stdarg.h>
#include <dlfcn.h> /* functions used to dynamically link to libs */
#include <pthread.h>

#include <dbaudiolib.h>
#include <dbdebug.h>

#include "dbaudio.h"

#define DBIN_BUFSIZE    (PIPE_BUF * 10)
#define DBIN_WRITESIZE  (PIPE_BUF * 2)
#define DBIN_READSIZE   (PIPE_BUF)

extern int errno;
extern int debug_level;

int  loop_flag, the_other_loop_flag;
int  in_fd,out_fd;
int  numch,sample_size,sample_rate,chindex;
enum channel_type_e type;
char audio_source_path[256];
int  line_flag;
int  stdout_flag;

#ifdef USE_DYNAMIC_LIBRARY
DBAudioLibFunctions * dbaudio;
void * h;
#endif

pthread_t buffer_thread;
gint read_index, write_index;
char  * buffer;

oss_control_struct main_audio;


/***************************************************************************/
/*  Initialization functions                                               */
/***************************************************************************/

/*
  print_help - outputs help to the console. 

  inputs - none
  outputs - text to the console.
 */

void print_version()
{
	printf("DBIn - line in and microphone client for DBMix\n");
	printf("%s\n",DBMIX_COPYRIGHT);
	printf("Compiled for DBMix version %s\n",DBMIX_VERSION);	
}


void print_help()
{
	print_version();

	printf("\n");
	printf("Usage:\n");
	printf("  dbin [options]\n");

	printf("\n");
	printf("Supported options  (argument type in parens) [default value in brackets]:\n");
	printf("  -d    print debug messages           (none)    [off]\n");
	printf("  -l    audio input source in line in  (none)    [mic]\n");
	printf("  -a    set audio device path          (string)  [/dev/dsp]\n");
	printf("  -c X  use channel X if available     (int)     [first available]\n");
    printf("  -m    input data is mono.            (none)    [Stereo]\n");
    printf("  -r N  input sampling rate is N Hz    (int)     [44100 Hz]\n");
    printf("  -8    input data is 8 bit unsigned   (none)    [16 bit signed]\n");
	printf("  -s    output to stdout, not dbmix    (none)    [dbmix]\n");
	printf("  -h    print help screen              (none)    [mic]\n");
	printf("\n\n");
}


/*
  parse_cmdline - steps through argv and sets all the variables

  inputs - argc argv
  outputs - sets system variables
 */
void parse_cmdline(int argc, char* argv[])
{
	int opt;
   
	debug_level = 0;
	numch = 2;
	sample_size = 16;
	sample_rate = 44100;
	chindex = 0;
	line_flag = 0;
	stdout_flag = 0;
	
	strcpy(audio_source_path,"/dev/dsp0");

	while ((opt = getopt(argc,argv,"hsvda:lr:c:m8")) != -1)
	{
		switch (opt)
		{
			case 'd':
				debug_level = 1;
				break;
			case 'h':
				print_help();
				exit(0);
				break;
			case 's':
				stdout_flag = 1;
				break;
			case 'm':
				numch = 1;
				break;
			case 'l':
				line_flag = 1;
				break;
			case 'a':
				strcpy(audio_source_path,optarg);
				Debug("Set master audio device to be: \"%s\"",audio_source_path);
				break;
			case '8':
				sample_size = 8;
				Debug("set sample size to be AFMT_S8");
				break;
			case 'r':
				sample_rate = atoi(optarg);
				Debug("sample_rate is %d",sample_rate);
				break;
			case 'c':
				chindex = atoi(optarg);
				Debug("chindex is %d",chindex);
				break;
			case 'v':
				print_version();
				exit(0);
				break;
			case ':':
				Debug("option needs a value");
				exit(0);
			case '?':
				Debug("unknown option: %c",optopt);
				exit(0);
			default: break;
		}
	}
}


/*
  sigexit - changes system variables to force clean shutdown
    inputs - none
    outputs - none
 */
void sigexit(int signum)
{
	loop_flag = FALSE;
	the_other_loop_flag = FALSE;
	return;
}


/* function to sleep a thread for usec microseconds*/
void dbin_usleep(gint usec)
{
	struct timeval tv;
	
	tv.tv_sec = usec / 1000000;
	usec -= tv.tv_sec * 1000000;
	tv.tv_usec = usec;
	select(0, NULL, NULL, NULL, &tv);
}


/* buffer thread loop - takes data from buffer and writes it to dbmix */
void * write_audio_loop(void * args)
{
	gint used, count;
	char * buf;

	while (the_other_loop_flag)
	{
		buf = buffer + read_index;

		/* determine if there is data to be written */
		if (write_index >= read_index)
		{
			used = write_index - read_index;
		}
		
		used = DBIN_BUFSIZE - (read_index - write_index);

		if (used > 0)
		{
			/* write to stdout if desired */
			if (stdout_flag)
			{
				write(1,buf,DBIN_WRITESIZE);
			}
			else
			{
				/* write to dbmix */
#ifdef USE_DYNAMIC_LIBRARY
				if ((count = dbaudio->DBAudio_Write(buf,DBIN_WRITESIZE)) == FAILURE)
#else
				if ((count = DBAudio_Write(buf,DBIN_WRITESIZE)) == FAILURE)
#endif
				{
#ifdef USE_DYNAMIC_LIBRARY
					dbaudio->DBAudio_perror("DBIN: DBAudio_Write failed::");
#else
					DBAudio_perror("DBIN: DBAudio_Write failed::");
#endif
				}
			}
			
			/* update buffer control variables */
			read_index += DBIN_WRITESIZE;
			read_index = read_index % DBIN_BUFSIZE;
		}
		else
		{
			printf("writer sleeping");
			dbin_usleep(10000);
		}
	}

	pthread_exit(NULL);
}



/***************************************************************************/
/*                MAIN                                                     */
/***************************************************************************/

int main(int argc, char **argv)
{
	int  error,count;
	int  mixer_fd, value;
	oss_control_struct main_audio;

#ifdef USE_DYNAMIC_LIBRARY

	void * (*get_fxns) (); /* define function pointer to hold pntr to DBAudio_Get_Functions() */
	
	dbaudio = NULL;
#endif

	type = PIPE_CHANNEL;
	buffer = NULL;

	/*install signal handlers*/
	signal(SIGINT,sigexit);
	signal(SIGTERM,sigexit);

	parse_cmdline(argc,argv);

	count = DBIN_BUFSIZE;
	buffer = (char * ) malloc(count);

	/* open audio device */
	{
		int flags;
		int frag;

		flags = O_RDONLY;
		

		main_audio.sample_format = sample_size;
		main_audio.channels = numch;
		main_audio.sample_rate = sample_rate; 

		if((main_audio.fd_dsp = open(audio_source_path,flags,0)) == -1)
		{
			char str[256];
			sprintf(str,"init_audio: cannot open device \"%s\"",audio_source_path);
			perror(str);
			exit(0);
		}

		Debug("opened audio device %s",audio_source_path);

		frag = (10 << 16) | (DB_FRAGSIZE);
		
		if(ioctl(main_audio.fd_dsp,SNDCTL_DSP_SETFRAGMENT,&frag) == -1)
		{
			perror("init_audio: could not set fragment size.");
			goto cleanup;
		}
		else
		{
			Debug("init_audio: num fragments %d  fragement size: %d",DB_NUM_FRAGMENTS,2^DB_FRAGSIZE);
		}
		
		if(ioctl(main_audio.fd_dsp,SNDCTL_DSP_SETFMT,&(main_audio.sample_format)) == -1)
		{
			perror("init_audio: failure setting format.");
			/*perror("SNDCTL_DSP_SETFMT");*/
			goto cleanup;
		}
		else
		{
			Debug("set sound format to be %d",main_audio.sample_format);
		}
		
		main_audio.channels = DBSTEREO;	
		
		if(ioctl(main_audio.fd_dsp,SNDCTL_DSP_STEREO,&(main_audio.channels)) == -1)
		{
			perror("init_audio: Error setting stereo");
			/*perror("SNDCTL_DSP_STEREO");*/
			goto cleanup;
		}
		else
		{
			Debug("set stereo mode");
		}
		
		/*set sample rate*/
		if(ioctl(main_audio.fd_dsp,SNDCTL_DSP_SPEED,&(main_audio.sample_rate)) == -1)
		{
			perror("init_audio: Could not set sample rate.");
			/*perror("SNDCTL_DSP_SPEED");*/
			goto cleanup;
		}
		else
		{
			Debug("sample rate set to be %d",main_audio.sample_rate);
		}
		
		{
			int frag_size;
			audio_buf_info info;
			
			Debug("****************");
			Debug("device: %s",audio_source_path);
			Debug("getting fragment data...");
			if (ioctl(main_audio.fd_dsp, SNDCTL_DSP_GETBLKSIZE, &frag_size) == -1) perror("getting frag size");
			
			Debug("Frag size:  %d",frag_size);
			
			ioctl(main_audio.fd_dsp, SNDCTL_DSP_GETOSPACE, &info);
			
			Debug("fragments:  %d",info.fragments);
			Debug("fragstotal: %d",info.fragstotal);
			Debug("fragsize:   %d",info.fragsize);
			Debug("bytes:      %d",info.bytes);
			Debug("****************");
		}

	}


	/* init recording source */
	{
		mixer_fd = open("/dev/mixer0",O_RDONLY);
		
		if (mixer_fd != -1)
		{
			if(line_flag)
			{
				Debug("Setting source to be line");
				value =  (1 << SOUND_MIXER_LINE);
			}
			else
			{
				Debug("Setting source to be mic");
				value = (1 << SOUND_MIXER_MIC);
			}
			
			if (ioctl(mixer_fd,SOUND_MIXER_WRITE_RECSRC, &value) != 0)
			{
				Error("Could not set record source.");
				goto cleanup;
			}
			else
			{
				Debug("Set record source.");
			}
			
			close(mixer_fd);
		}
		else
		{
			perror("DBin: failed to set record source.");
		}
	}


	if (!stdout_flag)
	{
#ifdef USE_DYNAMIC_LIBRARY
		/* link against DBAudioLib and retrieve fucntion structure */
		{

			if ((h = dlopen("libdbaudiolib.so",RTLD_NOW)) == NULL) 
			{
				Debug("Failed to open DBAudioLib:: %s",dlerror());
				goto cleanup;
			}
			
			if ((get_fxns = dlsym(h,"DBAudio_Get_Functions")) == NULL)
			{
				Debug("Failed to retrieve pntr to DBAudio_Get_Functions: %s",dlerror());
				goto cleanup;
			}
			
			dbaudio = (DBAudioLibFunctions*)get_fxns();
		}
#endif		
		
		/* init dbaudiolib */
		{
#ifdef USE_DYNAMIC_LIBRARY
			error = dbaudio->DBAudio_Init("DBin",sample_size,sample_rate,numch,type,chindex);
			
			dbaudio->DBAudio_Clear_Channel_Flag(PITCH_FLAG);
			dbaudio->DBAudio_Clear_Channel_Flag(PAUSE_FLAG);	
#else
			error = DBAudio_Init("DBin",sample_size,sample_rate,numch,type,chindex);
			
			DBAudio_Clear_Channel_Flag(PITCH_FLAG);
			DBAudio_Clear_Channel_Flag(PAUSE_FLAG);			
#endif			

			if (!line_flag)
			{
#ifdef USE_DYNAMIC_LIBRARY
				dbaudio->DBAudio_Set_Channel_Flag(MIC_FLAG);
#else
				DBAudio_Set_Channel_Flag(MIC_FLAG);
#endif
			}
			
			if (error != SUCCESS)
			{
				Debug("Error. Is dbfsd running?");
				goto cleanup;
			}
			else
			{
				Debug("Opened dbmix channel.");
			}
		}
	}  

	/*init variables for main loop*/
	loop_flag = TRUE;
	the_other_loop_flag = TRUE;
	count = 0;

	pthread_create(&buffer_thread,NULL,write_audio_loop,NULL);

	/* read data from soundcard */
	{
		gint len, free;
		char * buf;
		
		while (loop_flag)
		{			
			buf = buffer + write_index;
			
			/* determine if there is enough room in the buffer for new data */
			if (read_index > write_index)
			{
				free = read_index - write_index;
			}
			else
			{
				free = DBIN_BUFSIZE - write_index;
			}
			
			if (read_index == write_index) 
			{
				free = 0;
			}
		   
			/* if there is free space */
			if (free >= DBIN_READSIZE)
			{
				/* read data into buffer */
				if ((len = read(main_audio.fd_dsp,buf,DBIN_READSIZE)) > 0)
				{
					write_index += DBIN_READSIZE;
					write_index = write_index % DBIN_BUFSIZE;
				}
				else 
				{
					if(errno != EIO)
					{
						loop_flag = FALSE;
						perror("DBin main(): unexpected error:");
					}
				}
			}
			else
			{
				/* sleep if no space available */
				Debug("reader sleeping");
				dbin_usleep(5000);
			}
		}
	}

 cleanup:

/* 	pthread_join(buffer_thread, NULL); */

   	close(main_audio.fd_dsp);

	if (buffer != NULL)
	{
		free(buffer);
	}

#ifdef USE_DYNAMIC_LIBRARY
	if (dbaudio != NULL) 
	{
		dbaudio->DBAudio_Close();
	}

	dlclose(h);	
#else
	if (DBAudio_Ready())
	{
		DBAudio_Close();
	}
#endif

	Debug("DBin has completed. Have a nice day!\n");

	exit(0);
}

