/* $Id: kmo_make_image-test.c,v 1.7 2013-10-08 11:18:56 aagudo Exp $
 *
 * This file is part of the KMOS Library
 * Copyright (C) 2002-2006 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301 USA
 */

/*
 * $Author: aagudo $
 * $Date: 2013-10-08 11:18:56 $
 * $Revision: 1.7 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <sys/stat.h>

#include <cpl.h>

#include "kmo_dfs.h"

const char  *test_global_path_test_data;
float   test_global_seed_data, test_global_seed_noise;
int     test_global_size_x, test_global_size_y, test_global_size_z, test_global_nr_frames;
void kmo_get_pipe_command(char*, const char*, const char*, int);
int kmo_test_file_exists(const char*);
void kmo_test_verbose_off();
void kmo_test_verbose_on();
double kmo_test_esorex_data(const char*, int);
const char* kmo_test_cat_strings(char*, const char*, const char*);
int  kmo_test_create_RAW_data(int, int, const char*, char**, char**, int*, int, char**, char**, int*, int, float, float);
int  kmo_test_create_RAW_data_zero(int, int, const char*);
int  kmo_test_create_F2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2D_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F2I_data_noise(int, int, int, const char*, char *, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_infinite(int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data2(int, int, int, int, int*, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F3I_data_noise(int, int, int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_F1I_data_noise(int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);
int  kmo_test_create_B2D_data(int, int, int, const char*, char*, char*, int*, int, char**, char**, int*, int);

const char      *path_recipe        = "kmo_make_image/";

const char      *valid_files[]      = {"v_data_F3I_all.fits",
                                       "v_data_noise_F3I_all.fits",
                                       "v_data_F3I_all_w_oh_all.fits",
                                       "v_data_F3I_all_w_oh_2.fits",
                                       "v_data_noise_F3I_all_w_oh_small.fits",
                                       "ohspec.fits",
                                       "oh_spec_f1s.fits"};

const char      *invalid_files[]    = {"i_data_F2D.fits"};

/**
    @defgroup kmo_make_image_test   kmo_make_image unit tests

    @{
*/

/**
 * @brief   test with two frames
 */
static void test_make_image(const char *path_op1,
                            const char *path_op2,
                            const char *method,
                            int ret_val)
{
    // create sof-file
    const char *sof_path = "test_make_image.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s\n", path_op1);
    if (path_op2 != NULL) {
        fprintf (fh, "%s\n", path_op2);
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_make_image --cmethod=%s %s",
            method, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_make_image.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

static void test_make_image_ranges(const char *path_op1,
                                   const char *path_op2,
                                   const char *method,
                                   int ret_val)
{
    // create sof-file
    const char *sof_path = "test_make_image_ranges.sof";
    FILE *fh = fopen(sof_path, "w");
    fprintf (fh, "%s\n", path_op1);
    if (path_op2 != NULL) {
        fprintf (fh, "%s\n", path_op2);
    }
    fclose(fh);

    // create esorex-command
    char esorex_command[256];
    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_make_image "
            "--range=\"0.9,0.95;1.0,1.5\" --cmethod=%s %s",
            method, sof_path);

    kmo_get_pipe_command(esorex_command,
                         "log_kmo_make_image.txt", sof_path, TRUE);

    // execute esorex-command
    if (ret_val == 0) {
        cpl_test_eq(0, system(esorex_command));
    } else {
        cpl_test_noneq(0, system(esorex_command));
    }
}

/**
    @brief
        Generates test data for kmo_make_image.

    @param path Path where the generated test date should be saved to.
*/
static int kmo_generate_test_data_make_image(const char *path)
{
    char        file_path[256];
    char        *sub_keys[3];
    char        *sub_vals[3];
    int         sub_types[3];

    /* ----- valid test data ----- */
    // F3I
    cpl_test_eq(0,
            kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                         test_global_size_z,  FALSE, test_global_nr_frames,
                         kmo_test_cat_strings(file_path, path, valid_files[0]),
                         NULL, NULL, NULL, 0,
                         NULL, NULL, NULL, 0));

    cpl_test_eq(0,
            kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y,
                        test_global_size_z,  FALSE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[1]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    // F3I with CRPIX3, CDELT3, CRVAL3 in a way that all 7 slices are valid
    // and whole IFU-spectrum lies in oh-spectrum
    sub_keys[0] = "CRPIX3";
    sub_vals[0] = "1";
    sub_types[0] = CPL_TYPE_DOUBLE;

    sub_keys[1] = "CRVAL3";
    sub_vals[1] = "1.2";        // all 7 slices valid
    sub_types[1] = CPL_TYPE_DOUBLE;

    sub_keys[2] = "CDELT3";
    sub_vals[2] = "1.0000E-04";
    sub_types[2] = CPL_TYPE_DOUBLE;

    cpl_test_eq(0,
            kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                        test_global_size_z,  FALSE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[2]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, 3));

    // F3I with CRPIX3, CDELT3, CRVAL3 in a way that only 2 slices are valid
    // and whole IFU-spectrum lies in oh-spectrum
    sub_keys[0] = "CRPIX3";
    sub_vals[0] = "1";
    sub_types[0] = CPL_TYPE_DOUBLE;

    sub_keys[1] = "CRVAL3";
    sub_vals[1] = "1.4182";   // only first 2 slices valid
    sub_types[1] = CPL_TYPE_DOUBLE;

    sub_keys[2] = "CDELT3";
    sub_vals[2] = "1.0000E-04";
    sub_types[2] = CPL_TYPE_DOUBLE;

    cpl_test_eq(0,
            kmo_test_create_F3I_data(test_global_size_x, test_global_size_y,
                        test_global_size_z,  FALSE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[3]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, 3));

    // F3I with CRPIX3, CDELT3, CRVAL3 in a way that not the whole IFU-spectrum
    // lies in oh-spectrum AND noise
    sub_keys[0] = "CRPIX3";
    sub_vals[0] = "1";
    sub_types[0] = CPL_TYPE_DOUBLE;

    sub_keys[1] = "CRVAL3";
    sub_vals[1] = "0.7";
    sub_types[1] = CPL_TYPE_DOUBLE;

    sub_keys[2] = "CDELT3";
    sub_vals[2] = "0.4";
    sub_types[2] = CPL_TYPE_DOUBLE;

    cpl_test_eq(0,
            kmo_test_create_F3I_data_noise(test_global_size_x, test_global_size_y,
                        test_global_size_z,  FALSE, test_global_nr_frames,
                        kmo_test_cat_strings(file_path, path, valid_files[4]),
                        NULL, NULL, NULL, 0,
                        sub_keys, sub_vals, sub_types, 3));

    /* ----- invalid test data ----- */
    // F2D
    cpl_test_eq(0,
            kmo_test_create_F2D_data(test_global_size_x, test_global_size_y, FALSE,
                        kmo_test_cat_strings(file_path, path, invalid_files[0]),
                        NULL, NULL, NULL, 0,
                        NULL, NULL, NULL, 0));

    return 0;
}


/**
  @brief    Test of kmo_make_image recipe.
  @param    argc   the number of parameters
  @param    argv   the parameter list

  Call @c kmo_make_image-test @c --generate when only the test data for this
  recipe should be generated.
  Call @c kmo_make_image-test @c --no-gen when only the tests should be
  executed and no data should be generated.

 */
int main(int argc, char *argv[])
{
    float   tol     = 0.01;

    char    test_path[256],
            op1_file[256],
            op2_file[256],
            esorex_command[256];

    const char *sof_path = "oh.sof";

    cpl_test_init("<kmos-spark@mpe.mpg.de>", CPL_MSG_WARNING);

    strcpy(test_path, test_global_path_test_data);
    strcat(test_path, path_recipe);

    strcpy(op2_file, valid_files[6]);

    if (kmo_test_file_exists(test_path) == FALSE) {
        mkdir(test_path, 0777);
    }

    if ((argc == 1) || ((argc > 1) && (strcmp(argv[1], "--no-gen") != 0))) {
        cpl_test_zero(kmo_generate_test_data_make_image(test_path));
    }

    if ((argc > 1) && (strcmp(argv[1], "--generate") == 0)) {
        // just generate the data
        return cpl_test_end(0);
    }

    /* ----- run the tests ----- */
    FILE *fd = fopen("log_kmo_make_image.txt", "w");
    fprintf(fd, "%s\n", " ");
    fclose(fd);

    FILE *fh = fopen(sof_path, "w");
    char *test_global_path_ref_data = cpl_sprintf("%s/ref_data/", getenv("srcdir"));
    fprintf (fh, "%s%s     STACK_DATA\n", test_global_path_ref_data, valid_files[5]);
    fclose(fh);
    cpl_free(test_global_path_ref_data);

    sprintf(esorex_command,
            "esorex --suppress-prefix=TRUE kmo_fits_stack -type=F1S "
            "-filename=oh_spec_f1s %s", sof_path);
    kmo_get_pipe_command(esorex_command,
                         "log_kmo_make_image.txt", sof_path, TRUE);
    cpl_test_eq(0, system(esorex_command));

    /* ----- valid tests ----- */
    // F3I, all slices
    kmo_test_cat_strings(op1_file, test_path, valid_files[0]);
    test_make_image(op1_file, NULL, "median", 0);
    cpl_test_abs(kmo_test_esorex_data(MAKE_IMAGE, 0), 77.85, tol);

    // F3I with noise, all slices
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    test_make_image(op1_file, NULL, "median", 0);
    cpl_test_abs(kmo_test_esorex_data(MAKE_IMAGE, 0), 80.2995, tol);

    // F3I with noise, all slices
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    test_make_image(op1_file, NULL, "ksigma", 0);
    cpl_test_abs(kmo_test_esorex_data(MAKE_IMAGE, 0), 80.2995, tol);

    // F3I with noise, all slices
    kmo_test_cat_strings(op1_file, test_path, valid_files[1]);
    test_make_image(op1_file, NULL, "min_max", 0);
    cpl_test_abs(kmo_test_esorex_data(MAKE_IMAGE, 0), 79.9713, tol);

    // F3I without noise, with ohspec (all slices)
    kmo_test_cat_strings(op1_file, test_path, valid_files[2]);
    test_make_image(op1_file, op2_file, "ksigma", 0);
    cpl_test_abs(kmo_test_esorex_data(MAKE_IMAGE, 0), 77.85, tol);

    // F3I without noise and ranges, with ohspec (all slices)
    kmo_test_cat_strings(op1_file, test_path, valid_files[2]);
    test_make_image_ranges(op1_file, op2_file, "ksigma", 0);
    cpl_test_abs(kmo_test_esorex_data(MAKE_IMAGE, 0), 77.85, tol);

    // F3I without noise, with ohspec (2 slices)
    kmo_test_cat_strings(op1_file, test_path, valid_files[3]);
    test_make_image(op1_file, op2_file, "median", 0);
    cpl_test_abs(kmo_test_esorex_data(MAKE_IMAGE, 0), 70.35, tol);

    // F3I with noise, with IFU-spec larger than ohspec (all slices)
    kmo_test_cat_strings(op1_file, test_path, valid_files[4]);
    test_make_image(op1_file, op2_file, "ksigma", 0);
    cpl_test_abs(kmo_test_esorex_data(MAKE_IMAGE, 0), 78.2865, tol);

    /* ----- invalid tests ----- */
    // NULL input
    test_make_image(NULL, NULL, "median", -1);

    // F2D
    kmo_test_cat_strings(op1_file, test_path, invalid_files[0]);
    test_make_image(op1_file, NULL, "median", -1);

    return cpl_test_end(0);
}

/** @} */
