//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Beam/FootprintItems.h
//! @brief     Declares FootprintItem classes
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2018
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_BEAM_FOOTPRINTITEMS_H
#define BORNAGAIN_GUI_MODEL_BEAM_FOOTPRINTITEMS_H

#include "GUI/Model/Descriptor/DoubleProperty.h"
#include <memory>

class IFootprint;

class FootprintItem {
public:
    virtual ~FootprintItem() = default;

    virtual void writeTo(QXmlStreamWriter*) const {}
    virtual void readFrom(QXmlStreamReader*) {}

    virtual std::unique_ptr<IFootprint> createFootprint() const = 0;
};

class FootprintNoneItem : public FootprintItem {
public:
    std::unique_ptr<IFootprint> createFootprint() const override;
};

class FootprintGaussianItem : public FootprintItem {
public:
    explicit FootprintGaussianItem(double value = 0.0);

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    std::unique_ptr<IFootprint> createFootprint() const override;

    DoubleProperty& gaussianFootprintValue() { return m_gaussianFootprintValue; }
    const DoubleProperty& gaussianFootprintValue() const { return m_gaussianFootprintValue; }
    void setGaussianFootprintValue(double v) { m_gaussianFootprintValue.setValue(v); }

protected:
    DoubleProperty m_gaussianFootprintValue;
};

class FootprintSquareItem : public FootprintItem {
public:
    explicit FootprintSquareItem(double value = 0.0);

    void writeTo(QXmlStreamWriter* w) const override;
    void readFrom(QXmlStreamReader* r) override;

    std::unique_ptr<IFootprint> createFootprint() const override;

    DoubleProperty& squareFootprintValue() { return m_squareFootprintValue; }
    const DoubleProperty& squareFootprintValue() const { return m_squareFootprintValue; }
    void setSquareFootprintValue(double v) { m_squareFootprintValue.setValue(v); }

protected:
    DoubleProperty m_squareFootprintValue;
};

#endif // BORNAGAIN_GUI_MODEL_BEAM_FOOTPRINTITEMS_H
