/*
 * Copyright (c) 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/**
 * This application is meant to be run to create a classlist file representing
 * common use.
 *
 * The classlist is produced by adding -XX:DumpLoadedClassList=classlist
 */
package build.tools.classlist;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.util.Scanner;

/**
 * The classlist generated by build.tools.classlist.HelloClasslist
 * may have non-deterministic contents, affected by Java thread execution order.
 * SortClasslist sorts the file to make the JDK image's contents more deterministic.
 */
public class SortClasslist {
    public static void main(String args[]) throws FileNotFoundException  {
        ArrayList<String> classes = new ArrayList<>();
        ArrayList<String> lambdas = new ArrayList<>();

        FileInputStream fis = new FileInputStream(args[0]);
        Scanner scanner = new Scanner(fis);
        Pattern p = Pattern.compile("^(.*)[ ]+id:[ ]+([0-9]+)$");
        while (scanner.hasNextLine()) {
            String line = scanner.nextLine();
            Matcher m = p.matcher(line);
            if (line.startsWith("#")) {
                // Comments -- print them first without sorting. These appear only at the top
                // of the file.
                System.out.println(line);
            } else if (line.startsWith("@")) {
                // @lambda-form-invoker, @lambda-proxy, etc.
                lambdas.add(line);
            } else if (m.find()) {
                // We found a pattern like this:
                //
                //    <beginning of line>java/lang/Object id: 0<end of line>
                //
                // This is a class used by one of the three builtin class loaders
                // (boot/platform/app). Since the default classlist does not support unregistered
                // classes, the ID is unused. Let's omit the ID, as it may be non-deterministic.
                String className = m.group(1); // matches the (.*) part of the pattern.
                classes.add(className);
            } else {
                // HelloClasslist should not load classes in custom class loaders, or else
                // we might end up with output like this:
                //
                //     SomeClass id: 123 super: 0 source: foo.jar
                //
                // Such classes won't be usable for common applications, so they should
                // not be included in the JDK's default classlist.
                System.err.println("Unexpected line: " + line);
                System.err.println("The default classlist should not contain unregistered classes");
                System.exit(1);
            }
        }

        Collections.sort(classes);
        Collections.sort(lambdas);

        for (String s : classes) {
            System.out.println(s);
        }
        for (String s : lambdas) {
            System.out.println(s);
        }
    }
}
