#! /usr/bin/env python

# Genetic
# Copyright (C) 2001 Jean-Baptiste LAMY
#
# This program is free software. See README or LICENSE for the license terms.

import random, string, math, Tkinter, thread
from genetic import organism, lifecycle

SIZE = 250
OBSTACLE_HEIGHT = 100 # You may try some harder challanger, e.g. 150. Or no obstacle, e.g. 0.

class JumperOrganism(organism.Organism):
  characteristics = [
    organism.Characteristic("ax", lambda ax: ax, organism.PERCHROMOSOM_DOMINANCY_PHENOTYPE),
    organism.Characteristic("bx", lambda bx: bx, organism.PERCHROMOSOM_DOMINANCY_PHENOTYPE),
    organism.Characteristic("cx", lambda cx: cx, organism.PERCHROMOSOM_DOMINANCY_PHENOTYPE),
    organism.Characteristic("ay", lambda ay: ay, organism.PERCHROMOSOM_DOMINANCY_PHENOTYPE),
    organism.Characteristic("by", lambda by: by, organism.PERCHROMOSOM_DOMINANCY_PHENOTYPE),
    organism.Characteristic("cy", lambda cy: cy, organism.PERCHROMOSOM_DOMINANCY_PHENOTYPE),
    ]
  
  def __init__(self, genotype):
    organism.Organism.__init__(self, genotype)
    self.reset()
    
  def __cmp__(self, other):
    # Say how we compare 2 organisms. Here, the lower the circle radius is the better the organism is.
    return cmp(self.jumptime, other.jumptime)

  def play(self, time):
    dx = self.ax * self.x + self.bx * self.y + self.cx
    dy = self.ay * self.x + self.by * self.y + self.cy
    
    length = math.sqrt(dx ** 2 + dy ** 2) / 2.0
    #print length, dx, dy
    #print length, dx / length, dy / length
    
    newx = self.x + dx / length
    newy = self.y + dy / length
    
    #if newx > self.x + 2.0: newx = self.x + 2.0
    #if newx < self.x - 2.0: newx = self.x - 2.0
    #if newy > self.y + 2.0: newy = self.y + 2.0
    #if newy < self.y - 2.0: newy = self.y - 2.0
    
    if canpass(newx, newy): self.x, self.y = newx, newy
    else:
      if   canpass(newx, self.y): self.x = newx
      elif canpass(self.x, newy): self.y = newy
      
    if arrived(self.x, self.y):
      self.arrived  = 1
      self.jumptime = time
      
  def reset(self):
    self.x = 0.0
    self.y = float(SIZE)
    self.jumptime = None
    self.arrived = 0
      

def randomjumper():
  return JumperOrganism([
    organism.Chromosom(ax = rand(), bx = rand(), cx = rand(), __mutampl__ = 0.1, __mutation__ = 0.8, __break__ = 0.0),
    organism.Chromosom(ay = rand(), by = rand(), cy = rand(), __mutampl__ = 0.1, __mutation__ = 0.8, __break__ = 0.0),
    ])

def rand(): return (random.random() - 0.5) / 30.0

def canpass(x, y):
  return 0 < x < SIZE and 0 < y < SIZE and not (100 < x < 150 and y > (SIZE - OBSTACLE_HEIGHT))

def arrived(x, y):
  return x > 220 and y > 150


class Graph(Tkinter.Canvas):
  def __init__(self, master):
    Tkinter.Canvas.__init__(self, master, bg = "white", width = SIZE, height = SIZE)
    self.running = 0
    
    self.obstacle = self.create_rectangle(100, SIZE - OBSTACLE_HEIGHT, 150, 250);
    
    self.organisms = [randomjumper() for i in range(20)]
    
  def generation(self, nb = 1):
    self.running = 1
    thread.start_new_thread(self.run, ())
    
  def run(self):
    tags = []
    
    for organism in self.organisms:
      organism.tag = self.create_text(organism.x, organism.y, text = "O")
      tags.append(organism.tag)
        
    for time in range(500): # Let give them 500 moves !
      if not self.running: break # Stop !
      nooneleft = 1
      
      for organism in self.organisms:
        if not organism.arrived:
          oldx, oldy = organism.x, organism.y
          organism.play(time)
          self.move(organism.tag, organism.x - oldx, organism.y - oldy)
          nooneleft = 0
          
      if nooneleft: break
      
    
    # Remove the looser (jumptime is None)
    self.organisms = filter(lambda organism: not organism.jumptime is None, self.organisms)
    print "%s organisms arrived at destination !" % len(self.organisms)
    
    # Sort them
    self.organisms.sort()
    
    # Take the 10 first
    self.organisms = self.organisms[:10]
    
    # Add random organisms if less than 3
    while len(self.organisms) < 3: self.organisms.append(randomjumper())
    
    # Elitism (keep the best one). We need to reset this champion to the start line !
    self.organisms[0].reset()

    # Keep the best one, and 20 new organisms.
    self.organisms = [self.organisms[0]] + lifecycle.make_love(self.organisms, 20)
    
    self.delete(*tags)
    
    self.running = 0
    

class GraphFrame(Tkinter.Tk):
  def __init__(self):
    Tkinter.Tk.__init__(self, className = "genetic jump !")
    Tkinter.Label(self, text = "Organisms starts on the lower left corner; they must arrive at the lower right corner. The rectangle is an obstacle they must jump !\nRun some generations, and organisms will learn how to jump !", wraplength = SIZE + 30).pack()
    self.graph = Graph(self)
    self.graph.pack()
    Tkinter.Button(self, text = "1 generation" , command = self.generation1  ).pack()
        
  def generation1(self, event = None):
    if not self.graph.running:
      self.graph.generation(1)
      self.graph.update()
    else: self.graph.running = 0
    

graphframe = GraphFrame()
Tkinter.mainloop()
