/*
 * Copyright (c) 2010-2022 Belledonne Communications SARL.
 *
 * This file is part of Liblinphone
 * (see https://gitlab.linphone.org/BC/public/liblinphone).
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef LINPHONE_CONFERENCE_H
#define LINPHONE_CONFERENCE_H

#include "linphone/api/c-conference-info.h"
#include "linphone/types.h"
#include <mediastreamer2/mediastream.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef enum { LinphoneConferenceClassLocal, LinphoneConferenceClassRemote } LinphoneConferenceClass;

/**
 * @addtogroup call_control
 * @{
 */

/**
 * Create an object of type a #LinphoneConferenceParams.
 * @param core The #LinphoneCore to pass to the constructor. @notnil
 * @return The created #LinphoneConferenceParams. @notnil
 */
LINPHONE_PUBLIC LinphoneConferenceParams *linphone_conference_params_new(LinphoneCore *core);

/**
 * Take a reference on a #LinphoneConferenceParams.
 * @param params The #LinphoneConferenceParams to ref. @notnil
 * @return The freshly refed #LinphoneConferenceParams. @notnil
 */
LINPHONE_PUBLIC LinphoneConferenceParams *linphone_conference_params_ref(LinphoneConferenceParams *params);

/**
 * Release a #LinphoneConferenceParams.
 * @param params The #LinphoneConferenceParams to release. @notnil
 */
LINPHONE_PUBLIC void linphone_conference_params_unref(LinphoneConferenceParams *params);

/**
 * Free a #LinphoneConferenceParams
 * @param params #LinphoneConferenceParams to free @notnil
 * @deprecated 17/03/2017 Use linphone_conference_params_unref() instead.
 * @donotwrap
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED void linphone_conference_params_free(LinphoneConferenceParams *params);

/**
 * Clone a #LinphoneConferenceParams
 * @param params The #LinphoneConferenceParams to clone @notnil
 * @return An allocated #LinphoneConferenceParams with the same parameters than params @notnil
 */
LINPHONE_PUBLIC LinphoneConferenceParams *linphone_conference_params_clone(const LinphoneConferenceParams *params);

/**
 * Set the conference subject
 * @param params A #LinphoneConferenceParams @notnil
 * @param subject conference subject @maybenil
 */
LINPHONE_PUBLIC void linphone_conference_params_set_subject(LinphoneConferenceParams *params, const char *subject);

/**
 * Get the conference subject
 * @param params A #LinphoneConferenceParams @notnil
 * @return conference subject. @maybenil
 */
LINPHONE_PUBLIC const char *linphone_conference_params_get_subject(const LinphoneConferenceParams *params);

/**
 * Enable audio capabilities
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable If TRUE, audio will be enabled during conference
 * @deprecated 16/12/2021 Use linphone_conference_params_enable_audio() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED void linphone_conference_params_set_audio_enabled(LinphoneConferenceParams *params,
                                                                                      bool_t enable);

/**
 * Enable audio capabilities
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable If TRUE, audio will be enabled during conference
 */
LINPHONE_PUBLIC void linphone_conference_params_enable_audio(LinphoneConferenceParams *params, bool_t enable);

/**
 * Check whether audio capabilities are enabled
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if the conference supports audio capabilities, FALSE otherwise
 * @deprecated 16/12/2021 Use linphone_conference_params_audio_enabled() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED bool_t
linphone_conference_params_is_audio_enabled(const LinphoneConferenceParams *params);

/**
 * Check whether audio capabilities are enabled
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if the conference supports audio capabilities, FALSE otherwise
 */
LINPHONE_PUBLIC bool_t linphone_conference_params_audio_enabled(const LinphoneConferenceParams *params);

/**
 * Enable video capabilities
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable If TRUE, video will be enabled during conference
 * @deprecated 16/12/2021 Use linphone_conference_params_enable_video() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED void linphone_conference_params_set_video_enabled(LinphoneConferenceParams *params,
                                                                                      bool_t enable);

/**
 * Check whether video capabilities are enabled
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if the conference supports video capabilities, FALSE otherwise
 * @deprecated 16/12/2021 Use linphone_conference_params_video_enabled() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED bool_t
linphone_conference_params_is_video_enabled(const LinphoneConferenceParams *params);

/**
 * Enable video capabilities
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable If TRUE, video will be enabled during conference
 */
LINPHONE_PUBLIC void linphone_conference_params_enable_video(LinphoneConferenceParams *params, bool_t enable);

/**
 * Check whether video capabilities are enabled
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if the conference supports video capabilities, FALSE otherwise
 */
LINPHONE_PUBLIC bool_t linphone_conference_params_video_enabled(const LinphoneConferenceParams *params);

/**
 * Enable chat capabilities
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable If TRUE, chat is enabled during conference
 * @deprecated 16/12/2021 Use linphone_conference_params_enable_chat() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED void linphone_conference_params_set_chat_enabled(LinphoneConferenceParams *params,
                                                                                     bool_t enable);

/**
 * Enable chat capabilities
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable If TRUE, chat is enabled during conference
 */
LINPHONE_PUBLIC void linphone_conference_params_enable_chat(LinphoneConferenceParams *params, bool_t enable);

/**
 * Check whether chat capabilities are enabled
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if the conference supports chat capabilities, FALSE otherwise
 * @deprecated 16/12/2021 Use linphone_conference_params_chat_enabled() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED bool_t
linphone_conference_params_is_chat_enabled(const LinphoneConferenceParams *params);

/**
 * Check whether chat capabilities are enabled
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if the conference supports chat capabilities, FALSE otherwise
 */
LINPHONE_PUBLIC bool_t linphone_conference_params_chat_enabled(const LinphoneConferenceParams *params);

/**
 * Sets the account for the conference
 * @param params A #LinphoneConferenceParams @notnil
 * @param account a pointer to the account. @maybenil
 * @warning The account can only be changed upon creation of a conference when calling
 * linphone_core_create_conference_with_params
 */
LINPHONE_PUBLIC void linphone_conference_params_set_account(LinphoneConferenceParams *params, LinphoneAccount *account);

/**
 * Returns the account for the conference
 * @param params A #LinphoneConferenceParams @notnil
 * @return a pointer to the account or NULL if it is not set. @maybenil
 */
LINPHONE_PUBLIC LinphoneAccount *linphone_conference_params_get_account(const LinphoneConferenceParams *params);

/**
 * Returns the proxy configuration for the conference
 * @param params A #LinphoneConferenceParams @notnil
 * @return a pointer to the proxy configuration or NULL if it is not set. @maybenil
 * @deprecated 11/01/2022 Use linphone_conference_params_get_account() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED LinphoneProxyConfig *
linphone_conference_params_get_proxy_cfg(const LinphoneConferenceParams *params);

/**
 * Enable local participant to enter the conference.
 * The local participant is the one driving the local #LinphoneCore. It uses the local sound devices.
 * The default value is TRUE. Setting to FALSE is mostly helpful when using liblinphone on a server application.
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable TRUE if local participant is automatically added to the conference, FALSE otherwise
 * @deprecated 16/12/2021 Use linphone_conference_params_enable_local_participant() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED void
linphone_conference_params_set_local_participant_enabled(LinphoneConferenceParams *params, bool_t enable);

/**
 * Enable local participant to enter the conference.
 * The local participant is the one driving the local #LinphoneCore. It uses the local sound devices.
 * The default value is TRUE. Setting to FALSE is mostly helpful when using liblinphone on a server application.
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable TRUE if local participant is automatically added to the conference, FALSE otherwise
 */
LINPHONE_PUBLIC void linphone_conference_params_enable_local_participant(LinphoneConferenceParams *params,
                                                                         bool_t enable);

/**
 * Returns whether local participant has to enter the conference.
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if local participant is by default part of the conference, FALSE otherwise
 * @deprecated 16/12/2021 Use linphone_conference_params_local_participant_enabled() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED bool_t
linphone_conference_params_is_local_participant_enabled(const LinphoneConferenceParams *params);

/**
 * Enable local participant to enter the conference.
 * The local participant is the one driving the local #LinphoneCore. It uses the local sound devices.
 * The default value is TRUE. Setting to FALSE is mostly helpful when using liblinphone on a server application.
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable TRUE if local participant is automatically added to the conference, FALSE otherwise
 */
LINPHONE_PUBLIC void linphone_conference_params_enable_local_participant(LinphoneConferenceParams *params,
                                                                         bool_t enable);

/**
 * Returns whether local participant has to enter the conference.
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if local participant is by default part of the conference, FALSE otherwise
 */
LINPHONE_PUBLIC bool_t linphone_conference_params_local_participant_enabled(const LinphoneConferenceParams *params);

/**
 * Enable conference with one participant.
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable TRUE if conference can have only one participant, FALSE otherwise
 * @deprecated 16/12/2021 Use linphone_conference_params_enable_one_participant_conference() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED void
linphone_conference_params_set_one_participant_conference_enabled(LinphoneConferenceParams *params, bool_t enable);

/**
 * Enable conference with one participant.
 * @param params A #LinphoneConferenceParams @notnil
 * @param enable TRUE if conference can have only one participant, FALSE otherwise
 */
LINPHONE_PUBLIC void linphone_conference_params_enable_one_participant_conference(LinphoneConferenceParams *params,
                                                                                  bool_t enable);

/**
 * Returns whether conference can have only one participant
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if the conference can have only one participant, FALSE otherwise
 * @deprecated 16/12/2021 Use linphone_conference_params_one_participant_conference_enabled() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED bool_t
linphone_conference_params_is_one_participant_conference_enabled(const LinphoneConferenceParams *params);

/**
 * Returns whether conference can have only one participant
 * @param params A #LinphoneConferenceParams @notnil
 * @return TRUE if the conference can have only one participant, FALSE otherwise
 */
LINPHONE_PUBLIC bool_t
linphone_conference_params_one_participant_conference_enabled(const LinphoneConferenceParams *params);

/**
 * Retrieve the desired security level of the conference.
 * @param params The #LinphoneConferenceParams object. @notnil
 * @return The desired security level of the conference.
 */
LINPHONE_PUBLIC LinphoneConferenceSecurityLevel
linphone_conference_params_get_security_level(const LinphoneConferenceParams *params);

/**
 * Set the desired security level of the conference.
 * @param params The #LinphoneConferenceParams object. @notnil
 * @param security_level The desired security level of the conference.
 */
LINPHONE_PUBLIC void linphone_conference_params_set_security_level(LinphoneConferenceParams *params,
                                                                   LinphoneConferenceSecurityLevel security_level);

/**
 * Take a reference on a #LinphoneConference.
 * @param conference The #LinphoneConference to ref. @notnil
 * @return The same #LinphoneConference object. @notnil
 */
LINPHONE_PUBLIC LinphoneConference *linphone_conference_ref(LinphoneConference *conference);

/**
 * Release a #LinphoneConference.
 * @param conference The #LinphoneConference to release. @notnil
 */
LINPHONE_PUBLIC void linphone_conference_unref(LinphoneConference *conference);

/**
 * Get URIs of all participants of one conference
 * The returned bctbx_list_t contains URIs of all participants. That list must be
 * freed after use and each URI must be unref with linphone_address_unref()
 * @warning The returned list does not include me.
 * @param conference A #LinphoneConference @notnil
 * @return The list of the participants' address active in the conference. \bctbx_list{LinphoneAddress} @maybenil
 * @tobefreed
 * @deprecated 10/07/2020 Use linphone_conference_get_participant_list() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED bctbx_list_t *
linphone_conference_get_participants(const LinphoneConference *conference);

/**
 * Get list of all participants of a conference
 * @warning The returned list does not include me.
 * @param conference A #LinphoneConference @notnil
 * @return The list of participants of the conference. \bctbx_list{LinphoneParticipant} @maybenil @tobefreed
 */
LINPHONE_PUBLIC bctbx_list_t *linphone_conference_get_participant_list(const LinphoneConference *conference);

/**
 * Get list of all participant devices of a conference including me if it is in
 * @param conference A #LinphoneConference @notnil
 * @return The list of participant devices of the conference. \bctbx_list{LinphoneParticipantDevice} @maybenil
 * @tobefreed
 */
LINPHONE_PUBLIC bctbx_list_t *linphone_conference_get_participant_device_list(const LinphoneConference *conference);

/**
 * @param conference A #LinphoneConference @notnil
 * @param uri URI of the participant to remove @notnil
 * @warning The passed participant uri must be one of those returned by linphone_conference_get_participants()
 * @return 0 if succeeded, -1 if failed
 * @deprecated 10/07/2020 Use linphone_conference_remove_participant_2() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED LinphoneStatus
linphone_conference_remove_participant(LinphoneConference *conference, const LinphoneAddress *uri);

/**
 * @param conference A #LinphoneConference @notnil
 * @param participant participant to remove @notnil
 * @warning The passed participant must be one of those returned by linphone_conference_get_participant_list()
 * @warning This method may destroy the conference if the only remaining participant had an existing call to the local
 * participant before the conference was created
 * @return 0 if succeeded, -1 if failed
 */
LINPHONE_PUBLIC LinphoneStatus linphone_conference_remove_participant_2(LinphoneConference *conference,
                                                                        LinphoneParticipant *participant);

/**
 * @param conference A #LinphoneConference @notnil
 * @param call call to remove @notnil
 * @return 0 if succeeded, -1 if failed
 * @deprecated 10/07/2020 Use linphone_conference_remove_participant_2() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED LinphoneStatus
linphone_conference_remove_participant_3(LinphoneConference *conference, LinphoneCall *call);

/**
 * Find a participant from a conference
 * @param conference A #LinphoneConference. @notnil
 * @param uri SIP URI of the participant to search. @notnil
 * @return a pointer to the participant found or nullptr. @maybenil
 */
LINPHONE_PUBLIC LinphoneParticipant *linphone_conference_find_participant(LinphoneConference *conference,
                                                                          const LinphoneAddress *uri);

/**
 * Invite participants to the conference, by supplying a list of #LinphoneAddress
 * If the conference is in the state LinphoneConferenceStateCreationPending, then the conference will start on the input
 *and output audio devices used for the currently active call, if any
 * @param conference The #LinphoneConference object. @notnil
 * @param addresses A list of SIP addresses to invite. @bctbx_list{LinphoneAddress} @notnil
 * @param params #LinphoneCallParams to use for inviting the participants. @maybenil
 **/
LINPHONE_PUBLIC LinphoneStatus linphone_conference_invite_participants(LinphoneConference *conference,
                                                                       const bctbx_list_t *addresses,
                                                                       const LinphoneCallParams *params);

/**
 * Add participants to the conference, by supplying a list of #LinphoneCall. If the conference is in the state
 *LinphoneConferenceStateCreationPending, then the conference will start on the input and output audio devices used for
 *the currently active call, if any
 * @param conference The #LinphoneConference object. @notnil
 * @param calls A list of calls to add to the conference. @bctbx_list{LinphoneCall} @notnil
 **/
LINPHONE_PUBLIC LinphoneStatus linphone_conference_add_participants(LinphoneConference *conference,
                                                                    const bctbx_list_t *calls);

/**
 * Add participants to the conference, by supplying a list of #LinphoneAddress.
 * @param conference The #LinphoneConference object. @notnil
 * @param addresses A list of calls to add to the conference. @bctbx_list{LinphoneAddress} @notnil
 **/
LINPHONE_PUBLIC LinphoneStatus linphone_conference_add_participants_2(LinphoneConference *conference,
                                                                      const bctbx_list_t *addresses);

/**
 * Join an existing call to the conference.
 * If the conference is in the state LinphoneConferenceStateCreationPending, then the conference will start on the input
 * and output audio devices used for the currently active call, if any
 * @param conference The #LinphoneConference object. @notnil
 * @param call a #LinphoneCall that has to be added to the conference. @notnil
 * @warning This function guarantees that the local endpoint is added to the conference only if one of calls added is in
 * state StreamsRunning. It is highly recommended to call linphone_confererence_enter() to guarantee that the local
 * endpoint is added to the conference.
 */
LINPHONE_PUBLIC LinphoneStatus linphone_conference_add_participant(LinphoneConference *conference, LinphoneCall *call);

/**
 * Join a participant to the conference.
 * @param conference The #LinphoneConference object. @notnil
 * @param uri a #LinphoneAddress that has to be added to the conference. @notnil
 * @warning This function guarantees that the local endpoint is added to the conference only if there is a call state
 * StreamsRunning towards one of the addresses. It is highly recommended to call linphone_confererence_enter() to
 * guarantee that the local endpoint is added to the conference.
 */
LINPHONE_PUBLIC LinphoneStatus linphone_conference_add_participant_2(LinphoneConference *conference,
                                                                     const LinphoneAddress *uri);

/**
 * Update parameters of the conference.
 * This is typically used enable or disable the video stream in the conference.
 * @param conference The #LinphoneConference object. @notnil
 * @param params the new parameters to apply. @notnil
 */
LINPHONE_PUBLIC int linphone_conference_update_params(LinphoneConference *conference,
                                                      const LinphoneConferenceParams *params);

/**
 * Get current parameters of the conference.
 * @param conference The #LinphoneConference object. @notnil
 * @return a #LinphoneConferenceParams . @notnil
 */
LINPHONE_PUBLIC const LinphoneConferenceParams *
linphone_conference_get_current_params(const LinphoneConference *conference);

/**
 * Get the conference subject
 * @param conference The #LinphoneConference object. @notnil
 * @return conference subject. @maybenil
 */
LINPHONE_PUBLIC const char *linphone_conference_get_subject(const LinphoneConference *conference);

/**
 * Set the conference subject
 * @param conference The #LinphoneConference object. @notnil
 * @param subject conference subject @maybenil
 */
LINPHONE_PUBLIC void linphone_conference_set_subject(LinphoneConference *conference, const char *subject);

/**
 * Get the conference username
 * @param conference The #LinphoneConference object. @notnil
 * @return conference subject. @maybenil
 */
LINPHONE_PUBLIC const char *linphone_conference_get_username(const LinphoneConference *conference);

/**
 * Set the conference username
 * @param conference The #LinphoneConference object. @notnil
 * @param username conference subject @maybenil
 */
LINPHONE_PUBLIC void linphone_conference_set_username(LinphoneConference *conference, const char *username);

/**
 * Set stream capability on me device of a local conference
 * @param conference The #LinphoneConference object. @notnil
 * @param direction the direction of stream of type stream_type
 * @param type The #LinphoneStreamType
 */
LINPHONE_PUBLIC void linphone_conference_set_local_participant_stream_capability(LinphoneConference *conference,
                                                                                 const LinphoneMediaDirection direction,
                                                                                 const LinphoneStreamType type);

/**
 * Get the conference duration
 * @param conference The #LinphoneConference object. @notnil
 * @return conference duration. @maybenil
 */
LINPHONE_PUBLIC int linphone_conference_get_duration(const LinphoneConference *conference);

/**
 * Get the conference start time
 * @param conference The #LinphoneConference object. @notnil
 * @return conference start time. @maybenil
 */
LINPHONE_PUBLIC time_t linphone_conference_get_start_time(const LinphoneConference *conference);

/**
 * Get number of participants without me
 * @param conference The #LinphoneConference object. @notnil
 * @return the number of participants excluding me in a #LinphoneConference
 */
LINPHONE_PUBLIC int linphone_conference_get_participant_count(const LinphoneConference *conference);

/**
 * For a local audio video conference, this function returns the participant hosting the conference
 * For a remote audio video conference, this function returns the local participant of the conference
 * @param conference The #LinphoneConference object. @notnil
 * @return a #LinphoneParticipant . @notnil
 */
LINPHONE_PUBLIC LinphoneParticipant *linphone_conference_get_me(const LinphoneConference *conference);

/**
 * Terminates conference
 * @param conference The #LinphoneConference object. @notnil
 * @return 0 if the termination is successful, -1 otherwise.
 */
LINPHONE_PUBLIC int linphone_conference_terminate(LinphoneConference *conference);

/**
 * Retrieves the user pointer that was given to linphone_conference_new()
 * @param conference #LinphoneConference object @notnil
 * @return The user data associated with the #LinphoneConference object. @maybenil
 * @ingroup initializing
 **/
LINPHONE_PUBLIC void *linphone_conference_get_user_data(const LinphoneConference *conference);

/**
 * Associate a user pointer to the linphone conference.
 * @param conference #LinphoneConference object @notnil
 * @param user_data The user data to associate with the #LinphoneConference object. @maybenil
 * @ingroup initializing
 **/
LINPHONE_PUBLIC void linphone_conference_set_user_data(LinphoneConference *conference, void *user_data);

/**
 * Change the admin status of a participant of a conference (you need to be an admin yourself to do this).
 * @param conference A #LinphoneConference object @notnil
 * @param participant The Participant for which to change the admin status @notnil
 * @param is_admin A boolean value telling whether the participant should now be an admin or not
 */
LINPHONE_PUBLIC void linphone_conference_set_participant_admin_status(LinphoneConference *conference,
                                                                      LinphoneParticipant *participant,
                                                                      bool_t is_admin);

/**
 * For a local conference, the local participant joins the conference
 * For a remote conference, the participant rejoins the conference after leaving it earlier on
 * @param conference A #LinphoneConference object @notnil
 * @return 0 if succeeded. Negative number if failed
 */
LINPHONE_PUBLIC int linphone_conference_enter(LinphoneConference *conference);

/**
 * For a local conference, the local participant leaves the conference
 * For a remote conference, the participant leaves the conference after joining it earlier on
 * @param conference A #LinphoneConference object @notnil
 * @return 0 if succeeded. Negative number if failed
 */
LINPHONE_PUBLIC int linphone_conference_leave(LinphoneConference *conference);

/**
 * For a local audio video conference, this function compares the address provided as argument with that of participant
 * hosting the conference For a remote audio video conference, this function compares the address provided as argument
 * with that of the local participant of the conference
 * @param conference A #LinphoneConference object @notnil
 * @param uri A #LinphoneAddress object @notnil
 * @return TRUE if the participant is me, FALSE otherwise.
 */
LINPHONE_PUBLIC bool_t linphone_conference_is_me(const LinphoneConference *conference, const LinphoneAddress *uri);

/**
 * For a local conference, it returns whether the local participant is enabled
 * For a remote conference, it return whether the remote participant has left the conference without bein removed from
 * it
 * @param conference A #LinphoneConference object @notnil
 * @return TRUE if the local participant is in a conference, FALSE otherwise.
 */
LINPHONE_PUBLIC bool_t linphone_conference_is_in(const LinphoneConference *conference);

/**
 * Sets the given #LinphoneAudioDevice as input for this conference only.
 * @param conference The #LinphoneConference @notnil
 * @param audio_device The #LinphoneAudioDevice. NULL does nothing. @maybenil
 */
LINPHONE_PUBLIC void linphone_conference_set_input_audio_device(LinphoneConference *conference,
                                                                LinphoneAudioDevice *audio_device);

/**
 * Sets the given #LinphoneAudioDevice as output for this conference only.
 * @param conference The #LinphoneConference @notnil
 * @param audio_device The #LinphoneAudioDevice. NULL does nothing. @maybenil
 */
LINPHONE_PUBLIC void linphone_conference_set_output_audio_device(LinphoneConference *conference,
                                                                 LinphoneAudioDevice *audio_device);

/**
 * Gets the current input device for this conference.
 * @param conference The #LinphoneConference @notnil
 * @return the #LinphoneAudioDevice used by this conference as input or NULL if there is currently no soundcard
 * configured (depending on the state of the conference) @maybenil
 */
LINPHONE_PUBLIC const LinphoneAudioDevice *
linphone_conference_get_input_audio_device(const LinphoneConference *conference);

/**
 * Gets the current output device for this conference.
 * @param conference The #LinphoneConference @notnil
 * @return the #LinphoneAudioDevice used by this conference as output or NULL if there is currently no soundcard
 * configured (depending on the state of the conference) @maybenil
 */
LINPHONE_PUBLIC const LinphoneAudioDevice *
linphone_conference_get_output_audio_device(const LinphoneConference *conference);

/**
 * Retrieves the volume of a specific participant
 * @param conference A #LinphoneConference object @notnil
 * @param device The Participant @notnil
 * @return The volume of the participant expressed in dbm0.
 */
LINPHONE_PUBLIC int linphone_conference_get_participant_device_volume(LinphoneConference *conference,
                                                                      LinphoneParticipantDevice *device);

/**
 * Starts recording the conference.
 * @param conference A #LinphoneConference object @notnil
 * @param path Where to record the conference @notnil
 * @return 0 if succeeded. Negative number in case of failure.
 */
LINPHONE_PUBLIC int linphone_conference_start_recording(LinphoneConference *conference, const char *path);

/**
 * Stops the conference recording.
 * @param conference A #LinphoneConference object @notnil
 * @return 0 if succeeded. Negative number in case of failure.
 */
LINPHONE_PUBLIC int linphone_conference_stop_recording(LinphoneConference *conference);

/**
 * Gets whether the conference is currently being recorded.
 * @param conference A #LinphoneConference object @notnil
 * @return TRUE if conference is being recorded, FALSE otherwise.
 */
LINPHONE_PUBLIC bool_t linphone_conference_is_recording(const LinphoneConference *conference);

/**
 * Gets the call that is controlling a conference.
 * - for the local conference, it will return NULL
 * - for the remote conference, it will return call associated to the conference
 *
 * @param conference The #LinphoneConference @notnil
 * @return the #LinphoneCall controlling the conference or NULL if none or local conference @maybenil
 */
LINPHONE_PUBLIC LinphoneCall *linphone_conference_get_call(const LinphoneConference *conference);

/**
 * Retrieves the volume of a specific participant
 * @param conference A #LinphoneConference object @notnil
 * @return TRUE if the microphone is muted, FALSE otherwise
 */
LINPHONE_PUBLIC bool_t linphone_conference_get_microphone_muted(const LinphoneConference *conference);

/**
 * Get microphone muted state.
 * Note that the microphone may be disabled globally if FALSE was given to
 * linphone_core_enable_mic().
 * @param conference A #LinphoneConference object @notnil
 * @param muted The microphone muted state.
 **/
LINPHONE_PUBLIC void linphone_conference_set_microphone_muted(LinphoneConference *conference, bool_t muted);

/**
 * Retrieves the volume of a specific participant
 * @param conference A #LinphoneConference object @notnil
 * @return The volume of the participant expressed in dbm0.
 */
LINPHONE_PUBLIC float linphone_conference_get_input_volume(const LinphoneConference *conference);

/************ */
/* DEPRECATED */
/* ********** */

/**
 * Get the conference id as string
 * @param conference The #LinphoneConference object. @notnil
 * @return the conference id @notnil
 * @deprecated 10/07/2020 Use linphone_conference_get_conference_address() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED const char *linphone_conference_get_ID(const LinphoneConference *conference);

/**
 * Set the conference id as string
 * @param conference The #LinphoneConference object. @notnil
 * @param conference_id the conference id to set. @notnil
 * @deprecated 10/07/2020 Use linphone_conference_set_conference_address() instead.
 */
LINPHONE_PUBLIC LINPHONE_DEPRECATED void linphone_conference_set_ID(LinphoneConference *conference,
                                                                    const char *conference_id);

LinphoneConference *linphone_local_conference_new(LinphoneCore *core, LinphoneAddress *addr);
LinphoneConference *linphone_local_conference_new_with_params(LinphoneCore *core,
                                                              LinphoneAddress *addr,
                                                              const LinphoneConferenceParams *params);
LinphoneConference *linphone_remote_conference_new(LinphoneCore *core, LinphoneAddress *addr);
LinphoneConference *linphone_remote_conference_new_with_params(LinphoneCore *core,
                                                               LinphoneAddress *focus,
                                                               LinphoneAddress *addr,
                                                               const LinphoneConferenceParams *params);

/* This is actually only used by the ToneManager. TODO: encapsulate this better. */
AudioStream *linphone_conference_get_audio_stream(LinphoneConference *conference);

void linphone_conference_on_call_terminating(LinphoneConference *conference, LinphoneCall *call);

LINPHONE_PUBLIC bool_t linphone_conference_check_class(LinphoneConference *conference, LinphoneConferenceClass _class);

void linphone_conference_notify_audio_device_changed(LinphoneConference *conference, LinphoneAudioDevice *audio_device);

/**
 * @}
 */

#ifdef __cplusplus
}
#endif

#endif // LINPHONE_CONFERENCE_H
