from sshpubkeys.exceptions import (
    InvalidKeyError, InvalidKeyLengthError, InvalidTypeError, MalformedDataError, TooLongKeyError, TooShortKeyError
)

keys = [
    [
        "ssh-dss AAAAB3NzaC1kc3MAAACBAPlHIP5sD+T8/Sx1DGEiCzCXqpl7ww40jBg7wTkxu44OH6pNog5PjJt5M4NBULhKva/i+bhIM3ba+H1Or+aHWWFHACV6W2FCGk/k37ApRF8sIa4hsnN0P9qn6VfhbJKee+DBxa21WjjY/MZiljmJz7IQHx5RTxX9I/hJ7cL+aNmrAAAAFQCKteqc4IkgIrjpcpStsxYAhb3MqQAAAIEA+SfIKuTr7QPcinsZQDdmZOXqcg+u9TLzHA4c47y0Kns3T3BVPr9rWdmuh6eImzLO4wMLxLvcg3ecrqFuiCp1IHvXENkGlpB17S+uOXlVDY+sTdXyvYKRKirg5IZefIAP/m08c0QGkhFDbo4ysr9D5gXgH3LB2rMPIAbvMWm/HZQAAACBAKWtAE3hXRQX5KtI4AoIWVTly/6T4JNBt4u24ZRqV7X//CZEZ0cS5YpR/frlpUDI3WKoMtS+VmT3cBFZINashIxZyfBF8+0UX3s34HwNfp0hDW3ZdgZJU56GC2eclMantYGeVrMxgTQd80pxZFgByEhoXGeZaAwUzN8ULo9jHQo=",
        MalformedDataError, "too_short_data", ["loose", "strict"]
    ],
    [
        "ssh-dss AAAAB3NzaC1kc3MAAACBAPlHIP5sD+T8/Sx1DGEiCzCXqpl7ww40jBg7wTkxu44OH6pNog5PjJt5M4NBULhKva/i+bhIM3ba+H1Or+aHWWFHACV6W2FCGk/k37ApRF8sIa4hsnN0P9qn6VfhbJKee+DBxa21WjjY/MZiljmJz7IQHx5RTxX9I/hJ7cL+aNmrAAAAFQCKteqc4IkgIrjpcpStsxYAhb3MqQAAAIEA+SfIKuTr7QPcinsZQDdmZOXqcg+u9TLzHA4c47y0Kns3T3BVPr9rWdmuh6eImzLO4wMLxLvcg3ecrqFuiCp1IHvXENkGlpB17S+uOXlVDY+sTdXyvYKRKirg5IZefIAP/m08c0QGkhFDbo4ysr9D5gXgH3LB2rMPIAbvMWm/HZQAAACBAKWtAE3hXRQX5KtI4AoIWVTly/6T4JNBt4u24ZRqV7X//CZEZ0cS5YpR/frlpUDI3WKoMtS+VmT3cBFZINashIxZyfBF8+0UX3s34HwNfp0hDW3ZdgZJU56GC2eclMantYGeVrMxgTQd80pxZFgByEhoXGeZaAwUzN8ULo9jHQqMjA==",
        MalformedDataError, "too_long_data", ["loose", "strict"]
    ],
    [
        "ssh-dss AAAAB3NzaC1yc2EAAACBAPlHIP5sD+T8/Sx1DGEiCzCXqpl7ww40jBg7wTkxu44OH6pNog5PjJt5M4NBULhKva/i+bhIM3ba+H1Or+aHWWFHACV6W2FCGk/k37ApRF8sIa4hsnN0P9qn6VfhbJKee+DBxa21WjjY/MZiljmJz7IQHx5RTxX9I/hJ7cL+aNmrAAAAFQCKteqc4IkgIrjpcpStsxYAhb3MqQAAAIEA+SfIKuTr7QPcinsZQDdmZOXqcg+u9TLzHA4c47y0Kns3T3BVPr9rWdmuh6eImzLO4wMLxLvcg3ecrqFuiCp1IHvXENkGlpB17S+uOXlVDY+sTdXyvYKRKirg5IZefIAP/m08c0QGkhFDbo4ysr9D5gXgH3LB2rMPIAbvMWm/HZQAAACBAKWtAE3hXRQX5KtI4AoIWVTly/6T4JNBt4u24ZRqV7X//CZEZ0cS5YpR/frlpUDI3WKoMtS+VmT3cBFZINashIxZyfBF8+0UX3s34HwNfp0hDW3ZdgZJU56GC2eclMantYGeVrMxgTQd80pxZFgByEhoXGeZaAwUzN8ULo9jHQqM",
        InvalidTypeError, "key_type_mismatch", ["loose", "strict"]
    ],
    [
        "ssh-dss AAAAB3NzaC1yc2EAAAADAQABAAAEAgDGrGaNv7i+sGSelzf+7JsCECa9a0sqSg8q4foGkjeV6RkS2tWvKXoT9rICjEdXXodj0CCVhe/V7dmAO0AK8KM0mcvPfTSC8zH1ZBsqaFFTWwmBD01fbH9axrrg3hM0f+AL4bMMWUdxdNrVo90s8PKU6k/HmUNLVx4gC6uQ4A6YczvOVZkuJ4f7HDYK/v1LNTRNeAkw94YpSIZVAoTOZN943+fRCE9cm155pwmFsS+wfzK9+jjhGXNEK0xooiVBRwQM7qetN076vV5FiiM0LO1qYi5JrIqK/70ske86x2mMhMkOe6jqQQbt32PFVmYqYJWcAYXz+bhcQw6oru0c6gNq53aGOnuqI0uh/zV2XH+cN4c8ABcOplzH5YQEUepNVzxylkvpWxdg/ZzR1pvyu5C8RkJWrE3AlCwpix1ak2xTDzgc3rwTTggNSYqvzmYq0mYJhZk2VWsLVxUgdxfwC3LvIHMXSTU9iU2Aqrlhy7bJAqxQFKWy05wsIOI6raPBLqZnPmJ76Ld9aXTrhBFfIDiigr9ZVsVAdOvmyAGCIj4x3Xnlol/3lN0M2+OSV1SU/5ZrS6dIlXPZDak/OXHU0iIFIODhYU5r8EI1M6BI/jsgQ8HatXmOJkfnIkVP0HxD1KvoAFKjVG5sM9KG12LqsnzfD1KL6PzxpOOgoVgznpOjSzVmPKAkU8N/r6R4VIAmZqxpF8Hlzqg/Gfh5kf6CJXXx8OQt1Z/DAsfnl3LvHFNuE8GgXgrUE022W9pV4oONgojc97JSgBXaFkK885UnJKTceAdGQvChEhsU1j3TiyKPox6ICGpoC2nGONJoDE8VQ8dE/YiZmqkZ1lJWX07EwevrIcnz1UBHFaR72aiAADRYClsitLA5+1mnydVstkQ8XQuuKNOFT7miaWUzRHwj9BYGb7oGhNd9oi1VTVjH/5Yq1UiHHESGaIjeLi5uG2KguDFpcvy2ngtUy3ZbvDj+DVOLL+3vAlycRPjN0nBE4e/J6UqdpLg0DbG56zNj86aU0ZgL8kL8NRkFHyV+5zG5iLFkGklbm4nwCxSW/bVT0PFD1is6JbtIk5i+liS+hiuzSF6NGouSuxDy95yWSG8/84fgPDFtvXtOD7Kl4P7EpEAL+VBZnremT9I8tRl1wOHxJKe7jbEcWC2zkuHNlju0Nv5SFijF9c+krRbHDYEzsxPpdqlI4gPtDFdkKwaKN6BrsxBsz9u+PhS1AloUYcxKRqWbqHuDBrKmxnhOgFqJ9ITX0RajtrApt1LfkSBXcFrVEx2nhQkGa6VwjcX/zw2I2iuJFOCQmc9udHIlyaCCSe1PqOIbOlOk5h/Gl1QvRNwSIgf9dZ05lZr6dc+VX8YGdyHsjQ==",
        InvalidTypeError, "rsa_key_no_match", ["loose", "strict"]
    ],
    [
        "ssh-dss AAAAB3NzaC1kc3MAAACAwQBAoKCAoECggSAgAOAAwOCg4AEAIMBAgMBAYSBg4GBggKAgwOBAwEDBAMCgwECg4GCBAAEhAABhIOCBIIEAQECg4GEAoQEgQQBgAGEAYQEAgCDAoCDgoIDBICAAAOCgwGDhAAAAwQEBIKCAYEBA4ODg4EDggSAgAGEgAQCAAOAAAAAgAAAAAAAAAAAAAAAAAACEAYKAgICCAIMAAYCBAYICAoAAAACAgKEgQCEhIKDgISCAAIDgwGEgASEg4GAgwQDAoCEAAEDBICDhAAABACCgIACBAIBAwOEgQEAA4KBBAOAAIIDAIQAgoGBgYGCgYCCAYEEgoEEgAABAIACAIAChAKCgoKAhIMDAgMEBIIDgAQEgwCDAAMBA4KAgACBAgCCgIGBAoGAAAACAwKAgAIBgIIBhAIBgQSAAgMAAQICggQDg4SCgoACgASCAwKCAYSAAoMCAYAEAoKBgoQEBIAAgICBAoMDBAMEgYIAgYKDggSDAoQDgQEAgAOBgQAAggQCAoCAgwAEBIEDgYABAgOEAwECAQECAAQDAAECBIGBAgEBA4AAgQSEBAKA=",
        InvalidKeyError, "invalid_q_for_dsa", ["loose", "strict"]
    ],
    [
        "ssh-dss AAAAB3NzaC1kc3MAAACAgIBAAIBgQJAwgJCAQCBwICAwYFAwcEBQMBBwUHAgcCAAkEAQQECQkBAgkGAwcEAAkIAAMAAwAJAAcABQQCAwIHAwIIAwYAAwICCQkAAAUHAwIBCAMBBQcCAQAFBggEBAQDAgIHAwUJAwkFCQUDBwkIAQEGAwICAQADAwcJAAEJAAAAAgAAAAAAAAAAAAAAAAhIIAggODBAAEgwIAhIGChIICBAAAAACAwQJBAcECQQBAQIGBAEHBQEGBgYCAAQGCAIECQMHBwAIAAIDBAMIBgEEAQADBwIDAgcABAYHBwMDAgcABQgABwAHAAkBBwYAAwYJBwgAAwQFCQICAQgFAQICAgAIAwQHBQYJBQADAgAIAgMHBwkJAQEFBAMGAQAGBQQHAggBAQIA=",
        MalformedDataError, "missing_y_from_dsa", ["loose", "strict"]
    ],
    [
        'ssh-dss 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',
        InvalidKeyLengthError, "too_long_dsa_4096", ["loose", "strict"]
    ],
    [
        'ssh-dss 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',
        InvalidKeyLengthError, "too_long_dsa_16384", ["loose", "strict"]
    ],
    [
        "ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAEAgDGrGaNv7i+sGSelzf+7JsCECa9a0sqSg8q4foGkjeV6RkS2tWvKXoT9rICjEdXXodj0CCVhe/V7dmAO0AK8KM0mcvPfTSC8zH1ZBsqaFFTWwmBD01fbH9axrrg3hM0f+AL4bMMWUdxdNrVo90s8PKU6k/HmUNLVx4gC6uQ4A6YczvOVZkuJ4f7HDYK/v1LNTRNeAkw94YpSIZVAoTOZN943+fRCE9cm155pwmFsS+wfzK9+jjhGXNEK0xooiVBRwQM7qetN076vV5FiiM0LO1qYi5JrIqK/70ske86x2mMhMkOe6jqQQbt32PFVmYqYJWcAYXz+bhcQw6oru0c6gNq53aGOnuqI0uh/zV2XH+cN4c8ABcOplzH5YQEUepNVzxylkvpWxdg/ZzR1pvyu5C8RkJWrE3AlCwpix1ak2xTDzgc3rwTTggNSYqvzmYq0mYJhZk2VWsLVxUgdxfwC3LvIHMXSTU9iU2Aqrlhy7bJAqxQFKWy05wsIOI6raPBLqZnPmJ76Ld9aXTrhBFfIDiigr9ZVsVAdOvmyAGCIj4x3Xnlol/3lN0M2+OSV1SU/5ZrS6dIlXPZDak/OXHU0iIFIODhYU5r8EI1M6BI/jsgQ8HatXmOJkfnIkVP0HxD1KvoAFKjVG5sM9KG12LqsnzfD1KL6PzxpOOgoVgznpOjSzVmPKAkU8N/r6R4VIAmZqxpF8Hlzqg/Gfh5kf6CJXXx8OQt1Z/DAsfnl3LvHFNuE8GgXgrUE022W9pV4oONgojc97JSgBXaFkK885UnJKTceAdGQvChEhsU1j3TiyKPox6ICGpoC2nGONJoDE8VQ8dE/YiZmqkZ1lJWX07EwevrIcnz1UBHFaR72aiAADRYClsitLA5+1mnydVstkQ8XQuuKNOFT7miaWUzRHwj9BYGb7oGhNd9oi1VTVjH/5Yq1UiHHESGaIjeLi5uG2KguDFpcvy2ngtUy3ZbvDj+DVOLL+3vAlycRPjN0nBE4e/J6UqdpLg0DbG56zNj86aU0ZgL8kL8NRkFHyV+5zG5iLFkGklbm4nwCxSW/bVT0PFD1is6JbtIk5i+liS+hiuzSF6NGouSuxDy95yWSG8/84fgPDFtvXtOD7Kl4P7EpEAL+VBZnremT9I8tRl1wOHxJKe7jbEcWC2zkuHNlju0Nv5SFijF9c+krRbHDYEzsxPpdqlI4gPtDFdkKwaKN6BrsxBsz9u+PhS1AloUYcxKRqWbqHuDBrKmxnhOgFqJ9ITX0RajtrApt1LfkSBXcFrVEx2nhQkGa6VwjcX/zw2I2iuJFOCQmc9udHIlyaCCSe1PqOIbOlOk5h/Gl1QvRNwSIgf9dZ05lZr6dc+VX8YGdyHsjQ=",
        InvalidKeyError, "broken_rsa_base64", ["loose", "strict"]
    ],
    [
        "ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAH0GODBKRjsFB/1v3pDRGpA6xR+QpOJg9vat0brlbUNA=", InvalidKeyLengthError,
        "too_short_ed25519", ["loose", "strict"]
    ],
    [
        "ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIUGODBKRjsFB/1v3pDRGpA6xR+QpOJg9vat0brlbUNDDpA==", InvalidKeyLengthError,
        "too_long_ed25519", ["loose", "strict"]
    ],
    [
        "ecdsa-sha2-nistp256 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTYAAAAIbmlzdHAyNTYAAABBBE2gqbAChP2h3fTPx3Jy2KdOJUiBGEiqBUwoosfzllw+KrqmGiDEWlufSxdiSOFuLd4a8PSwhoWbdQgRVFrZAvFE=",
        InvalidKeyError, "invalid_ecdsa_key", ["loose", "strict"]
    ],
    [
        "ecdsa-sha2-nistp255 AAAAE2VjZHNhLXNoYTItbmlzdHAyNTUAAAAIbmlzdHAyNTUAAABBBE2gqbAChP2h3fTPx3Jy2KdOJUiBGEiqBUwoosfzllw+KrqmGiDEWlufSxdiSOFuLd4a8PSwhoWbdQRVFrZAvFE=",
        NotImplementedError, "invalid_nist_curve", ["loose", "strict"]
    ],
    ["", InvalidKeyError, "empty_key", ["loose", "strict"]],
    ["- -", InvalidKeyError, "no_content", ["loose", "strict"]],
    [
        "ssh-invalid-key-type AAAAFHNzaC1pbnZhbGlkLWtleS10eXBlAAAAgMEgAGDgAKEBAMAgoECggSCAQIEAoIAhASEhAKDAAOAAwGCBIKAAYGCgYKDhAEBAwICAoIDgIADAYKAAYGAAYAEgwGBBIGDhIOBg4CAAoQDAIEAgIEEggEDgQGCAIGAhAQDgoKAhIAAggSBA4CBggOEBIEDgoMDAwKBg4MDAYIDgAAAAIAAAAAAAAAAAAAAAAIAggEEggSBAIIBgAMDAQADAIQEgAAAAgJAQABBgICBQUABQcFAAcJBgUIAAIHAAkJBQEIAQMEAgIHBQEFCAUBBQUHAQMGBgQBCAEEAgICBQYDBwIJBAYFCQYDBwIAAQIHCQMEBwYCBAICAAEFBQAHAgAIBQAJBwgEBwYGCAEHBgkBAwEBAQcHAQMGCQMCBQIABAUJAgEIAgAAAAgIGAAYIDgoSBA4ABAgAEAIEEAoEAg4IDAgQCAwCDgQMAgQABgoGAAgMCgoOCA4QABAGBAQQDgoKABASBhAQDgoIBgIQEgAQEAQSBg4IEBICEggKBAAABgYKDAgGDgYAAgoGCAoODgYGEAwIBAoACgQADBAGDgASCgQMDgAABAISA",
        NotImplementedError, "not_implemented_key_type", ["loose", "strict"]
    ],

    # 512 bit ed25519 keys are not yet supported by OpenSSH
    [
        "ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAQIMcGCUjHYKD/rfvSGiNSB1ij8hScTB7e1bo3XK2oaGGAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA=",
        InvalidKeyLengthError, "512bit_ed25519", ["loose", "strict"]
    ],
    [
        "ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAQIBBAIDAICBAgEABACAggGCgAABAwOEAwGBAgEEg4SBAAKBBIGEAgSEgAIEgoMCgICAgYMCgYQDgoKBgIIDgwCA=",
        InvalidKeyLengthError, "512bit_ed25519", ["loose", "strict"]
    ],
    [
        'command="dump /home,no-pty,no-port-forwarding ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAABADR9kolU4uiD26LMrbakQlNf4QWB2xrdY3nASf6CJdQYzTMjNmbt6sJ4A4pGnCupFrzL04EYDvbVmT4GEZm6CU4BsY61yosnpGSqqcVCdw5xW1k4bCSDPW75WHLCVmYyROhZ+yyo8uAcIy5UIyBZXF/PO7taJrrIi5RwdqIPwtCrJ3dJkcFWa3qZWJykLAFQD5A/lta/egS/u/nyCap2e16WGnvSluz9CyYtGFNS9axzOwHxLFEv2ocOsJjYgzV+Jfpiao94A4VzLKbUDHlfV57KS0tJaT8FKKsg34vN3bsD0zUftLUPpUFgJfMwje0C2rCJkCzwgya2vxLqj2fg0Q0= ojarva@ojar-laptop.local',
        MalformedDataError, "missing_quote", ["loose", "strict"]
    ],
    [
        'ssh-dss AAAAB3NzaC1kc3MAAAEBAM4SvSD+Gpu4L5TvkroBbAcVPeQID1gdZTpr5fEuOCwTVxMyuPwfHH6txh4Nq0K7MRee5zuVKHNkxj84EMF/+g4eJISmE87wzfYNJQhGNOmXO3yYGCmFaDgrlsWH9OhGglWTxz768gJy4IUP2hqvsbotdpjhRvTFCIep12r4L16rFQJeVVti6Xml3UZHXvoMbwJC+BulwU8hnosXvH4mYikgoaSmDUfJkkLB1nD/g48vo172iu2mfEYqAbn9lX46vnetmpz87IHxbCHmIuJxhMfH+vMY5zvF7vUy018XUPrAcqDkmszhJOFuMhB2SQ6IT8nQZnZ+aGGnoQ2kgjCrrusAAAAVAN/iBeqRO1CPAuNIec08Y9VP+MFTAAABAGDVzqw4i64Fd1mLi6BRKYZdbMI56gx+Cu+EfAjH7dx6AqEW7PiSB2yQrXl1dbCbLYhQhEXIaJW+8epTCsm9o22+KZ5yIyyNc5SxPad7W4W/gvSzqMZC+NMBYia8hrq3BHk9OYxFUZfCx47adbs9sKFi1GhgCbXiDsrb8EvzMWMyyKyAl1AaCaMYWVXnDRBtJnASK2GyiR4GWHA+4z9syrEHknM0RonP2B24552cYq3gxCxRvZ6XJBRQaRnS9RjIaziDImXoLa5X/82ZT6cVzwdZa9S9LGJusfBLrE12O8nis1iI1CliukGtOmWvogFqYABtkzGizrnajCpU4doiuoUAAAEAchR3IrCbXnDekPQGt+uM4cSdZ9OIt2wROlxJODV/Z57OOGMm7RFJkuC4olN1h032dhHM6hsKzsMoRkin0zLuZgsInKs2LnmaL27wwB5wcriLnZm1HK9PUEH5A/wqR4Vc2SHUa1Zzvjddqe0fk2rRzYhaACctSii8CH68nmECZDgn0xePz8n4p4CoZalkQvcjEW/O6nzgrudW3VahxEl4YijuWocQ+y0vWIa6OByuYbDAxST971RF84Q0M8I1jnEwteGcYnLV8CVZQaZ5fmI3t4Z8CcRN339rKpvakVgNYYATJcTc402TIhyRHMLew63et11xkIVGocLaaYaI2gzavA==',
        TooLongKeyError, "dsa_2048", ["strict"]
    ],
    [
        'ssh-rsa AAAAB3NzaC1yc2EAAAADAQABAAAAYQbdtLTII+vP98NSDlK2LXxVARELRYO0NODFYQ0imYxsmBMB7BrfljFppLJyjU6cziOT6YFj6rVd8MmCogdCR32u63EV11uT6RCFfJMQJtIi+B1JJipTxLzURsiUOOgAHJc= ojarva@ojar-laptop.local',
        TooShortKeyError, "rsa_771", ["strict"]
    ],
    [
        'sk-ssh-ed25519@openssh.com AAAAGnNrLXNzaC1lZDI1NTE5QG9wZW5zc2guY29tAAAAIJ+CG2daFaeUC2GMvxGe5bTXRxforrL3MNOcsxbMbQeHAAAABGludjo=',
        InvalidKeyError, 'invalid_appid_sk-ssh-ed25519', ["loose", "strict"]
    ],
    [
        'sk-ssh-ed25519@openssh.com AAAAGnNrLXNzaC1lZDI1NTE5QG9wZW5zc2guY29tAAAAIHIoT5FA++PiJ4g4cORu3DBK4HWBO8mMjbp1Jtp4twl4AAAAB3NzaDovL1s=',
        InvalidKeyError, 'invalid_url_sk-ssh-ed25519', ["loose", "strict"]
    ],
]
