# 
# Siconos is a program dedicated to modeling, simulation and control
#  of non smooth dynamical systems.
#  Siconos is a free software; you can redistribute it and/or modify
#  it under the terms of the GNU Lesser General Public License as published by
#  the Free Software Foundation; either version 2 of the License, or
#  (at your option) any later version.
#  Siconos is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Lesser General Public License for more details.
# 
#  You should have received a copy of the GNU Lesser General Public License
#  along with Siconos; if not, write to the Free Software
#  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
# 
#  Contact: Vincent ACARY, siconos-team@lists.gforge.inria.fr

#[=======================================================================[.rst:
doxy2swig_docstrings

Build targets to generate python
docstrings from xml doxygen output.

1) Doxygen to create xml outputs.
2) Doxy2swig to
   create docstrings from xml outputs.

This macro must be called for each
component, in LibraryProjectSetup.

-- Targets :
1) headers --> xml using doxygen
2) xml files --> .i using doxy2swig (one .i for each xml file)
3) .i files --> ${COMP}-docstrings.i (one single file)

1 : target ${COMP}_xml4swig
2 and 3 : target ${COMP}_docstrings

Note FP : the whole process must be re-executed for any change in a header file of the component
(unless we set a doxygen conf for each header, don't think we need to bother with that.)

Doxygen steps are driven by cmake while doxy2swig and related are hidden in build_docstrings python
script.
#]=======================================================================]
macro(doxy2swig_docstrings COMP)
  if(WITH_${COMP}_DOXY2SWIG)
    
    # -- doxygen/xml config --

    # 1 - Create COMPdoxy.config.xml in binary dir,
    #     from doxy.config (source dir), taking
    #     into account sources/headers files
    #     of current component COMP.
    #
    # 2 - Run doxygen to build xml documentation
    #
    # Results in CMAKE_BINARY_DIR/docs/build/xml-docstrings
    #
    # Warning : output path must be different from
    # output for xml used by sphinx/breathe/exhale.
    # -----------------------------------------------------
    
    # Path where xml files will be generated by doxygen as inputs for swig.
    set(DOXY2SWIG_OUTPUT ${DOXYGEN_OUTPUT}/doxy2swig-xml/${COMP})
    file(MAKE_DIRECTORY ${DOXY2SWIG_OUTPUT})

    # Set doxygen config file name
    set(DOXY_CONFIG_XML "${CMAKE_BINARY_DIR}/docs/config/${COMP}doxy2swig-xml.config")
    # Create doxygen configuration file in binary dir
    set(DOXYGEN_INPUTS ${${COMP}_DOXYGEN_INPUTS}) #  list of inputs
    set(DOXY_QUIET "YES")
    set(DOXY_WARNINGS "NO")
    set(GENERATE_HTML NO)
    set(GENERATE_XML YES)
    set(EXTRACT_ALL NO)
    set(EXTRACT_PRIVATE NO)
    set(XML_OUTPUT doxy2swig-xml/${COMP})
    message(" -- Create doxygen conf (xml for docstrings) for component ${COMP}")
    configure_file(${CMAKE_SOURCE_DIR}/docs/config/doxy2swig.config.in ${DOXY_CONFIG_XML} @ONLY)

    # -- target to generate xml files (doxygen) from current component source files --
    #
    add_custom_target(${COMP}_xml4swig
      COMMAND ${DOXYGEN_EXECUTABLE} ${DOXY_CONFIG_XML}
      OUTPUT_FILE ${DOXYGEN_OUTPUT}/${COMP}doxy.log ERROR_FILE ${DOXYGEN_OUTPUT}/${COMP}doxy.log
      COMMENT " -- Build xml (for swig) doc for component ${COMP} ..."
      )
    
    # -- command to build .i files from xml (doxygen outputs) for current component  --
    # Call a python function from gendoctools (build_docstrings).
    add_custom_command(OUTPUT  ${SICONOS_SWIG_ROOT_DIR}/${COMP}-docstrings.i
      DEPENDS ${COMP}_xml4swig
      COMMAND ${CMAKE_COMMAND} -E env PYTHONPATH=${CMAKE_BINARY_DIR}/share ${PYTHON_EXECUTABLE} -c
      "from gendoctools.python2rst import build_docstrings as f; f('${${COMP}_HDRS}', '${COMP}', '${DOXY_CONFIG_XML}', '${SICONOS_SWIG_ROOT_DIR}')"
      VERBATIM
      )
    add_custom_target(${COMP}_docstrings DEPENDS ${SICONOS_SWIG_ROOT_DIR}/${COMP}-docstrings.i
      COMMENT "Create swig files from xml for component ${COMP}.")
  else()
    # No doxy2swig but 
    # generate empty ${COMP}-docstrings.i file (required because of %include in swig files)
    add_custom_command(OUTPUT ${SICONOS_SWIG_ROOT_DIR}/${COMP}-docstrings.i
      COMMAND ${CMAKE_COMMAND} -E touch
      ARGS ${SICONOS_SWIG_ROOT_DIR}/${COMP}-docstrings.i
      )
    add_custom_target(${COMP}_docstrings DEPENDS ${SICONOS_SWIG_ROOT_DIR}/${COMP}-docstrings.i
      COMMENT "This target has no real effect since swig/docstrings is not activated (PYTHON_WRAPPER and/or DOXY2SWIG is OFF).")

  endif()
  

endmacro()

