% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/iSpline.R
\name{iSpline}
\alias{iSpline}
\title{I-Spline Basis for Polynomial Splines}
\usage{
iSpline(
  x,
  df = NULL,
  knots = NULL,
  degree = 3L,
  intercept = TRUE,
  Boundary.knots = NULL,
  derivs = 0L,
  ...
)
}
\arguments{
\item{x}{The predictor variable.  Missing values are allowed and will be
returned as they are.}

\item{df}{Degree of freedom that equals to the column number of the returned
matrix.  One can specify \code{df} rather than \code{knots}, then the
function chooses \code{df - degree - as.integer(intercept)} internal
knots at suitable quantiles of \code{x} ignoring missing values and
those \code{x} outside of the boundary.  If internal knots are specified
via \code{knots}, the specified \code{df} will be ignored.}

\item{knots}{The internal breakpoints that define the splines.  The default
is \code{NULL}, which results in a basis for ordinary polynomial
regression.  Typical values are the mean or median for one knot,
quantiles for more knots.}

\item{degree}{The degree of I-spline defined to be the degree of the
associated M-spline instead of actual polynomial degree. For example,
I-spline basis of degree 2 is defined as the integral of associated
M-spline basis of degree 2.}

\item{intercept}{If \code{TRUE} by default, all spline bases are included.
Notice that when using I-Spline for monotonic regression,
\code{intercept = TRUE} should be set even when an intercept term is
considered additional to the spline bases in the model.}

\item{Boundary.knots}{Boundary points at which to anchor the splines.  By
default, they are the range of \code{x} excluding \code{NA}.  If both
\code{knots} and \code{Boundary.knots} are supplied, the basis
parameters do not depend on \code{x}. Data can extend beyond
\code{Boundary.knots}.}

\item{derivs}{A non-negative integer specifying the order of derivatives of
I-splines.}

\item{...}{Optional arguments that are not used.}
}
\value{
A numeric matrix of \code{length(x)} rows and \code{df} columns if
    \code{df} is specified or \code{length(knots) + degree +
    as.integer(intercept)} columns if \code{knots} are specified instead.
    Attributes that correspond to the arguments specified are returned
    mainly for other functions in this package.
}
\description{
Generates the I-spline (integral of M-spline) basis matrix for a polynomial
spline or the corresponding derivatives of given order.
}
\details{
It is an implementation of the close form I-spline basis based on the
recursion formula given by Ramsay (1988).
}
\examples{
library(splines2)

## Example given in the reference paper by Ramsay (1988)
x <- seq.int(0, 1, by = 0.01)
knots <- c(0.3, 0.5, 0.6)
isMat <- iSpline(x, knots = knots, degree = 2)

op <- par(mar = c(2.5, 2.5, 0.2, 0.1), mgp = c(1.5, 0.5, 0))
matplot(x, isMat, type = "l", ylab = "I-spline basis")
abline(v = knots, lty = 2, col = "gray")

## reset to previous plotting settings
par(op)

## the derivative of I-splines is M-spline
msMat1 <- iSpline(x, knots = knots, degree = 2, derivs = 1)
msMat2 <- mSpline(x, knots = knots, degree = 2, intercept = TRUE)
stopifnot(all.equal(msMat1, msMat2))
}
\references{
Ramsay, J. O. (1988). Monotone regression splines in action.
\emph{Statistical Science}, 3(4), 425--441.
}
\seealso{
\code{\link{mSpline}} for M-splines;
\code{\link{cSpline}} for C-splines;
}
