/*
 *
 * Copyright 2020 gRPC authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package weightedtarget

import (
	"testing"

	"github.com/google/go-cmp/cmp"
	"google.golang.org/grpc/balancer"
	internalserviceconfig "google.golang.org/grpc/internal/serviceconfig"
	_ "google.golang.org/grpc/xds/internal/balancer/cdsbalancer"
)

const (
	testJSONConfig = `{
  "targets": {
	"cluster_1" : {
	  "weight":75,
	  "childPolicy":[{"cds_experimental":{"cluster":"cluster_1"}}]
	},
	"cluster_2" : {
	  "weight":25,
	  "childPolicy":[{"cds_experimental":{"cluster":"cluster_2"}}]
	}
  }
}`

	cdsName = "cds_experimental"
)

var (
	cdsConfigParser = balancer.Get(cdsName).(balancer.ConfigParser)
	cdsConfigJSON1  = `{"cluster":"cluster_1"}`
	cdsConfig1, _   = cdsConfigParser.ParseConfig([]byte(cdsConfigJSON1))
	cdsConfigJSON2  = `{"cluster":"cluster_2"}`
	cdsConfig2, _   = cdsConfigParser.ParseConfig([]byte(cdsConfigJSON2))
)

func Test_parseConfig(t *testing.T) {
	tests := []struct {
		name    string
		js      string
		want    *lbConfig
		wantErr bool
	}{
		{
			name:    "empty json",
			js:      "",
			want:    nil,
			wantErr: true,
		},
		{
			name: "OK",
			js:   testJSONConfig,
			want: &lbConfig{
				Targets: map[string]target{
					"cluster_1": {
						Weight: 75,
						ChildPolicy: &internalserviceconfig.BalancerConfig{
							Name:   cdsName,
							Config: cdsConfig1,
						},
					},
					"cluster_2": {
						Weight: 25,
						ChildPolicy: &internalserviceconfig.BalancerConfig{
							Name:   cdsName,
							Config: cdsConfig2,
						},
					},
				},
			},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := parseConfig([]byte(tt.js))
			if (err != nil) != tt.wantErr {
				t.Errorf("parseConfig() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !cmp.Equal(got, tt.want) {
				t.Errorf("parseConfig() got unexpected result, diff: %v", cmp.Diff(got, tt.want))
			}
		})
	}
}
