﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/DateTime.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm/SSM_EXPORTS.h>
#include <aws/ssm/model/AlarmConfiguration.h>
#include <aws/ssm/model/AlarmStateInformation.h>
#include <aws/ssm/model/MaintenanceWindowExecutionStatus.h>
#include <aws/ssm/model/MaintenanceWindowTaskType.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace SSM {
namespace Model {

/**
 * <p>Information about a task execution performed as part of a maintenance window
 * execution.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/ssm-2014-11-06/MaintenanceWindowExecutionTaskIdentity">AWS
 * API Reference</a></p>
 */
class MaintenanceWindowExecutionTaskIdentity {
 public:
  AWS_SSM_API MaintenanceWindowExecutionTaskIdentity() = default;
  AWS_SSM_API MaintenanceWindowExecutionTaskIdentity(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API MaintenanceWindowExecutionTaskIdentity& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_SSM_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The ID of the maintenance window execution that ran the task.</p>
   */
  inline const Aws::String& GetWindowExecutionId() const { return m_windowExecutionId; }
  inline bool WindowExecutionIdHasBeenSet() const { return m_windowExecutionIdHasBeenSet; }
  template <typename WindowExecutionIdT = Aws::String>
  void SetWindowExecutionId(WindowExecutionIdT&& value) {
    m_windowExecutionIdHasBeenSet = true;
    m_windowExecutionId = std::forward<WindowExecutionIdT>(value);
  }
  template <typename WindowExecutionIdT = Aws::String>
  MaintenanceWindowExecutionTaskIdentity& WithWindowExecutionId(WindowExecutionIdT&& value) {
    SetWindowExecutionId(std::forward<WindowExecutionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ID of the specific task execution in the maintenance window
   * execution.</p>
   */
  inline const Aws::String& GetTaskExecutionId() const { return m_taskExecutionId; }
  inline bool TaskExecutionIdHasBeenSet() const { return m_taskExecutionIdHasBeenSet; }
  template <typename TaskExecutionIdT = Aws::String>
  void SetTaskExecutionId(TaskExecutionIdT&& value) {
    m_taskExecutionIdHasBeenSet = true;
    m_taskExecutionId = std::forward<TaskExecutionIdT>(value);
  }
  template <typename TaskExecutionIdT = Aws::String>
  MaintenanceWindowExecutionTaskIdentity& WithTaskExecutionId(TaskExecutionIdT&& value) {
    SetTaskExecutionId(std::forward<TaskExecutionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The status of the task execution.</p>
   */
  inline MaintenanceWindowExecutionStatus GetStatus() const { return m_status; }
  inline bool StatusHasBeenSet() const { return m_statusHasBeenSet; }
  inline void SetStatus(MaintenanceWindowExecutionStatus value) {
    m_statusHasBeenSet = true;
    m_status = value;
  }
  inline MaintenanceWindowExecutionTaskIdentity& WithStatus(MaintenanceWindowExecutionStatus value) {
    SetStatus(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The details explaining the status of the task execution. Not available for
   * all status values.</p>
   */
  inline const Aws::String& GetStatusDetails() const { return m_statusDetails; }
  inline bool StatusDetailsHasBeenSet() const { return m_statusDetailsHasBeenSet; }
  template <typename StatusDetailsT = Aws::String>
  void SetStatusDetails(StatusDetailsT&& value) {
    m_statusDetailsHasBeenSet = true;
    m_statusDetails = std::forward<StatusDetailsT>(value);
  }
  template <typename StatusDetailsT = Aws::String>
  MaintenanceWindowExecutionTaskIdentity& WithStatusDetails(StatusDetailsT&& value) {
    SetStatusDetails(std::forward<StatusDetailsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time the task execution started.</p>
   */
  inline const Aws::Utils::DateTime& GetStartTime() const { return m_startTime; }
  inline bool StartTimeHasBeenSet() const { return m_startTimeHasBeenSet; }
  template <typename StartTimeT = Aws::Utils::DateTime>
  void SetStartTime(StartTimeT&& value) {
    m_startTimeHasBeenSet = true;
    m_startTime = std::forward<StartTimeT>(value);
  }
  template <typename StartTimeT = Aws::Utils::DateTime>
  MaintenanceWindowExecutionTaskIdentity& WithStartTime(StartTimeT&& value) {
    SetStartTime(std::forward<StartTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The time the task execution finished.</p>
   */
  inline const Aws::Utils::DateTime& GetEndTime() const { return m_endTime; }
  inline bool EndTimeHasBeenSet() const { return m_endTimeHasBeenSet; }
  template <typename EndTimeT = Aws::Utils::DateTime>
  void SetEndTime(EndTimeT&& value) {
    m_endTimeHasBeenSet = true;
    m_endTime = std::forward<EndTimeT>(value);
  }
  template <typename EndTimeT = Aws::Utils::DateTime>
  MaintenanceWindowExecutionTaskIdentity& WithEndTime(EndTimeT&& value) {
    SetEndTime(std::forward<EndTimeT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the task that ran.</p>
   */
  inline const Aws::String& GetTaskArn() const { return m_taskArn; }
  inline bool TaskArnHasBeenSet() const { return m_taskArnHasBeenSet; }
  template <typename TaskArnT = Aws::String>
  void SetTaskArn(TaskArnT&& value) {
    m_taskArnHasBeenSet = true;
    m_taskArn = std::forward<TaskArnT>(value);
  }
  template <typename TaskArnT = Aws::String>
  MaintenanceWindowExecutionTaskIdentity& WithTaskArn(TaskArnT&& value) {
    SetTaskArn(std::forward<TaskArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The type of task that ran.</p>
   */
  inline MaintenanceWindowTaskType GetTaskType() const { return m_taskType; }
  inline bool TaskTypeHasBeenSet() const { return m_taskTypeHasBeenSet; }
  inline void SetTaskType(MaintenanceWindowTaskType value) {
    m_taskTypeHasBeenSet = true;
    m_taskType = value;
  }
  inline MaintenanceWindowExecutionTaskIdentity& WithTaskType(MaintenanceWindowTaskType value) {
    SetTaskType(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The details for the CloudWatch alarm applied to your maintenance window
   * task.</p>
   */
  inline const AlarmConfiguration& GetAlarmConfiguration() const { return m_alarmConfiguration; }
  inline bool AlarmConfigurationHasBeenSet() const { return m_alarmConfigurationHasBeenSet; }
  template <typename AlarmConfigurationT = AlarmConfiguration>
  void SetAlarmConfiguration(AlarmConfigurationT&& value) {
    m_alarmConfigurationHasBeenSet = true;
    m_alarmConfiguration = std::forward<AlarmConfigurationT>(value);
  }
  template <typename AlarmConfigurationT = AlarmConfiguration>
  MaintenanceWindowExecutionTaskIdentity& WithAlarmConfiguration(AlarmConfigurationT&& value) {
    SetAlarmConfiguration(std::forward<AlarmConfigurationT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The CloudWatch alarm that was invoked by the maintenance window task.</p>
   */
  inline const Aws::Vector<AlarmStateInformation>& GetTriggeredAlarms() const { return m_triggeredAlarms; }
  inline bool TriggeredAlarmsHasBeenSet() const { return m_triggeredAlarmsHasBeenSet; }
  template <typename TriggeredAlarmsT = Aws::Vector<AlarmStateInformation>>
  void SetTriggeredAlarms(TriggeredAlarmsT&& value) {
    m_triggeredAlarmsHasBeenSet = true;
    m_triggeredAlarms = std::forward<TriggeredAlarmsT>(value);
  }
  template <typename TriggeredAlarmsT = Aws::Vector<AlarmStateInformation>>
  MaintenanceWindowExecutionTaskIdentity& WithTriggeredAlarms(TriggeredAlarmsT&& value) {
    SetTriggeredAlarms(std::forward<TriggeredAlarmsT>(value));
    return *this;
  }
  template <typename TriggeredAlarmsT = AlarmStateInformation>
  MaintenanceWindowExecutionTaskIdentity& AddTriggeredAlarms(TriggeredAlarmsT&& value) {
    m_triggeredAlarmsHasBeenSet = true;
    m_triggeredAlarms.emplace_back(std::forward<TriggeredAlarmsT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_windowExecutionId;
  bool m_windowExecutionIdHasBeenSet = false;

  Aws::String m_taskExecutionId;
  bool m_taskExecutionIdHasBeenSet = false;

  MaintenanceWindowExecutionStatus m_status{MaintenanceWindowExecutionStatus::NOT_SET};
  bool m_statusHasBeenSet = false;

  Aws::String m_statusDetails;
  bool m_statusDetailsHasBeenSet = false;

  Aws::Utils::DateTime m_startTime{};
  bool m_startTimeHasBeenSet = false;

  Aws::Utils::DateTime m_endTime{};
  bool m_endTimeHasBeenSet = false;

  Aws::String m_taskArn;
  bool m_taskArnHasBeenSet = false;

  MaintenanceWindowTaskType m_taskType{MaintenanceWindowTaskType::NOT_SET};
  bool m_taskTypeHasBeenSet = false;

  AlarmConfiguration m_alarmConfiguration;
  bool m_alarmConfigurationHasBeenSet = false;

  Aws::Vector<AlarmStateInformation> m_triggeredAlarms;
  bool m_triggeredAlarmsHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSM
}  // namespace Aws
