﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ssm-incidents/SSMIncidentsRequest.h>
#include <aws/ssm-incidents/SSMIncidents_EXPORTS.h>

#include <utility>

namespace Aws {
namespace SSMIncidents {
namespace Model {

/**
 */
class BatchGetIncidentFindingsRequest : public SSMIncidentsRequest {
 public:
  AWS_SSMINCIDENTS_API BatchGetIncidentFindingsRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "BatchGetIncidentFindings"; }

  AWS_SSMINCIDENTS_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>A list of IDs of findings for which you want to view details.</p>
   */
  inline const Aws::Vector<Aws::String>& GetFindingIds() const { return m_findingIds; }
  inline bool FindingIdsHasBeenSet() const { return m_findingIdsHasBeenSet; }
  template <typename FindingIdsT = Aws::Vector<Aws::String>>
  void SetFindingIds(FindingIdsT&& value) {
    m_findingIdsHasBeenSet = true;
    m_findingIds = std::forward<FindingIdsT>(value);
  }
  template <typename FindingIdsT = Aws::Vector<Aws::String>>
  BatchGetIncidentFindingsRequest& WithFindingIds(FindingIdsT&& value) {
    SetFindingIds(std::forward<FindingIdsT>(value));
    return *this;
  }
  template <typename FindingIdsT = Aws::String>
  BatchGetIncidentFindingsRequest& AddFindingIds(FindingIdsT&& value) {
    m_findingIdsHasBeenSet = true;
    m_findingIds.emplace_back(std::forward<FindingIdsT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the incident for which you want to view
   * finding details.</p>
   */
  inline const Aws::String& GetIncidentRecordArn() const { return m_incidentRecordArn; }
  inline bool IncidentRecordArnHasBeenSet() const { return m_incidentRecordArnHasBeenSet; }
  template <typename IncidentRecordArnT = Aws::String>
  void SetIncidentRecordArn(IncidentRecordArnT&& value) {
    m_incidentRecordArnHasBeenSet = true;
    m_incidentRecordArn = std::forward<IncidentRecordArnT>(value);
  }
  template <typename IncidentRecordArnT = Aws::String>
  BatchGetIncidentFindingsRequest& WithIncidentRecordArn(IncidentRecordArnT&& value) {
    SetIncidentRecordArn(std::forward<IncidentRecordArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::Vector<Aws::String> m_findingIds;
  bool m_findingIdsHasBeenSet = false;

  Aws::String m_incidentRecordArn;
  bool m_incidentRecordArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace SSMIncidents
}  // namespace Aws
