﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/bedrock-agent/BedrockAgent_EXPORTS.h>
#include <aws/bedrock-agent/model/RdsFieldMapping.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace BedrockAgent {
namespace Model {

/**
 * <p>Contains details about the storage configuration of the knowledge base in
 * Amazon RDS. For more information, see <a
 * href="https://docs.aws.amazon.com/bedrock/latest/userguide/knowledge-base-setup-rds.html">Create
 * a vector index in Amazon RDS</a>.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/bedrock-agent-2023-06-05/RdsConfiguration">AWS
 * API Reference</a></p>
 */
class RdsConfiguration {
 public:
  AWS_BEDROCKAGENT_API RdsConfiguration() = default;
  AWS_BEDROCKAGENT_API RdsConfiguration(Aws::Utils::Json::JsonView jsonValue);
  AWS_BEDROCKAGENT_API RdsConfiguration& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_BEDROCKAGENT_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the vector store.</p>
   */
  inline const Aws::String& GetResourceArn() const { return m_resourceArn; }
  inline bool ResourceArnHasBeenSet() const { return m_resourceArnHasBeenSet; }
  template <typename ResourceArnT = Aws::String>
  void SetResourceArn(ResourceArnT&& value) {
    m_resourceArnHasBeenSet = true;
    m_resourceArn = std::forward<ResourceArnT>(value);
  }
  template <typename ResourceArnT = Aws::String>
  RdsConfiguration& WithResourceArn(ResourceArnT&& value) {
    SetResourceArn(std::forward<ResourceArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the secret that you created in Secrets
   * Manager that is linked to your Amazon RDS database.</p>
   */
  inline const Aws::String& GetCredentialsSecretArn() const { return m_credentialsSecretArn; }
  inline bool CredentialsSecretArnHasBeenSet() const { return m_credentialsSecretArnHasBeenSet; }
  template <typename CredentialsSecretArnT = Aws::String>
  void SetCredentialsSecretArn(CredentialsSecretArnT&& value) {
    m_credentialsSecretArnHasBeenSet = true;
    m_credentialsSecretArn = std::forward<CredentialsSecretArnT>(value);
  }
  template <typename CredentialsSecretArnT = Aws::String>
  RdsConfiguration& WithCredentialsSecretArn(CredentialsSecretArnT&& value) {
    SetCredentialsSecretArn(std::forward<CredentialsSecretArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of your Amazon RDS database.</p>
   */
  inline const Aws::String& GetDatabaseName() const { return m_databaseName; }
  inline bool DatabaseNameHasBeenSet() const { return m_databaseNameHasBeenSet; }
  template <typename DatabaseNameT = Aws::String>
  void SetDatabaseName(DatabaseNameT&& value) {
    m_databaseNameHasBeenSet = true;
    m_databaseName = std::forward<DatabaseNameT>(value);
  }
  template <typename DatabaseNameT = Aws::String>
  RdsConfiguration& WithDatabaseName(DatabaseNameT&& value) {
    SetDatabaseName(std::forward<DatabaseNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the table in the database.</p>
   */
  inline const Aws::String& GetTableName() const { return m_tableName; }
  inline bool TableNameHasBeenSet() const { return m_tableNameHasBeenSet; }
  template <typename TableNameT = Aws::String>
  void SetTableName(TableNameT&& value) {
    m_tableNameHasBeenSet = true;
    m_tableName = std::forward<TableNameT>(value);
  }
  template <typename TableNameT = Aws::String>
  RdsConfiguration& WithTableName(TableNameT&& value) {
    SetTableName(std::forward<TableNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Contains the names of the fields to which to map information about the vector
   * store.</p>
   */
  inline const RdsFieldMapping& GetFieldMapping() const { return m_fieldMapping; }
  inline bool FieldMappingHasBeenSet() const { return m_fieldMappingHasBeenSet; }
  template <typename FieldMappingT = RdsFieldMapping>
  void SetFieldMapping(FieldMappingT&& value) {
    m_fieldMappingHasBeenSet = true;
    m_fieldMapping = std::forward<FieldMappingT>(value);
  }
  template <typename FieldMappingT = RdsFieldMapping>
  RdsConfiguration& WithFieldMapping(FieldMappingT&& value) {
    SetFieldMapping(std::forward<FieldMappingT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_resourceArn;
  bool m_resourceArnHasBeenSet = false;

  Aws::String m_credentialsSecretArn;
  bool m_credentialsSecretArnHasBeenSet = false;

  Aws::String m_databaseName;
  bool m_databaseNameHasBeenSet = false;

  Aws::String m_tableName;
  bool m_tableNameHasBeenSet = false;

  RdsFieldMapping m_fieldMapping;
  bool m_fieldMappingHasBeenSet = false;
};

}  // namespace Model
}  // namespace BedrockAgent
}  // namespace Aws
