/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import com.google.common.util.concurrent.RateLimiter;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BooleanSupplier;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.apache.cassandra.concurrent.ExecutorFactory;
import org.apache.cassandra.concurrent.ExecutorPlus;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.hints.HintsCatalog;
import org.apache.cassandra.hints.HintsDescriptor;
import org.apache.cassandra.hints.HintsDispatcher;
import org.apache.cassandra.hints.HintsReader;
import org.apache.cassandra.hints.HintsService;
import org.apache.cassandra.hints.HintsStore;
import org.apache.cassandra.hints.InputPosition;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.concurrent.Future;
import org.apache.cassandra.utils.concurrent.UncheckedInterruptedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class HintsDispatchExecutor {
    private static final Logger logger = LoggerFactory.getLogger(HintsDispatchExecutor.class);
    private final File hintsDirectory;
    private final ExecutorPlus executor;
    private final AtomicBoolean isPaused;
    private final Predicate<InetAddressAndPort> isAlive;
    private final Map<UUID, Future> scheduledDispatches;

    HintsDispatchExecutor(File hintsDirectory, int maxThreads, AtomicBoolean isPaused, Predicate<InetAddressAndPort> isAlive) {
        this.hintsDirectory = hintsDirectory;
        this.isPaused = isPaused;
        this.isAlive = isAlive;
        this.scheduledDispatches = new ConcurrentHashMap<UUID, Future>();
        this.executor = (ExecutorPlus)ExecutorFactory.Global.executorFactory().withJmxInternal().configurePooled("HintsDispatcher", maxThreads).withThreadPriority(1).build();
    }

    void shutdownBlocking() {
        this.scheduledDispatches.clear();
        this.executor.shutdownNow();
        try {
            this.executor.awaitTermination(1L, TimeUnit.MINUTES);
        }
        catch (InterruptedException e) {
            throw new UncheckedInterruptedException(e);
        }
    }

    boolean isScheduled(HintsStore store) {
        return this.scheduledDispatches.containsKey(store.hostId);
    }

    Future dispatch(HintsStore store) {
        return this.dispatch(store, store.hostId);
    }

    Future dispatch(HintsStore store, UUID hostId) {
        return this.scheduledDispatches.computeIfAbsent(hostId, uuid -> this.executor.submit(new DispatchHintsTask(store, hostId)));
    }

    Future transfer(HintsCatalog catalog, Supplier<UUID> hostIdSupplier) {
        return this.executor.submit(new TransferHintsTask(catalog, hostIdSupplier));
    }

    void completeDispatchBlockingly(HintsStore store) {
        Future future = this.scheduledDispatches.get(store.hostId);
        try {
            if (future != null) {
                future.get();
            }
        }
        catch (InterruptedException e) {
            throw new UncheckedInterruptedException(e);
        }
        catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    void interruptDispatch(UUID hostId) {
        Future future = this.scheduledDispatches.remove(hostId);
        if (null != future) {
            future.cancel(true);
        }
    }

    public boolean isPaused() {
        return this.isPaused.get();
    }

    public boolean hasScheduledDispatches() {
        return !this.scheduledDispatches.isEmpty();
    }

    private final class DispatchHintsTask
    implements Runnable {
        private final HintsStore store;
        private final UUID hostId;
        private final RateLimiter rateLimiter;

        DispatchHintsTask(HintsStore store, UUID hostId, boolean isTransfer) {
            this.store = store;
            this.hostId = hostId;
            int nodesCount = isTransfer ? 1 : Math.max(1, StorageService.instance.getTokenMetadata().getAllEndpoints().size() - 1);
            double throttleInBytes = (double)DatabaseDescriptor.getHintedHandoffThrottleInKiB() * 1024.0 / (double)nodesCount;
            this.rateLimiter = RateLimiter.create((double)(throttleInBytes == 0.0 ? Double.MAX_VALUE : throttleInBytes));
        }

        DispatchHintsTask(HintsStore store, UUID hostId) {
            this(store, hostId, false);
        }

        @Override
        public void run() {
            try {
                this.dispatch();
            }
            finally {
                HintsDispatchExecutor.this.scheduledDispatches.remove(this.hostId);
            }
        }

        private void dispatch() {
            HintsDescriptor descriptor;
            while (!HintsDispatchExecutor.this.isPaused.get() && (descriptor = this.store.poll()) != null) {
                try {
                    if (this.dispatch(descriptor)) continue;
                    break;
                }
                catch (FSReadError e) {
                    logger.error(String.format("Failed to dispatch hints file %s: file is corrupted", descriptor.fileName()), (Throwable)e);
                    this.store.cleanUp(descriptor);
                    this.store.markCorrupted(descriptor);
                    throw e;
                }
            }
        }

        private boolean dispatch(HintsDescriptor descriptor) {
            logger.trace("Dispatching hints file {}", (Object)descriptor.fileName());
            InetAddressAndPort address = StorageService.instance.getEndpointForHostId(this.hostId);
            if (address != null) {
                return this.deliver(descriptor, address);
            }
            this.convert(descriptor);
            return true;
        }

        private boolean deliver(HintsDescriptor descriptor, InetAddressAndPort address) {
            File file = descriptor.file(HintsDispatchExecutor.this.hintsDirectory);
            InputPosition offset = this.store.getDispatchOffset(descriptor);
            BooleanSupplier shouldAbort = () -> !HintsDispatchExecutor.this.isAlive.test(address) || HintsDispatchExecutor.this.isPaused.get();
            Throwable throwable = null;
            try (HintsDispatcher dispatcher = HintsDispatcher.create(file, this.rateLimiter, address, descriptor.hostId, shouldAbort);){
                block19: {
                    if (offset != null) {
                        dispatcher.seek(offset);
                    }
                    try {
                        if (!dispatcher.dispatch()) break block19;
                        this.store.delete(descriptor);
                        this.store.cleanUp(descriptor);
                        logger.info("Finished hinted handoff of file {} to endpoint {}: {}", new Object[]{descriptor.fileName(), address, this.hostId});
                        boolean bl = true;
                        return bl;
                    }
                    catch (UncheckedInterruptedException e) {
                        try {
                            this.handleDispatchFailure(dispatcher, descriptor, address);
                            throw e;
                        }
                        catch (Throwable throwable2) {
                            throwable = throwable2;
                            throw throwable2;
                        }
                    }
                }
                this.handleDispatchFailure(dispatcher, descriptor, address);
                boolean bl = false;
                return bl;
            }
        }

        private void handleDispatchFailure(HintsDispatcher dispatcher, HintsDescriptor descriptor, InetAddressAndPort address) {
            this.store.markDispatchOffset(descriptor, dispatcher.dispatchPosition());
            this.store.offerFirst(descriptor);
            logger.info("Finished hinted handoff of file {} to endpoint {}: {}, partially", new Object[]{descriptor.fileName(), address, this.hostId});
        }

        private void convert(HintsDescriptor descriptor) {
            File file = descriptor.file(HintsDispatchExecutor.this.hintsDirectory);
            try (HintsReader reader = HintsReader.open(file, this.rateLimiter);){
                reader.forEach(page -> page.hintsIterator().forEachRemaining(HintsService.instance::writeForAllReplicas));
                this.store.delete(descriptor);
                this.store.cleanUp(descriptor);
                logger.info("Finished converting hints file {}", (Object)descriptor.fileName());
            }
        }
    }

    private final class TransferHintsTask
    implements Runnable {
        private final HintsCatalog catalog;
        private final Supplier<UUID> hostIdSupplier;

        private TransferHintsTask(HintsCatalog catalog, Supplier<UUID> hostIdSupplier) {
            this.catalog = catalog;
            this.hostIdSupplier = hostIdSupplier;
        }

        @Override
        public void run() {
            UUID hostId = this.hostIdSupplier.get();
            InetAddressAndPort address = StorageService.instance.getEndpointForHostId(hostId);
            logger.info("Transferring all hints to {}: {}", (Object)address, (Object)hostId);
            if (this.transfer(hostId)) {
                return;
            }
            logger.warn("Failed to transfer all hints to {}: {}; will retry in {} seconds", new Object[]{address, hostId, 10});
            try {
                TimeUnit.SECONDS.sleep(10L);
            }
            catch (InterruptedException e) {
                throw new UncheckedInterruptedException(e);
            }
            hostId = this.hostIdSupplier.get();
            logger.info("Transferring all hints to {}: {}", (Object)address, (Object)hostId);
            if (!this.transfer(hostId)) {
                logger.error("Failed to transfer all hints to {}: {}", (Object)address, (Object)hostId);
                throw new RuntimeException("Failed to transfer all hints to " + hostId);
            }
        }

        private boolean transfer(UUID hostId) {
            this.catalog.stores().map(store -> new DispatchHintsTask((HintsStore)store, hostId, true)).forEach(Runnable::run);
            return !this.catalog.hasFiles();
        }
    }
}

